/**
 * @file geis_xcb_backend_token.c
 * @brief GEIS filter token for the XCB-based grail back end
 *
 * Copyright 2011 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 3 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#include "geis_config.h"
#include "geis_xcb_backend_token.h"

#include "geis_logging.h"
#include "geis_private.h"
#include "geis_xcb_backend.h"
#include "grail_gestures.h"
#include <stdlib.h>
#include <string.h>
#include "xcb_gesture.h"


/* The number of uint32_t in the grail bitmask. */
#define GRAIL_XCB_BITMASK_LEN 2

/* The maximum nuimber of devices supported. */
#define MAX_NUM_DEVICES 10

/* The maximum nuimber of regions supported. */
#define MAX_NUM_WINDOWS 10


struct XcbBackendToken
{
  struct GeisBackendToken  base;
  GeisXcbBackend           be;
  xcb_connection_t        *xcb_connection;
  uint16_t                 device_count;
  uint16_t                 devices[MAX_NUM_DEVICES]; 
  uint16_t                 grail_mask_len;
  uint32_t                 grail_mask[GRAIL_XCB_BITMASK_LEN];
  int                      window_count;
  xcb_window_t             windows[MAX_NUM_WINDOWS];
};

static GeisBackendToken _token_clone(GeisBackendToken);
static void             _token_finalize(GeisBackendToken);
static void             _token_compose(GeisBackendToken, GeisBackendToken);
static GeisStatus       _token_activate(GeisBackendToken);
static GeisStatus       _token_deactivate(GeisBackendToken);

static struct GeisBackendTokenVtable _token_vtbl = {
  _token_clone,
  _token_finalize,
  _token_compose,
  _token_activate,
  _token_deactivate,
};


/* Converts from a GeisBackendToken to an XcbBackendToken */
static inline XcbBackendToken
_xcb_token_from_geis_token(GeisBackendToken gbt)
{
  return (XcbBackendToken)gbt;
}


GeisBackendToken
geis_xcb_token_new(GeisBackend be, GeisBackendTokenInitState init_state)
{
  XcbBackendToken token = NULL;
  token = calloc(1, sizeof(struct XcbBackendToken));
  token->base.vtbl = &_token_vtbl;
  token->be = (GeisXcbBackend)be;
  token->grail_mask_len = GRAIL_XCB_BITMASK_LEN;
  if (init_state == GEIS_BACKEND_TOKEN_INIT_NONE)
  {
    _grail_mask_clear(token->grail_mask_len, token->grail_mask);
  }
  else
  {
    geis_xcb_backend_gesture_bitmask_init(&token->grail_mask_len,
                                          token->grail_mask);
  }
  return (GeisBackendToken)token;
}


GeisBackendToken 
_token_clone(GeisBackendToken token)
{
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  XcbBackendToken n = calloc(1, sizeof(struct XcbBackendToken));
  memcpy(n, t, sizeof(struct XcbBackendToken));
  return (GeisBackendToken)n;
}


void             
_token_finalize(GeisBackendToken token)
{
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  free(t);
}


uint16_t
geis_xcb_token_device_count(XcbBackendToken token)
{
  return token->device_count;
}


uint16_t
geis_xcb_token_device(XcbBackendToken token, uint16_t index)
{
  return token->devices[index];
}


int
geis_xcb_token_window_count(XcbBackendToken token)
{
  return token->window_count;
}


xcb_window_t
geis_xcb_token_window(XcbBackendToken token, int index)
{
  return token->windows[index];
}


uint16_t
geis_xcb_token_grail_mask_len(XcbBackendToken token)
{
  return token->grail_mask_len;
}


uint32_t*
geis_xcb_token_grail_mask(XcbBackendToken token)
{
  return token->grail_mask;
}


void
geis_xcb_token_set_xcb(GeisBackendToken token, xcb_connection_t *xcb_connection)
{
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  t->xcb_connection = xcb_connection;
}


void
_token_compose(GeisBackendToken lhs, GeisBackendToken rhs)
{
  XcbBackendToken token1 = _xcb_token_from_geis_token(lhs);
  XcbBackendToken token2 = _xcb_token_from_geis_token(rhs);

  /* Merge gesture masks. */
  _grail_mask_or(token1->grail_mask_len, token1->grail_mask, token2->grail_mask);

  /* Merge device lists */
  {
    int i;
    for (i = 0; i < token2->device_count; ++i)
    {
      int j;
      for (j = 0; j < token1->device_count; ++j)
      {
	if (token1->devices[j] == token2->devices[i])
	  break;
      }
      if (j == MAX_NUM_DEVICES)
      {
	geis_error("maximum devices reached.");
	break;
      }
      if (j == token1->device_count)
      {
	token1->devices[token1->device_count++] = token2->devices[i];
      }
    }
  }

  /* Merge window lists */
  {
    int i;
    for (i = 0; i < token2->window_count; ++i)
    {
      int j;
      for (j = 0; j < token1->window_count; ++j)
      {
	if (token1->windows[j] == token2->windows[i])
	  break;
      }
      if (j == MAX_NUM_WINDOWS)
      {
	geis_error("maximum windows reached.");
	break;
      }
      if (j == token1->window_count)
      {
	token1->windows[token1->window_count++] = token2->windows[i];
      }
    }
  }
  geis_debug("bitmap is now 0x%08x %08x",
             token1->grail_mask[1], token1->grail_mask[0]);
}


/**
 * Normalizes the devices for a token.
 *
 * Normalization in this case means that if there are no specific devices
 * associated with the token, then a special indicator of 'all devices' will
 * be associated with the token.
 */
static void
_normalize_token_devices(XcbBackendToken token)
{
  if (token->device_count == 0)
  {
    token->devices[token->device_count++] = 0;
  }
}


/**
 * Normalizes the devices for a token.
 *
 * Normalization in this case means that if there are no specific regions
 * associated with the token, then the 'root window' regions will be associated
 * with the token.
 */
static void
_normalize_token_regions(XcbBackendToken token)
{
  if (token->window_count == 0)
  {
    const xcb_setup_t *setup = xcb_get_setup(token->xcb_connection);
    geis_debug("defaulting to ROOT window");
    if (!setup)
    {
      geis_error("error getting xcb setup");
      return;
    }

    xcb_screen_iterator_t screen = xcb_setup_roots_iterator(setup);
    while (screen.rem)
    {
      token->windows[token->window_count++] = screen.data->root;
      xcb_screen_next(&screen);
    }
  }
}


/**
 * Selects XCB events for a token.
 */
static GeisStatus
_token_select_xcb_events(XcbBackendToken token)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  int dev;
  int win;

  for (dev = 0; dev < token->device_count; ++dev)
  {
    for (win = 0; win < token->window_count; ++win)
    {
      geis_debug("window_id=0x%08x device_id=%d",
                 token->windows[win], token->devices[dev]);
      geis_xcb_backend_select_events(token->be,
                                     token->devices[dev],
                                     token->windows[win]);
      status = GEIS_STATUS_SUCCESS;
    }
  }
  return status;
}


/**
 * Activates a token.
 *
 * Normalizes the token, registers it with the back end, then adjusts the
 * various back end subscriptions appropriately.
 */
static GeisStatus             
_token_activate(GeisBackendToken token)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  XcbBackendToken t = _xcb_token_from_geis_token(token);

  _normalize_token_devices(t);
  _normalize_token_regions(t);
  geis_xcb_backend_add_token(t->be, t);
  status = _token_select_xcb_events(t);

  return status;
}


/**
 * Deactivates a token.
 *
 * Unregisters the token with the back end, then adjusts the
 * various back end subscriptions appropriately.
 */
static GeisStatus             
_token_deactivate(GeisBackendToken token)
{
  GeisStatus status = GEIS_STATUS_SUCCESS;
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  geis_xcb_backend_remove_token(t->be, t);
  status = _token_select_xcb_events(t);

  return status;
}


GeisStatus
geis_xcb_token_add_class_term(GeisBackendToken     token,
                              void                *context,
                              GeisString           name,
                              GeisFilterOperation  op,
                              void                *value)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  GeisGestureClass gesture_class = (GeisGestureClass)context;

  if (0 == strcmp(name, GEIS_CLASS_ATTRIBUTE_NAME)
      && op == GEIS_FILTER_OP_EQ)
  {
    GeisString class_name = (GeisString)value;
    if (0 == strcmp(class_name, geis_gesture_class_name(gesture_class)))
    {
      geis_xcb_backend_gesture_bitmask_filter_class(class_name,
                                                    &t->grail_mask_len,
                                                    t->grail_mask);
    }
    status = GEIS_STATUS_SUCCESS;
  }
  else if (0 == strcmp(name, GEIS_CLASS_ATTRIBUTE_ID)
      && op == GEIS_FILTER_OP_EQ)
  {
    GeisInteger id = *(GeisInteger*)value;
    if (id == geis_gesture_class_id(gesture_class))
    {
      GeisString class_name = geis_gesture_class_name(gesture_class);
      geis_xcb_backend_gesture_bitmask_filter_class(class_name,
                                                    &t->grail_mask_len,
                                                    t->grail_mask);
    }
  }
  else if (0 == strcmp(name, GEIS_GESTURE_ATTRIBUTE_TOUCHES))
  {
    GeisInteger touches = *(GeisInteger*)value;
    geis_xcb_backend_gesture_bitmask_filter_touches(touches,
                                                    op,
                                                    &t->grail_mask_len,
                                                    t->grail_mask);
    status = GEIS_STATUS_SUCCESS;
  }
  return status;
}


static void
_add_device_to_list(GeisDevice device, XcbBackendToken token)
{
  uint16_t i;
  for (i = 0; i < token->device_count; ++i)
  {
    if (token->devices[i] == geis_device_id(device))
    {
      return;
    }
  }
  token->devices[token->device_count++] = geis_device_id(device);
}


/*
 * Modifies the token to incliude a specific device term.
 */
GeisStatus
geis_xcb_token_add_device_term(GeisBackendToken     token,
                               void                *context GEIS_UNUSED,
                               GeisString           name,
                               GeisFilterOperation  op,
                               void                *value)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  XcbBackendToken t = _xcb_token_from_geis_token(token);
  Geis geis = geis_xcb_backend_geis(t->be);

  if (0 == strcmp(name, GEIS_DEVICE_ATTRIBUTE_NAME)
     && op == GEIS_FILTER_OP_EQ)
  {
    GeisString device_name = (GeisString)value;
    geis_debug("attr name=\"%s\" value=\"%s\"", name, device_name);
    GeisDeviceBag device_bag = geis_devices(geis);
    GeisSize device_count = geis_device_bag_count(device_bag);
    GeisSize i;
    for (i = 0; i < device_count; ++i)
    {
      GeisDevice device = geis_device_bag_device(device_bag, i);
      if (0 == strcmp(geis_device_name(device), device_name))
      {
	_add_device_to_list(device, t);
      }
    }
  }
  else if (0 == strcmp(name, GEIS_DEVICE_ATTRIBUTE_ID)
           && op == GEIS_FILTER_OP_EQ)
  {
    GeisInteger device_id = *(GeisInteger*)value;
    geis_debug("attr name=\"%s\" value=%d", name, device_id);
    t->devices[t->device_count++] = device_id;
    status = GEIS_STATUS_SUCCESS;
  }
  else if (0 == strcmp(name, GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH)
           && op == GEIS_FILTER_OP_EQ)
  {
    GeisBoolean is_direct = *(GeisBoolean*)value;
    geis_debug("attr name=\"%s\" value=%d", name, is_direct);
    GeisDeviceBag device_bag = geis_devices(geis);
    GeisSize device_count = geis_device_bag_count(device_bag);
    GeisSize i;
    for (i = 0; i < device_count; ++i)
    {
      GeisDevice device = geis_device_bag_device(device_bag, i);
      GeisSize attr_count = geis_device_attr_count(device);
      GeisSize j;
      for (j = 0; j < attr_count; ++j)
      {
	GeisAttr attr = geis_device_attr(device, j);
	if (0 == strcmp(geis_attr_name(attr), GEIS_DEVICE_ATTRIBUTE_DIRECT_TOUCH))
	{
	  if (geis_attr_value_to_boolean(attr) == is_direct)
	  {
	    _add_device_to_list(device, t);
	  }
	  break;
	}
      }
    }
  }
  else if (0 == strcmp(name, GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH)
           && op == GEIS_FILTER_OP_EQ)
  {
    GeisBoolean is_independent = *(GeisBoolean*)value;
    geis_debug("attr name=\"%s\" value=%d", name, is_independent);
    GeisDeviceBag device_bag = geis_devices(geis);
    GeisSize device_count = geis_device_bag_count(device_bag);
    GeisSize i;
    for (i = 0; i < device_count; ++i)
    {
      GeisDevice device = geis_device_bag_device(device_bag, i);
      GeisSize attr_count = geis_device_attr_count(device);
      GeisSize j;
      for (j = 0; j < attr_count; ++j)
      {
	GeisAttr attr = geis_device_attr(device, j);
	if (0 == strcmp(geis_attr_name(attr), GEIS_DEVICE_ATTRIBUTE_INDEPENDENT_TOUCH))
	{
	  if (geis_attr_value_to_boolean(attr) == is_independent)
	  {
	    _add_device_to_list(device, t);
	  }
	  break;
	}
      }
    }
  }
  return status;
}


GeisStatus
geis_xcb_token_add_region_term(GeisBackendToken     token,
                               void                *context GEIS_UNUSED,
                               GeisString           name,
                               GeisFilterOperation  op,
                               void                *value)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  XcbBackendToken t = _xcb_token_from_geis_token(token);

  if (0 == strcmp(name, GEIS_REGION_ATTRIBUTE_WINDOWID)
     && op == GEIS_FILTER_OP_EQ)
  {
    GeisInteger window_id = *(GeisInteger*)value;
    geis_debug("attr name=\"%s\" windowid=0x%x", name, window_id);
    t->windows[t->window_count++] = window_id;
    status = GEIS_STATUS_SUCCESS;
  }
  return status;
}


GeisStatus
geis_xcb_token_add_feature_term(GeisBackendToken     token,
                                void                *context GEIS_UNUSED,
                                GeisString           name,
                                GeisFilterOperation  op,
                                void                *value)
{
  GeisStatus status = GEIS_STATUS_UNKNOWN_ERROR;
  XcbBackendToken t = _xcb_token_from_geis_token(token);

  if (0 == strcmp(name, GEIS_GESTURE_TYPE_SYSTEM)
     && op == GEIS_FILTER_OP_EQ)
  {
    GeisBoolean is_system = *(GeisBoolean*)value;
    geis_debug("attr name=\"%s\" is_system=%d", name, is_system);
    geis_xcb_backend_gesture_bitmask_filter_system(is_system, 
                                                   &t->grail_mask_len,
                                                   t->grail_mask);
    status = GEIS_STATUS_SUCCESS;
  }
  else if (0 == strcmp(name, "GRAB")
          && op == GEIS_FILTER_OP_EQ)
  {
    GeisBoolean is_grab = *(GeisBoolean*)value;
    geis_debug("attr name=\"%s\" is_grab=%d", name, is_grab);
    geis_xcb_backend_gesture_bitmask_filter_grabs(is_grab, 
                                                  &t->grail_mask_len,
                                                  t->grail_mask);
    status = GEIS_STATUS_SUCCESS;
  }
  return status;
}

