#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Creates a reference file of the current or given display configuration 
abstraction that can be used in an unit test.
"""

__author__ = "Sebastian Heinlein <devel@glatzor.de>"
__copyright__ = "Copyright © 2007 Sebastian Heinlein"
__licence__ = "GPL"

import os
import sys
from displayconfigabstraction import *

from ConfigParser import ConfigParser
from optparse import OptionParser

class CreateReference:
    def __init__(self, options):
        self.options = options
        SetDataFileDir(options.datadir)
        self.xsetup = XSetup(options.xconfigpath,
                             debug_scan_pci_filename=self.options.pcitable)
        self.xsetup._syncXorgConfig()
        self.reference = ConfigParser()
        primary = self.xsetup.getPrimaryScreen()

        self.reference.add_section("xsetup")
        self.reference.set("xsetup",
                           "layout",
                           self.xsetup.getLayout())
        self.reference.set("xsetup",
                           "orientation",
                           self.xsetup.getDualheadOrientation())
        primary = self.xsetup.getPrimaryScreen()
        self.reference.set("xsetup",
                           "primary",
                           primary._getXorgScreenSection().identifier)
        secondary = self.xsetup.getSecondaryScreen()
        self.reference.set("xsetup",
                           "secondary",
                           secondary._getXorgScreenSection().identifier)

        for gfxcard in self.xsetup.getGfxCards():
            # Create a section for each graphics card
            gfx_device = gfxcard.getXorgDeviceSection(0)
            section = gfx_device.identifier
            self.reference.add_section(section)
            self.reference.set(section,
                               "driver",
                               gfxcard.getGfxCardModel().getDriver())
            self.reference.set(section,
                               "videoram",
                               gfxcard.getVideoRam())
            self.reference.set(section,
                               "pcibusid",
                               gfxcard.getPCIBusID())
            self.reference.set(section,
                               "devices",
                               map(lambda g: g.identifier, gfxcard.x_device))
            for screen in gfxcard.getScreens():
                # Create a section for each screen
                screen_device = screen._getXorgScreenSection()
                section = screen_device.identifier
                self.reference.add_section(section)
                self.reference.set(section,
                                   "resolutions",
                                   screen.getAvailableResolutions())
                self.reference.set(section,
                                   "resolution",
                                   screen.getResolutionIndex())
                self.reference.set(section,
                                   "rate",
                                   screen.getRefreshRateIndex())
                self.reference.set(section,
                                   "rates",
                                   screen.getAvailableRefreshRates())
                self.reference.set(section,
                                   "aspect",
                                   screen.getMonitorAspect())
                self.reference.set(section,
                                   "gamma",
                                   screen.getAllGamma())
                self.reference.set(section,
                                   "model",
                                   screen.getMonitorModel())
                self.reference.set(section,
                                   "gfxcard",
                                   gfx_device.identifier)
    def write_reference(self, path):
        self.reference.write(open(path, "w"))

if __name__ == "__main__":
    parser = OptionParser(usage="%prog REFERENCEFILE")
    parser.add_option("-c", "--xconfig",
                      default="/etc/X11/xorg.conf",
                      action="store", type="string", dest="xconfigpath",
                      help="Use the given X.org configuration file")
    parser.add_option("-p", "--pcitable",
                      default=None,
                      action="store", type="string", dest="pcitable",
                      help="Use the given pcitable")
    parser.add_option("-d", "--data-dir",
                      default="/usr/share/apps/guidance",
                      action="store", type="string", dest="datadir",
                      help="Load data from the given path")
    (options, args) = parser.parse_args()

    if len(args) != 1:
        parser.print_usage()
        sys.exit(1)
    app = CreateReference(options)
    app.write_reference(args[0])

    sys.exit(0)

# vim:ts=4:sw=4:et
