# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Vincenzo Di Somma <vincenzo.di.somma@canonical.com>

"""This modules olds some code that should back ported to python-couchdb"""

import cgi
import errno
import re
import socket
import sys
import time
import urlparse

from httplib import BadStatusLine
from urlparse import urlsplit, urlunsplit

from oauth import oauth

# pylint can't deal with failing imports even when they're handled
# pylint: disable=F0401
try:
    from cStringIO import StringIO
except ImportError:
    from StringIO import StringIO

# pylint: enable=F0401

from couchdb.http import (
    Session, CHUNK_SIZE, CACHE_SIZE, RedirectLimit, ResponseBody, Unauthorized,
    PreconditionFailed, ServerError, ResourceNotFound, ResourceConflict)

from couchdb import json as couchdbjson

NORMALIZE_SPACE = re.compile(r'(?:\r\n)?[ \t]+')


class OAuthSession(Session):
    """Session that can handle OAuth"""

    def __init__(self, cache=None, timeout=None, max_redirects=5,
                 credentials=None):
        """Initialize an HTTP client session with oauth credential. """
        super(OAuthSession, self).__init__(cache=cache,
                                           timeout=timeout,
                                           max_redirects=max_redirects)
        self.credentials = credentials

    def request(self, method, url, body=None, headers=None, credentials=None,
                num_redirects=0):

        def normalize_headers(headers):
            """normalize the headers so oauth likes them"""
            return dict(
                [(key.lower(),
                  NORMALIZE_SPACE.sub(
                      value,
                      ' ').strip())  for (key, value) in headers.iteritems()])

        def oauth_sign(creds, url, method):
            """Sign the url with the tokens and return an header"""
            consumer = oauth.OAuthConsumer(creds['consumer_key'],
                                           creds['consumer_secret'])
            access_token = oauth.OAuthToken(creds['token'],
                                            creds['token_secret'])
            sig_method = oauth.OAuthSignatureMethod_HMAC_SHA1
            query = urlparse.urlparse(url)[4]
            querystr_as_dict = dict(cgi.parse_qsl(query))
            req = oauth.OAuthRequest.from_consumer_and_token(
                consumer,
                access_token,
                http_method=method,
                http_url=url,
                parameters=querystr_as_dict)
            req.sign_request(sig_method(), consumer, access_token)
            return req.to_header()

        if url in self.perm_redirects:
            url = self.perm_redirects[url]
        method = method.upper()

        if headers is None:
            headers = {}
        headers.setdefault('Accept', 'application/json')
        headers['User-Agent'] = self.user_agent

        cached_resp = None
        if method in ('GET', 'HEAD'):
            cached_resp = self.cache.get(url)
            if cached_resp is not None:
                etag = cached_resp[1].get('etag')
                if etag:
                    headers['If-None-Match'] = etag

        if body is not None:
            if not isinstance(body, basestring):
                try:
                    body = couchdbjson.encode(body).encode('utf-8')
                except TypeError:
                    pass
                else:
                    headers.setdefault('Content-Type', 'application/json')
            if isinstance(body, basestring):
                headers.setdefault('Content-Length', str(len(body)))
            else:
                headers['Transfer-Encoding'] = 'chunked'

        if credentials:
            creds = credentials
        elif self.credentials:
            creds = self.credentials
        else:
            creds = None
        if creds:
            headers.update(normalize_headers(
                    oauth_sign(creds, url, method)))

        path_query = urlunsplit(('', '') + urlsplit(url)[2:4] + ('',))
        conn = self._get_connection(url)

        def _try_request_with_retries(retries):
            """Retries the request if it fails for a socket problem"""
            while True:
                try:
                    return _try_request()
                except socket.error, e:
                    ecode = e.args[0]
                    if ecode not in self.retryable_errors:
                        raise
                    try:
                        delay = retries.next()
                    except StopIteration:
                        # No more retries, raise last socket error.
                        raise e
                    time.sleep(delay)
                    conn.close()

        def _try_request():
            """Tries the request and handle socket problems"""
            try:
                if conn.sock is None:
                    conn.connect()
                conn.putrequest(method, path_query, skip_accept_encoding=True)
                for header in headers:
                    conn.putheader(header, headers[header])
                conn.endheaders()
                if body is not None:
                    if isinstance(body, str):
                        conn.sock.sendall(body)
                    else:  # assume a file-like object and send in chunks
                        while 1:
                            chunk = body.read(CHUNK_SIZE)
                            if not chunk:
                                break
                            conn.sock.sendall(('%x\r\n' % len(chunk)) +
                                              chunk + '\r\n')
                        conn.sock.sendall('0\r\n\r\n')
                return conn.getresponse()
            except BadStatusLine, e:
                # httplib raises a BadStatusLine when it cannot read the status
                # line saying, "Presumably, the server closed the connection
                # before sending a valid response."
                # Raise as ECONNRESET to simplify retry logic.
                if e.line == '' or e.line == "''":
                    raise socket.error(errno.ECONNRESET)
                else:
                    raise

        resp = _try_request_with_retries(iter(self.retry_delays))
        status = resp.status

        # Handle conditional response
        if status == 304 and method in ('GET', 'HEAD'):
            resp.read()
            self._return_connection(url, conn)
            status, msg, data = cached_resp
            if data is not None:
                data = StringIO(data)
            return status, msg, data
        elif cached_resp:
            del self.cache[url]

        # Handle redirects
        if status == 303 or \
                method in ('GET', 'HEAD') and status in (301, 302, 307):
            resp.read()
            self._return_connection(url, conn)
            if num_redirects > self.max_redirects:
                raise RedirectLimit('Redirection limit exceeded')
            location = resp.getheader('location')
            if status == 301:
                self.perm_redirects[url] = location
            elif status == 303:
                method = 'GET'
            return self.request(method, location, body, headers,
                                num_redirects=num_redirects + 1)

        data = None
        streamed = False

        # Read the full response for empty responses so that the connection is
        # in good state for the next request
        if method == 'HEAD' or resp.getheader('content-length') == '0' or \
                status < 200 or status in (204, 304):
            resp.read()
            self._return_connection(url, conn)

        # Buffer small non-JSON response bodies
        elif int(resp.getheader('content-length', sys.maxint)) < CHUNK_SIZE:
            data = resp.read()
            self._return_connection(url, conn)

        # For large or chunked response bodies, do not buffer the full body,
        # and instead return a minimal file-like object
        else:
            data = ResponseBody(resp,
                                lambda: self._return_connection(url, conn))
            streamed = True

        # Handle errors
        if status >= 400:
            if data is not None:
                try:
                    data = couchdbjson.decode(data)
                    # pylint: disable=E1103
                    error = data.get('error'), data.get('reason')
                    # pylint: enable=E1103
                except ValueError:
                    error = data
            elif method != 'HEAD':
                error = resp.read()
                self._return_connection(url, conn)
            else:
                error = ''

            if status == 401:
                raise Unauthorized(error)
            elif status == 404:
                raise ResourceNotFound(error)
            elif status == 409:
                raise ResourceConflict(error)
            elif status == 412:
                raise PreconditionFailed(error)
            else:
                raise ServerError((status, error))

        # Store cachable responses
        if not streamed and method == 'GET' and 'etag' in resp.msg:
            self.cache[url] = (status, resp.msg, data)
            if len(self.cache) > CACHE_SIZE[1]:
                self._clean_cache()

        if not streamed and data is not None:
            data = StringIO(data)

        return status, resp.msg, data
