/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by:
 *               Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 */

/**
 * SECTION:dee-filters
 * @title: Filters
 * @short_description: A suite of simple #DeeFilter<!-- -->s for use with #DeeFilterModel<!-- -->s
 * @include: dee.h
 *
 * #DeeFilter<!-- -->s are used together with #DeeFilterModel<!-- -->s to build
 * "views" of some original #DeeModel. An example could be to build a view
 * of a model that exposes the rows of the original model sorted by a given
 * column (leaving the original model unaltered):
 * |[
 *   DeeModel  *model, *view;
 *   DeeFilter *collator;
 *
 *   // Create and populate a model with some unsorted rows
 *   model = dee_sequence_model_new ();
 *   dee_model_set_schema (model, "i", "s", NULL);
 *   dee_model_append (model, 27, "Foo");
 *   dee_model_append (model, 68, "Bar");
 *
 *   // Create a collator for column 1
 *   collator = dee_filter_new_collator (1);
 *
 *   // Create the sorted view
 *   view = dee_filter_model_new (collator, model);
 *   g_free (collator);
 *
 *   // When accessing the view the row with 'Bar' will be first
 * ]|
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "dee-filter-model.h"
#include "trace-log.h"

/* The CollatorFilter stores collation keys for the columns in a DeeModelTag */
typedef struct {
  guint        column;
  DeeModelTag *collation_key_tag;
} CollatorFilter;

typedef struct {
  guint        column;
  gchar       *key;
} KeyFilter;

typedef struct {
  guint        column;
  GRegex      *regex;
} RegexFilter;

typedef struct {
  guint        column;
  GVariant    *value;
} ValueFilter;

/*
 * Private impl
 */
static void _dee_filter_collator_map_notify (DeeModel *orig_model,
                                             DeeModelIter *orig_iter,
                                             DeeFilterModel *filter_model,
                                             gpointer user_data);

static void
_dee_filter_collator_map_func (DeeModel *orig_model,
                               DeeFilterModel *filter_model,
                               gpointer user_data)
{
  DeeModelIter   *iter, *end;
  CollatorFilter *filter;

  g_return_if_fail (user_data != NULL);

  filter = (CollatorFilter *) user_data;
  filter->collation_key_tag =
                 dee_model_register_tag (orig_model, (GDestroyNotify) g_free);

  iter = dee_model_get_first_iter (orig_model);
  end = dee_model_get_last_iter (orig_model);
  while (iter != end)
    {
      _dee_filter_collator_map_notify (orig_model, iter, filter_model, filter);
      iter = dee_model_next (orig_model, iter);
    }

}

static void
_dee_filter_collator_map_notify (DeeModel *orig_model,
                                 DeeModelIter *orig_iter,
                                 DeeFilterModel *filter_model,
                                 gpointer user_data)
{
  DeeModelIter   *iter, *end;
  CollatorFilter *filter;
  const gchar    *column_value, *test_value;
  gchar          *collation_key;

  g_return_if_fail (user_data != NULL);
  g_return_if_fail (orig_iter != NULL);

  filter = (CollatorFilter *) user_data;

  /* Build and set collation ley for the current row */
  column_value = dee_model_get_string (orig_model, orig_iter, filter->column);
  collation_key = g_utf8_collate_key (column_value, -1);
  dee_model_set_tag (orig_model, orig_iter,
                     filter->collation_key_tag, collation_key); // steal collation_key

  iter = dee_model_get_first_iter (DEE_MODEL (filter_model));
  end = dee_model_get_last_iter (DEE_MODEL (filter_model));
  while (iter != end)
    {
      test_value = dee_model_get_tag (orig_model, iter, filter->collation_key_tag);
      if (g_strcmp0 (test_value, column_value) < 0)
        {
          iter = dee_model_next (DEE_MODEL (filter_model), iter);
        }
      else
        {
          dee_filter_model_insert_iter_before (filter_model, orig_iter, iter);
          iter = NULL;
          break;
        }
    }

  if (iter == end)
    {
      dee_filter_model_append_iter(filter_model, orig_iter);
    }
}

static void
_dee_filter_key_map_func (DeeModel *orig_model,
                          DeeFilterModel *filter_model,
                          gpointer user_data)
{
  DeeModelIter   *iter, *end;
  KeyFilter      *filter;
  guint           column;
  const gchar    *key, *val;

  g_return_if_fail (user_data != NULL);

  filter = (KeyFilter *) user_data;
  key = filter->key;
  column = filter->column;

  iter = dee_model_get_first_iter (orig_model);
  end = dee_model_get_last_iter (orig_model);
  while (iter != end)
    {
      val = dee_model_get_string (orig_model, iter, column);
      if (g_strcmp0 (key, val) == 0)
        {
          dee_filter_model_append_iter (filter_model, iter);
        }
      iter = dee_model_next (orig_model, iter);
    }
}

static void
_dee_filter_key_map_notify (DeeModel *orig_model,
                            DeeModelIter *orig_iter,
                            DeeFilterModel *filter_model,
                            gpointer user_data)
{
  KeyFilter      *filter;
  const gchar    *val;

  g_return_if_fail (user_data != NULL);

  filter = (KeyFilter *) user_data;
  val = dee_model_get_string (orig_model, orig_iter, filter->column);

  /* Ignore rows that don't match the key */
  if (g_strcmp0 (filter->key, val) != 0)
    return;

  dee_filter_model_insert_iter_with_original_order (filter_model, orig_iter);
}

static void
_dee_filter_value_map_func (DeeModel *orig_model,
                            DeeFilterModel *filter_model,
                            gpointer user_data)
{
  DeeModelIter   *iter, *end;
  ValueFilter    *filter;
  GVariant       *val;

  g_return_if_fail (user_data != NULL);

  filter = (ValueFilter *) user_data;

  iter = dee_model_get_first_iter (orig_model);
  end = dee_model_get_last_iter (orig_model);
  while (iter != end)
    {
      val = dee_model_get_value (orig_model, iter, filter->column);
      if (g_variant_equal (filter->value, val))
        {
          dee_filter_model_append_iter (filter_model, iter);
        }
      iter = dee_model_next (orig_model, iter);
    }
}

static void
_dee_filter_value_map_notify (DeeModel *orig_model,
                              DeeModelIter *orig_iter,
                              DeeFilterModel *filter_model,
                              gpointer user_data)
{
  ValueFilter    *filter;
  GVariant       *val;

  g_return_if_fail (user_data != NULL);

  filter = (ValueFilter *) user_data;
  val = dee_model_get_value (orig_model, orig_iter, filter->column);

  /* Ignore rows that don't match the value */
  if (!g_variant_equal (filter->value, val))
    return;

  dee_filter_model_insert_iter_with_original_order (filter_model, orig_iter);
}

static void
_dee_filter_regex_map_func (DeeModel *orig_model,
                            DeeFilterModel *filter_model,
                            gpointer user_data)
{
  DeeModelIter   *iter, *end;
  RegexFilter    *filter;
  guint           column;
  GRegex         *regex;
  const gchar    *val;

  g_return_if_fail (user_data != NULL);

  filter = (RegexFilter *) user_data;
  regex = filter->regex;
  column = filter->column;

  iter = dee_model_get_first_iter (orig_model);
  end = dee_model_get_last_iter (orig_model);
  while (iter != end)
    {
      val = dee_model_get_string (orig_model, iter, column);
      if (g_regex_match (regex, val, 0, NULL))
        {
          dee_filter_model_append_iter (filter_model, iter);
        }
      iter = dee_model_next (orig_model, iter);
    }
}

static void
_dee_filter_regex_map_notify (DeeModel *orig_model,
                              DeeModelIter *orig_iter,
                              DeeFilterModel *filter_model,
                              gpointer user_data)
{
  RegexFilter    *filter;
  const gchar    *val;

  g_return_if_fail (user_data != NULL);

  filter = (RegexFilter *) user_data;
  val = dee_model_get_string (orig_model, orig_iter, filter->column);

  /* Ignore rows that don't match the key */
  if (!g_regex_match (filter->regex, val, 0, NULL))
    return;

  dee_filter_model_insert_iter_with_original_order (filter_model, orig_iter);
}

static void
key_filter_free (KeyFilter *filter)
{
  g_free (filter->key);
  g_free (filter);
}

static void
value_filter_free (ValueFilter *filter)
{
  g_variant_unref (filter->value);
  g_free (filter);
}

static void
regex_filter_free (RegexFilter *filter)
{
  g_regex_unref (filter->regex);
  g_free (filter);
}

/*
 * API
 */

/**
 * dee_filter_new_collator:
 * @column: The index of a column containing the strings to sort after
 *
 * Create a #DeeFilter that takes string values from a column in the model
 * and builds a #DeeFilterModel with the rows sorted according to the
 * collation rules of the current locale.
 *
 * Returns: (transfer full): A newly allocated #DeeFilter. Do not modify it.
 *          Free with g_free().
 */
DeeFilter*
dee_filter_new_collator    (guint column)
{
  DeeFilter      *filter;
  CollatorFilter *collator;

  filter = g_new0 (DeeFilter, 1);
  filter->map_func = _dee_filter_collator_map_func;
  filter->map_notify = _dee_filter_collator_map_notify;

  collator = g_new0 (CollatorFilter, 1);
  collator->column = column;

  filter->destroy = (GDestroyNotify) g_free;
  filter->user_data =collator;

  return filter;
}

/**
 * dee_filter_new_for_key_column:
 * @column: The index of a column containing the string key to match
 *
 * Create a #DeeFilter that only includes rows from the original model
 * which has an exact match on some string column. A #DeeFilterModel created
 * with this filter will be ordered in accordance with its parent model.
 *
 * Returns: (transfer full): A newly allocated #DeeFilter. Do not modify it.
 *          Free with g_free().
 */
DeeFilter*
dee_filter_new_for_key_column    (guint column, const gchar *key)
{
  DeeFilter      *filter;
  KeyFilter      *key_filter;

  filter = g_new0 (DeeFilter, 1);
  filter->map_func = _dee_filter_key_map_func;
  filter->map_notify = _dee_filter_key_map_notify;

  key_filter = g_new0 (KeyFilter, 1);
  key_filter->column = column;
  key_filter->key = g_strdup (key);

  filter->destroy = (GDestroyNotify) key_filter_free;
  filter->user_data = key_filter;

  return filter;
}

/**
 * dee_filter_new_for_any_column:
 * @column: The index of a column containing the string to match
 * @value: (transfer none): A #GVariant value columns must match exactly.
 *         The matching semantics are those of g_variant_equal(). If @value
 *         is floating the ownership will be transfered to the filter
 *
 * Create a #DeeFilter that only includes rows from the original model
 * which match a variant value in a given column. A #DeeFilterModel
 * created with this filter will be ordered in accordance with its parent model.
 *
 * This method will work on any column, disregarding its schema, since the
 * value comparison is done using g_variant_equal(). This means you can use
 * this filter as a convenient fallback when there is no predefined filter
 * for your column type if raw performance is not paramount.
 *
 * Returns: (transfer full): A newly allocated #DeeFilter. Do not modify it.
 *          Free with g_free().
 */
DeeFilter*
dee_filter_new_for_any_column (guint column, GVariant *value)
{
  DeeFilter      *filter;
  ValueFilter    *v_filter;

  g_return_val_if_fail (value != NULL, NULL);

  filter = g_new0 (DeeFilter, 1);
  filter->map_func = _dee_filter_value_map_func;
  filter->map_notify = _dee_filter_value_map_notify;

  v_filter = g_new0 (ValueFilter, 1);
  v_filter->column = column;
  v_filter->value = g_variant_ref_sink (value);

  filter->destroy = (GDestroyNotify) value_filter_free;
  filter->user_data = v_filter;

  return filter;
}

/**
 * dee_filter_new_regex:
 * @column: The index of a column containing the string to match
 * @regex: (transfer none):The regular expression @column must match
 *
 * Create a #DeeFilter that only includes rows from the original model
 * which match a regular expression on some string column. A #DeeFilterModel
 * created with this filter will be ordered in accordance with its parent model.
 *
 * Returns: (transfer full): A newly allocated #DeeFilter. Do not modify it.
 *          Free with g_free().
 */
DeeFilter*
dee_filter_new_regex (guint column, GRegex *regex)
{
  DeeFilter      *filter;
  RegexFilter    *r_filter;

  g_return_val_if_fail (regex != NULL, NULL);

  filter = g_new0 (DeeFilter, 1);
  filter->map_func = _dee_filter_regex_map_func;
  filter->map_notify = _dee_filter_regex_map_notify;

  r_filter = g_new0 (RegexFilter, 1);
  r_filter->column = column;
  r_filter->regex = g_regex_ref (regex);

  filter->destroy = (GDestroyNotify) regex_filter_free;
  filter->user_data = r_filter;

  return filter;
}
