/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Gordon Allott <gord.allott@canonical.com>
 *
 */
#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "ctk-menu-item.h"
#include "ctk-check-menu-item.h"
#include <glib.h>

#include "ctk-radio-menu-item.h"

#define CTK_RADIO_MENU_ITEM_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  CTK_TYPE_RADIO_MENU_ITEM, \
  CtkRadioMenuItemPrivate))

enum
{
  PROP_0,

  PROP_GROUP
};

enum
{
  GROUP_CHANGED,

  LAST_SIGNAL
};

struct _CtkRadioMenuItemPrivate
{
  int null;
};

void on_radio_menu_item_toggled (CtkRadioMenuItem *menuitem);

static guint radio_menu_item_signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (CtkRadioMenuItem, ctk_radio_menu_item, CTK_TYPE_CHECK_MENU_ITEM);

static void
ctk_radio_menu_item_init (CtkRadioMenuItem *menuitem)
{
  g_signal_connect (menuitem, "toggled", G_CALLBACK (on_radio_menu_item_toggled), NULL);
}

static void
ctk_radio_menu_item_finalize (GObject *object)
{
  /* TODO: Add deinitalization code here */

  G_OBJECT_CLASS (ctk_radio_menu_item_parent_class)->finalize (object);
}

static void
ctk_radio_menu_item_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (CTK_IS_RADIO_MENU_ITEM (object));
  CtkRadioMenuItem *menuitem = CTK_RADIO_MENU_ITEM (object);

  switch (prop_id)
    {
      case PROP_GROUP:
        ctk_radio_menu_item_set_group (menuitem, g_value_get_pointer (value));
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
ctk_radio_menu_item_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  g_return_if_fail (CTK_IS_RADIO_MENU_ITEM (object));
  CtkRadioMenuItem *menuitem = CTK_RADIO_MENU_ITEM (object);

  switch (prop_id)
    {
      case PROP_GROUP:
         g_value_set_pointer (value, ctk_radio_menu_item_get_group (menuitem));
        break;
      default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
        break;
    }
}

static void
ctk_radio_menu_item_group_changed (CtkRadioMenuItem *self)
{
  /* TODO: Add default signal handler implementation here */
}

static void
ctk_radio_menu_item_class_init (CtkRadioMenuItemClass *klass)
{
  GObjectClass* object_class = G_OBJECT_CLASS (klass);
  //CtkMenuItemClass* parent_class = CTK_MENU_ITEM_CLASS (klass);

  object_class->finalize = ctk_radio_menu_item_finalize;
  object_class->set_property = ctk_radio_menu_item_set_property;
  object_class->get_property = ctk_radio_menu_item_get_property;
  g_type_class_add_private (klass, sizeof (CtkRadioMenuItemPrivate));

  klass->group_changed = ctk_radio_menu_item_group_changed;

  g_object_class_install_property (object_class,
                                   PROP_GROUP,
                                   g_param_spec_pointer("group",
                                                        "group",
                                                        "The radio menu item whose group this widget belongs to",
                                                        G_PARAM_READABLE | G_PARAM_WRITABLE));

  radio_menu_item_signals[GROUP_CHANGED] =
    g_signal_new ("group-changed",
                  G_OBJECT_CLASS_TYPE (klass),
                  0,
                  G_STRUCT_OFFSET (CtkRadioMenuItemClass, group_changed),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

CtkRadioMenuItem *
ctk_radio_menu_item_new (GSList *group)
{
  CtkRadioMenuItem *item;
  item = g_object_new (CTK_TYPE_RADIO_MENU_ITEM,
                       "label", "",
                       "group", group,
                       NULL);
  return item;
}

CtkRadioMenuItem *
ctk_radio_menu_item_new_with_label (GSList *group, const gchar *label)
{
  CtkRadioMenuItem *item;
  item = g_object_new (CTK_TYPE_RADIO_MENU_ITEM,
                       "label", label,
                       "group", group,
                       NULL);
  return item;
}

void
ctk_radio_menu_item_set_group (CtkRadioMenuItem *menuitem, GSList *group)
{
  g_return_if_fail (CTK_IS_RADIO_MENU_ITEM (menuitem));
  CtkActor *old_group_singleton = NULL;
  CtkActor *new_group_singleton = NULL;

  // if we are already set we have to gracefully leave the previous group
  if (menuitem->group != NULL)
    {
      menuitem->group = g_slist_remove (menuitem->group,  menuitem);

      if (menuitem->group != NULL && !menuitem->group->next)
        old_group_singleton = g_object_ref (menuitem->group->data);

      // have to go through the group and set all their items to the new group location
      GSList *slist;
      for (slist = menuitem->group; slist; slist = slist->next)
        {
          CtkRadioMenuItem *item;
          item = slist->data;
          item->group = menuitem->group;
        }
    }

  if (group != NULL && group->next)
    new_group_singleton = g_object_ref (group->data);

  menuitem->group = g_slist_prepend (group, menuitem);

  if (group)
    {
      // have to go through the group and set all their items to the new group location
      GSList *slist;
      for (slist = group; slist; slist = slist->next)
        {
          CtkRadioMenuItem *item;
          item = slist->data;
          item->group = menuitem->group;
        }
    }
  else
    {
      ctk_check_menu_item_set_active (CTK_CHECK_MENU_ITEM (menuitem), TRUE);
    }

  g_object_ref (menuitem);

  // notify and emit signals on ourselfs and all the other actors involved in the groups
  g_object_notify (G_OBJECT (menuitem), "group");
  g_signal_emit (menuitem, radio_menu_item_signals[GROUP_CHANGED], 0);
  if (old_group_singleton)
    {
      g_signal_emit (old_group_singleton,
                     radio_menu_item_signals[GROUP_CHANGED],
                     0);
      g_object_unref (old_group_singleton);
    }
  if (new_group_singleton)
    {
      g_signal_emit (new_group_singleton,
                     radio_menu_item_signals[GROUP_CHANGED],
                     0);
      g_object_unref (new_group_singleton);
    }

  g_object_unref (menuitem);
}

GSList*
ctk_radio_menu_item_get_group (CtkRadioMenuItem *menuitem)
{
  g_return_val_if_fail (CTK_IS_RADIO_MENU_ITEM (menuitem), NULL);

  // if a group doesn't exist we make a new one, else return the one we have
  if (menuitem->group == NULL)
    {
      menuitem->group = g_slist_append (menuitem->group, menuitem);
    }

  return menuitem->group;

}

void
on_radio_menu_item_toggled (CtkRadioMenuItem *menuitem)
{
  g_return_if_fail (CTK_IS_RADIO_MENU_ITEM (menuitem));

  if (ctk_check_menu_item_get_active (CTK_CHECK_MENU_ITEM (menuitem)))
    {
      // we have become active so set all the other items in the group to not active
      GSList *slist;
      for (slist = menuitem->group; slist; slist = slist->next)
        {
          CtkRadioMenuItem *item;
          item = slist->data;
          if (item == menuitem)
              continue;

          ctk_check_menu_item_set_active (CTK_CHECK_MENU_ITEM (item), FALSE);
        }
    }


}
