# Copyright (C) 2005 Aaron Bentley, 2006 Michael Ellerman
# <aaron@aaronbentley.com>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import os

try:
    from bzrlib.errors import BzrCommandError as CommandError
    from bzrlib.errors import BzrError
except ImportError:
    class CommandError(Exception):
        pass

class PatchFailed(BzrError):

    _fmt = """Patch application failed"""


class PatchInvokeError(BzrError):

    _fmt = """Error invoking patch: %(errstr)s%(stderr)s"""
    internal_error = False

    def __init__(self, e, stderr=''):
        self.exception = e
        self.errstr = os.strerror(e.errno)
        self.stderr = '\n' + stderr


class NoColor(Exception):
    """Color not available on this terminal."""


class NoBzrtoolsColor(Exception):
    """Bzrtools is required for color display"""


class NotCheckout(CommandError):
    """Specified path is not a checkout."""
    def __init__(self, path):
        CommandError.__init__(self, "%s is not a checkout" % path)

class UncommittedCheckout(CommandError):
    """This checkout contains uncommitted changes"""
    def __init__(self):
        CommandError.__init__(self, "This checkout has uncommitted changes")

class ParentMissingRevisions(CommandError):
    """The parent branch is missing revisions."""
    def __init__(self, parent):
        CommandError.__init__(self,
            "The parent branch %s is missing revisions from this branch." %
            parent)
        self.parent = parent

class NoParent(CommandError):
    def __init__(self):
        CommandError.__init__(self, "There is no parent, so deleting the"
                                    " branch could destroy data.")

class ChangedBinaryFiles(BzrError):

    _fmt = 'Changes involve binary files.'


class NoConflictFiles(CommandError):

    _fmt = '%(base_name)s does not exist and there are no pending merges.'

    def __init__(self, base_name):
        CommandError.__init__(self, base_name=base_name)
