# Copyright (C) 2008, 2009, 2010 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA

"""Test directory service implementation"""

from bzrlib import (
    errors,
    transport,
    urlutils,
    )
from bzrlib.directory_service import (
    AliasDirectory,
    DirectoryServiceRegistry,
    directories,
    )
from bzrlib.tests import TestCase, TestCaseWithTransport


class FooService(object):
    """A directory service that maps the name to a FILE url"""

    # eg 'file:///foo' on Unix, or 'file:///C:/foo' on Windows
    base = urlutils.local_path_to_url('/foo')

    def look_up(self, name, url):
        return self.base + name


class TestDirectoryLookup(TestCase):

    def setUp(self):
        TestCase.setUp(self)
        self.registry = DirectoryServiceRegistry()
        self.registry.register('foo:', FooService, 'Map foo URLs to http urls')

    def test_get_directory_service(self):
        directory, suffix = self.registry.get_prefix('foo:bar')
        self.assertIs(FooService, directory)
        self.assertEqual('bar', suffix)

    def test_dereference(self):
        self.assertEqual(FooService.base + 'bar',
                         self.registry.dereference('foo:bar'))
        self.assertEqual('baz:qux', self.registry.dereference('baz:qux'))

    def test_get_transport(self):
        directories.register('foo:', FooService, 'Map foo URLs to http urls')
        self.addCleanup(directories.remove, 'foo:')
        self.assertEqual(FooService.base + 'bar/',
                         transport.get_transport('foo:bar').base)


class TestAliasDirectory(TestCaseWithTransport):

    def test_lookup_parent(self):
        branch = self.make_branch('.')
        branch.set_parent('http://a')
        self.assertEqual('http://a', directories.dereference(':parent'))

    def test_lookup_submit(self):
        branch = self.make_branch('.')
        branch.set_submit_branch('http://b')
        self.assertEqual('http://b', directories.dereference(':submit'))

    def test_lookup_public(self):
        branch = self.make_branch('.')
        branch.set_public_branch('http://c')
        self.assertEqual('http://c', directories.dereference(':public'))

    def test_lookup_bound(self):
        branch = self.make_branch('.')
        branch.set_bound_location('http://d')
        self.assertEqual('http://d', directories.dereference(':bound'))

    def test_lookup_push(self):
        branch = self.make_branch('.')
        branch.set_push_location('http://e')
        self.assertEqual('http://e', directories.dereference(':push'))

    def test_lookup_this(self):
        branch = self.make_branch('.')
        self.assertEqual(branch.base, directories.dereference(':this'))

    def test_extra_path(self):
        branch = self.make_branch('.')
        self.assertEqual(urlutils.join(branch.base, 'arg'),
                         directories.dereference(':this/arg'))

    def test_lookup_badname(self):
        branch = self.make_branch('.')
        e = self.assertRaises(errors.InvalidLocationAlias,
                              directories.dereference, ':booga')
        self.assertEqual('":booga" is not a valid location alias.',
                         str(e))

    def test_lookup_badvalue(self):
        branch = self.make_branch('.')
        e = self.assertRaises(errors.UnsetLocationAlias,
                              directories.dereference, ':parent')
        self.assertEqual('No parent location assigned.', str(e))

    def test_register_location_alias(self):
        branch = self.make_branch('.')
        self.addCleanup(AliasDirectory.branch_aliases.remove, "booga")
        AliasDirectory.branch_aliases.register("booga",
            lambda b: "UHH?", help="Nobody knows")
        self.assertEquals("UHH?", directories.dereference(":booga"))
