/*------------------------------------------------------------*
 | io.c                                                       |
 | copyright 1999,  Andrew Sumner (andrew_sumner@bigfoot.com) |
 |                                                            |
 | This is a source file for the awka package, a translator   |
 | of the AWK programming language to ANSI C.                 |
 |                                                            |
 | This library is free software; you can redistribute it     |
 | and/or modify it under the terms of the Awka Library       |
 | License, which may be found in the file LIBLICENSE.txt.    |
 |                                                            |
 | This library is distributed in the hope that it will be    |
 | useful, but WITHOUT ANY WARRANTY; without even the implied |
 | warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR    |
 | PURPOSE.                                                   |
 *------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define _IO_C
#define _IN_LIBRARY
#include "libawka.h"

_a_IOSTREAM *_a_iostream = NULL;
int _a_ioallc = 0, _a_ioused = 0;
char _a_char[256], _interactive = FALSE;

/*
 * _awka_sopen
 * opens a file or pipe for input/output
 */
void
_awka_sopen(_a_IOSTREAM *s, char flag)
{
  if (s->io != (char) _a_IO_CLOSED) return;
  s->interactive = FALSE;

  if ((s->pipe == 1))
  {
    switch (flag)
    {
      case _a_IO_READ:
        s->fp = popen(s->name, "r");
        if (s->fp) fflush(s->fp);
        if (_interactive) s->interactive = TRUE;
        break;
      case _a_IO_WRITE:
        if (!(s->fp = popen(s->name, "w")))
          awka_error("sopen: unable to open piped process '%s' for write access.\n",s->name);
        fflush(s->fp);
        break;
      case _a_IO_APPEND:
        if (!(s->fp = popen(s->name, "a")))
          awka_error("sopen: unable to open piped process '%s' for append access.\n",s->name);
        fflush(s->fp);
        break;
    }
  }
  else
  {
    switch (flag)
    {
      case _a_IO_READ:
        if (!strcmp(s->name, "-") || !strcmp(s->name, "/dev/stdin"))
          s->fp = stdin;
        else 
          s->fp = fopen(s->name, "r");
        if (_interactive || !strncmp(s->name, "/dev/", 5))
          s->interactive = TRUE;
        if (s->fp) fflush(s->fp);
        break;
      case _a_IO_WRITE:
        if (!(s->fp = fopen(s->name, "w")))
          awka_error("sopen: unable to open file '%s' for write access.\n",s->name);
        fflush(s->fp);
        break;
      case _a_IO_APPEND:
        if (!(s->fp = fopen(s->name, "a")))
          awka_error("sopen: unable to open file '%s' for append access.\n",s->name);
        fflush(s->fp);
        break;
    }
  }
  if (!s->fp)
    s->io = _a_IO_CLOSED;
  else
  {
    s->io = flag;
    if (flag == _a_IO_READ && !s->alloc)
    {
      s->alloc = A_BUFSIZ;
      malloc( &s->buf, A_BUFSIZ + 1 );
      s->buf[A_BUFSIZ] = '\0';
      s->current = s->end = s->buf;
    }
  }
}

/*
 * _awka_io_addstream
 * creates a new input or output stream
 */
int
_awka_io_addstream( char *name, char flag, int pipe )
{
  int i, j, k;

  if (!*name) 
    awka_error("io_addstream: empty filename, flag = %d.\n",flag);

  if (pipe != 0 && pipe != 1)
    awka_error("io_addstream: pipe argument must be 0 or 1, got %d.\n",pipe);

  for (i=0; i<_a_ioused; i++)
    if (_a_iostream[i].pipe == pipe &&
        !strcmp(name, _a_iostream[i].name) &&
        (_a_iostream[i].io == flag || 
        _a_iostream[i].io == _a_IO_CLOSED))
      break;

  if (i < _a_ioused)
  {
    if (_a_iostream[i].io == flag) 
      return i;
    _a_iostream[i].pipe = pipe;
    _awka_sopen(&_a_iostream[i], flag);
    return i;
  }

  j = _a_ioused++;
  if (_a_ioused >= _a_ioallc)
  {
    if (!_a_ioallc)
    {
      /* awka_init has not been called */
      awka_error("io_addstream: awka_init() not called!\n");
    }
    else
    {
      k = _a_ioallc;
      _a_ioallc *= 2;
      realloc( &_a_iostream, _a_ioallc * sizeof(_a_IOSTREAM) );
      for (i=k; i<_a_ioallc; i++) 
      {
        _a_iostream[i].name = _a_iostream[i].buf = _a_iostream[i].end = _a_iostream[i].current = NULL;
        _a_iostream[i].io = _a_IO_CLOSED;
        _a_iostream[i].fp = NULL;
        _a_iostream[i].alloc = _a_iostream[i].interactive = 0;
      }
    }
  }

  malloc( &_a_iostream[j].name, strlen(name)+1);
  strcpy(_a_iostream[j].name, name);
  _a_iostream[j].pipe = (char) pipe;
  _awka_sopen(&_a_iostream[j], flag);
  return j;
}
      
void
_awka_io_cleanbinchars(a_VAR *var)
{
  register char *r, *q;

  r = var->ptr + var->slen;
  q = var->ptr;
  if (var->slen >= 8)
  while (q<=(r-8)) 
  {
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
    *q = _a_char[*q++];
  }
  while (q<r)
    *q = _a_char[*q++];
}

int
_awka_io_fillbuff(_a_IOSTREAM *s)
{
  if (!fread(s->current, 1, s->alloc - (s->current - s->buf), s->fp))
    return 0;
  return 1;
}

#define _RS_REGEXP 1
#define _RS_NL     2
#define _RS_CHAR   3

/*
 * awka_io_readline
 * reads a new line from the designated strm and
 * inserts it into var.
 */
int
awka_io_readline( a_VAR *var, int strm, int fill_target)
{
  char *p = NULL, *q;
  register char recsep, rs_type;
  char *end = NULL, eof = FALSE;
  _a_IOSTREAM *s = &_a_iostream[strm];
  int j = 0, i = 0;

  if (strm >= _a_ioused)
    awka_error("io_readline: stream %d passed to io_readline, but highest available is %d.\n",strm,_a_ioused-1);

  if (s->io == _a_IO_WRITE || s->io == _a_IO_APPEND)
    awka_error("io_readline: output stream %d (%s) passed to io_readline!\n",strm,s->name);
  else if (s->io == _a_IO_CLOSED)
  {
    _awka_sopen(s, _a_IO_READ);
    if (s->io == _a_IO_CLOSED)
      return 0;
  }
  else if (s->io == _a_IO_EOF)
    return 0;

  switch (a_bivar[a_RS]->type)
  {
    case a_VARDBL:
    case a_VARNUL:
      awka_gets(a_bivar[a_RS]);
    case a_VARSTR:
    case a_VARUNK:
      if (a_bivar[a_RS]->slen <= 1)
      {
        recsep = a_bivar[a_RS]->ptr[0];
        rs_type = _RS_CHAR;
        if (!recsep) 
          rs_type = _RS_NL;
        break;
      }
      
      a_bivar[a_RS]->ptr = (char *) _awka_compile_regexp_SPLIT(
                              a_bivar[a_RS]->ptr, a_bivar[a_RS]->slen );
      a_bivar[a_RS]->type = a_VARREG;
    case a_VARREG:
      rs_type = _RS_REGEXP;
  }

  while (1)
  {
    p = NULL;

    if (s->end > s->buf)
    {
      /* identify RS in data already read */
      switch (rs_type)
      {
        case _RS_CHAR:
            p = memchr(s->current, recsep, s->end - s->current);
            break;
        case _RS_NL:
            q = s->current;
            while (*q == '\n' && q < s->end) q++;
            if (q == s->end) break;
            p = strstr(q, "\n\n");
            break;
        case _RS_REGEXP:
          i = awka_regexec((regexp *) a_bivar[a_RS]->ptr, 
                     s->current, &p, &end, (int) TRUE, FALSE);
      }

      if (p)
      {
        /* RS found */
        if (fill_target)
        {
          if (rs_type == _RS_NL)
            awka_strncpy(var, q, p - q);
          else
            awka_strncpy(var, s->current, p - s->current);
#ifdef NO_BIN_CHARS
          _awka_io_cleanbinchars(var);
#endif
        }

        switch (rs_type)
        {
          case _RS_REGEXP:
            awka_strncpy(a_bivar[a_RT], p, end - p);
            s->current = end;
            break;
          case _RS_NL:
            s->current = (p+2 > s->end ? s->end : p+2);
            break;
          case _RS_CHAR:
            s->current = p+1;
        }
  
        return 1;
      }
    }

    /* here because there's no data in buffer, or RS is not 
     * in the buffer's data, hence try to read in more data */

    if (eof == TRUE)
    {
      char ret = 1;
      /* end of file already reached */
      if (fill_target)
      {
        if (rs_type == _RS_NL && s->end > s->current)
        {
          /* scrub trailing newline characters */
          p = s->end - 1;
          while (*p == '\n') *(p--) = '\0';
          s->end = p+1;
        }
        else if (rs_type == _RS_REGEXP)
          awka_strcpy(a_bivar[a_RT], "");
           
        if (s->end > s->current)
        {
          awka_strncpy(var, s->current, (s->end - s->current));
#ifdef NO_BIN_CHARS
          _awka_io_cleanbinchars(var);
#endif
        }
      }
      if (s->end <= s->current) ret = 0;
      if (s->buf) free(s->buf);
      s->buf = s->current = s->end = NULL;
      s->io = _a_IO_EOF;
      return ret;
    }

    if (s->current - s->buf > s->alloc - 256)
    {
      /* about to hit end of buffer - move remainder to front */
      if (s->current < s->end)
      {
        /* there's a remainder to preserve */
        memmove(s->buf, s->current, s->end - s->current);
        s->end -= (s->current - s->buf);
        s->current = s->buf;
      }
      else
      {
        /* there isn't */
        *s->buf = '\0';
        s->end = s->current = s->buf;
      }
    }

    if (s->end - s->buf > s->alloc - 256)
    {
      /* increase buffer size */
      i = s->current - s->buf;
      j = s->end - s->buf;
      s->alloc = realloc( &s->buf, s->alloc * 2 );
      s->current = s->buf + i;
      s->end = s->buf + j;
    }

    if (s->interactive)
    {
      /* line buffered */
      if (!fgets(s->end, s->alloc - (s->end - s->buf) - 1, s->fp))
        eof = TRUE;
      else
        s->end = s->end + strlen(s->end);
    }
    else
    {
      /* block buffered */
      if (!(i = fread(s->end, 1, s->alloc - (s->end - s->buf) - 1, s->fp)))
        eof = TRUE;
      else
        s->end += i;
    }

  }
}


void
awka_exit( double ret )
{
  register int i;

  for (i=0; i<_a_ioused; i++)
  {
    if (_a_iostream[i].fp && _a_iostream[i].io != _a_IO_CLOSED)
    {
      if (_a_iostream[i].io == _a_IO_WRITE || _a_iostream[i].io == _a_IO_APPEND)
        fflush(_a_iostream[i].fp);
      if (_a_iostream[i].pipe == TRUE)
        pclose(_a_iostream[i].fp);
      else
      {
        if (strcmp(_a_iostream[i].name, "/dev/stdout") &&
            strcmp(_a_iostream[i].name, "/dev/stderr"))
          fclose(_a_iostream[i].fp);
      }
    }
  }

  _awka_kill_ivar();
  exit((int) ret);
}

void
awka_cleanup()
{
  register int i;

  for (i=0; i<_a_ioused; i++)
  {
    if (_a_iostream[i].fp && _a_iostream[i].io != _a_IO_CLOSED)
    {
      if (_a_iostream[i].io == _a_IO_WRITE || _a_iostream[i].io == _a_IO_APPEND)
        fflush(_a_iostream[i].fp);
      if (_a_iostream[i].pipe == TRUE)
        pclose(_a_iostream[i].fp);
      else
      {
        if (strcmp(_a_iostream[i].name, "/dev/stdout") &&
            strcmp(_a_iostream[i].name, "/dev/stderr"))
          fclose(_a_iostream[i].fp);
      }
    }
  }

  _awka_kill_ivar();
  _awka_kill_gvar();
}
