/* Source Installer, Copyright (c) 2005 Claudio Fontana

 package_list.c - list of managed packages

 This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

 This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
     along with this program (look for the file called COPYING);
     if not, write to the Free Software Foundation, Inc.,
         51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

     You can contact the author (Claudio Fontana) by sending a mail
     to claudio@gnu.org

*/

#include "src_stdinc.h"
#include "package_info.h"

void _init_package_element(struct _package_element* e, struct _package_info* info) {
  e->info = info;
  e->next = e->prev = 0;
}

void _free_package_element(struct _package_element* e) {
  if (e->info) {
    _free_package_info(e->info);
    free(e->info); e->info = 0;
  }
}

void _init_package_list(struct _package_list* l) {
  l->count = 0;
  l->first = l->last = 0;
  l->mode = 0;
}

void _free_package_list(struct _package_list* l) {
  struct _package_element* this, *next;

  this = l->first;

  while (this) {
    next = this->next;
    _free_package_element(this);
    free(this);
    this = next;
  }
  _init_package_list(l);
}
    
/* The following function loads the list of known packages.
   The MODE parameter can be used to tweak the memory usage:

   If called with MODE<0, the function will load all the package names plus
   ALL package information. Browsing packages will then require no more load
   operations, but the initial load time will be long.

   If called with MODE=0, the function will load only the package names, and NO
   other package information. Browsing packages will then require a load/free
   for each package browsed. Fast initial load time, but slower
   package-to-package browsing.

   Calling with MODE>0 is reserved for future changes. Currently DO NOT
   do it, as the result is undefined.
*/

int _load_package_list(struct _package_list* l, int mode) {
  struct srcinst_string_list files; size_t i;

  l->mode = mode;
  
  srcinst_init_string_list(&files);
  srcinst_glob_string_list(&files, srcinst_get_dir(SRCINST_DIR_PACK), "*");

  for (i = 0; i < files.count; i++) {
    struct _package_element* e;
    struct _package_info* info;
    
    info = srcinst_malloc(sizeof(struct _package_info));
    _init_package_info(info, files.strings[i]);
    
    e = _add_package_info(l, info);
       
    /* assert (e != 0)  -  a glob cannot contain duplicate filenames
                           Nonetheless we handle the case issuing a warning
                           and skipping the package */

    if (!e) {
      srcinst_warning(SRCINST_ERR_CORE, "duplicate package name, could not add", files.strings[i]);
      continue;
    }
    
    if (mode < 0) {
      /* preload all package information */
      if (!(_load_package_info(e->info))) {
	srcinst_warning(SRCINST_ERR_CORE, "failed to load package", files.strings[i]);
      }
    }
  }

  srcinst_free_string_list(&files);
  return 1;
}

/* this function adds a new package element to the d-linked list.
   We start looking at the end of the list (and move backwards) for the right
   place to add the element. The list is kept in alphabetical order,
   and we start from the end because globbing could give us the
   names is alpha order. */

struct _package_element* _add_package_info(struct _package_list* l, struct _package_info* info) {
  struct _package_element* e;

  e = srcinst_malloc(sizeof(struct _package_element));
  _init_package_element(e, info);

  if (_add_package_element(l, e))
    return e;
  
  _free_package_element(e); free(e);
  return 0;
}

int _add_package_element(struct _package_list* l, struct _package_element* e) {
  struct _package_element* this;

  if (!(this = l->last)) {
    l->first = l->last = e;
    l->count++;
    return 1;
  }
  
  do {
    int check = strcmp(e->info->name, this->info->name);

    if (!check) {
      /* package with the same name! */
      return 0;
    }

    if (check > 0) {
      /* we must insert here */

      if (this->next) {
	this->next->prev = e;
      } else {
	l->last = e;
      }
      e->next = this->next;
      this->next = e;
      e->prev = this;
      break;
    }
    
    this = this->prev;

  } while (this);

  if (!this) {
    /* we went back to the beginning of the list,
       so we have to set e as the new first element */
    l->first->prev = e;
    e->next = l->first;
    e->prev = 0;
    l->first = e;
  }

  l->count++;
  return 1;
}

/* remove a package from a list */

void _remove_package_list(struct _package_list* l, struct _package_element* e) {
  
  _free_package_element(e);

  if (e->prev) {
    e->prev->next = e->next;
  } else {
    l->first = e->next;
  }

  if (e->next) {
    e->next->prev = e->prev;
  } else {
    l->last = e->prev;
  }

  free(e);
  l->count--;
}

void _detach_package_list(struct _package_list* l, struct _package_element* e) {
  if (e->prev) {
    e->prev->next = e->next;
  } else {
    l->first = e->next;
  }

  if (e->next) {
    e->next->prev = e->prev;
  } else {
    l->last = e->prev;
  }
  l->count--;
}

struct _package_element* _search_package_list(struct _package_list* l, char* name) {
  struct _package_element* this = l->first;

  while (this) {
    int check = strcmp(name, this->info->name);
    if (check > 0) 
      this = this->next;
    else if (check == 0)
      return this;
    else 
      return 0;
  }

  return 0;
}


