# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import GDK, string, re
import mpd
import const
import random
import types
from i18n import _
import sys
import gettext

def int_to_intervallname(i, shortname=None, updown=None):
    if shortname:
        n = const.short_intervall_name[abs(i)]
    else:
        n = const.int_intervall[abs(i)]
    if updown:
        if i > 0:
            n = "%s %s" % (n, _("up"))
        elif i < 0:
            n = "%s %s" % (n, _("down"))
    return n


def filter_intervalls_within_range(lowest, highest, irange):
    assert type(lowest) == type(highest) == types.StringType
    lowest = mpd.notename_to_int(lowest)
    highest = mpd.notename_to_int(highest)
    i = highest - lowest
    return filter(lambda x, i=i: abs(x) <= i, irange)

def random_intervall(tonika, lowest, highest, irange):
    """
    Return an int representing the intervall.
    Return None if it is not possible to create an intervall.
    """
    if isinstance(tonika, mpd.MusicalPitch):
        tonika = tonika.semitone_pitch()
    assert type(tonika) == types.IntType
    assert type(lowest) == type(highest)
    if type(lowest) == types.StringType:
        lowest = mpd.notename_to_int(lowest)
    if type(highest) == types.StringType:
        highest = mpd.notename_to_int(highest)
    assert type(lowest) == type(highest) == types.IntType
    assert lowest <= highest
    assert type(irange) == types.ListType
    v = []
    for i in irange:
        if lowest <= tonika + i <= highest:
            v.append(i)
    if not v:
        return None
    return random.choice(v)


def random_tonika_and_intervall(lowest, highest, irange):
    """ Return a tuple (tonika, intervall) of types (MusicalPitch, int).
    Return (None, None) if it is not possible to make an intervall
    because of a conflict between the lowest/highest and the available
    intervalls

    lowest
    highest  an integer or a string representing a notename
    irange   list of integers representing intervalls. 1 is minor
             second up, -2 is major second down
    """
    assert type(lowest) == type(highest)
    if type(lowest) == types.StringType:
        lowest = mpd.notename_to_int(lowest)
    if type(highest) == types.StringType:
        highest = mpd.notename_to_int(highest)
    assert type(lowest) == type(highest) == types.IntType
    assert lowest <= highest
    assert type(irange) == types.ListType
    # first we find out what is the largest intervall we can have
    i = highest - lowest
    # then filter irange to only use intervalls <= i
    v = filter(lambda x, i=i: abs(x) <= i, irange)
    if not v:
        return None, None
    intervall = random.choice(v)
    # then, using that intervall, make a list of possible tonikas
    tl = []
    for t in range(lowest, highest + 1):
       if lowest <= t + intervall <= highest:
           tl.append(t)
    tonika = mpd.MusicalPitch(random.choice(tl))
    return tonika, intervall


def adjust_low_high_to_irange(lowest, highest, irange):
    """
    lowest
    highest  string representing a notename
    irange   list of integers representing intervalls. 1 is minor
             second up, -2 is major second down
    return tuple with two integers
    """
    assert type(lowest) == type(lowest) == types.StringType
    L = mpd.notename_to_int(lowest)
    H = mpd.notename_to_int(highest)
    # find the largest intervall that can be asked for
    r = max(abs(max(irange)), abs(min(irange)))
    # adjust the lowest and highest note, if the intervalls are larger than
    # the range we should try to stay within
    if H - L < r:
        H = H + (r - (H - L)) / 2
        L = L - (r - (H - L))
    return L, H
  

def parse_url(s):
    # protocol, action, fn, collection, lessonfile, config
    r = re.compile("""((?P<protocol>\w+):)? # protocol
           ((?P<action>\w[\w-]*))? #practise|config|statistics))? # 
           (/(?P<fn>[\w\-\.]+))? # exercise or html file name
           (/(?P<collection>\w+))? # collection
           (/(?P<lessonfile>[\w-]+))? # lesson file
           (\?(?P<config>.*))? """, re.VERBOSE|re.DOTALL)
    m = r.match(s)
    if not m.group('protocol'):
        return None, None, s, None, None, {}
    D = {}
    if m.group('config'):
        V = string.split(m.group('config'), ";")
        for e in V:
            n, v = string.split(e, "=")
            n = string.strip(n)
            v = string.strip(v)
            #if v[0] in string.letters:
            D[string.strip(n)] = v#eval("\""+v+"\"")
            #else:
            #    D[string.strip(n)] = eval(v)
    return m.group('protocol'), m.group('action'), m.group('fn'), \
           m.group('collection'), m.group('lessonfile'), D

def get_modifier(s):
    m = (('<ctrl>', GDK.CONTROL_MASK),
         ('<shift>', GDK.SHIFT_MASK),
         ('<alt>', 8))#GDK.MOD1_MASK))
    for mod, mask in m:
        if s[:len(mod)] == mod:
            return mask, s[len(mod):]
    return None, s 

def parse_key_string(string):
    if not string:
        return None, None
    mod = 0
    m, s = get_modifier(string)
    while m:
        mod = mod + m
        m, s = get_modifier(s)
    if len(s) == 1:
        return mod, ord(s)
    else:
        return mod, GDK.__dict__[s]

def exercise_name_to_module_name(name):
    # it is just plain luck that this works...
    return string.replace(name, '-', '')
