# GNU Solfege - eartraining for GNOME
# Copyright (C) 2000-2001  Tom Cato Amundsen
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import gtk, gnome
import gu, widgets, inputwidgets
import abstract, const
import statistics, statisticsviewer
from i18n import _
import types
import soundcard, utils, mpd

class Teacher(abstract.MelodicIntervallTeacher):
    def __init__(self, exname, app):
        abstract.MelodicIntervallTeacher.__init__(self, exname, app)
        self.m_statistics = statistics.IntervallStatistics(self)
        self.m_timeout_handle = None
    def give_up(self):
        """This function is only called *after* the user already has
        answered wrong once, so the statistics are already updated.
        """
        self.q_status = const.QSTATUS_GIVE_UP
    def guess_answer(self, answer, directions=None):
        """
        return TRUE value if correct
        """
        assert self.q_status not in (const.QSTATUS_NO, const.QSTATUS_GIVE_UP)
        if directions:
            q = self.m_question
        else:
            q = map(abs, self.m_question)
        if q == answer:
            if self.q_status == const.QSTATUS_NEW:
                if self.get_int('number_of_intervalls=1') == 1:
                    self.m_statistics.add_correct(self.m_question[0])
            self.maybe_auto_new_question()
            self.q_status = const.QSTATUS_SOLVED
            return 1
        else:
            if self.q_status == const.QSTATUS_NEW:
                if self.get_int('number_of_intervalls=1') == 1:
                    self.m_statistics.add_wrong(self.m_question[0], answer[0])
            self.q_status = const.QSTATUS_WRONG


class Gui(abstract.IntervallGui):
    def __init__(self, teacher, window):
        abstract.IntervallGui.__init__(self, teacher, window)
        self.g_give_up = gu.bButton(self.action_area, _("Give up"), self.give_up)
        self.g_give_up.set_sensitive(gtk.FALSE)
        ##############
        # config_box #
        ##############
        self.g_mici = widgets.MultipleIntervallConfigWidget(self.m_exname)
        self.config_box.pack_start(self.g_mici, gtk.FALSE, gtk.FALSE)
        self.config_box.pack_start(gtk.GtkHSeparator(), gtk.FALSE,
                                   padding=gnome.uiconsts.PAD)
        hbox = gu.bHBox(self.config_box, gtk.FALSE, gtk.FALSE)
        hbox.set_spacing(gnome.uiconsts.PAD)
        hbox.pack_start(gtk.GtkLabel(_("Try to keep question with range from")),
                        gtk.FALSE)
        lowspin = widgets.SimplestNotenameSpinButton(self.get_string('lowest_tone'))
        hbox.pack_start(lowspin, gtk.FALSE)
        gu.bLabel(hbox, _("to"), gtk.FALSE)
        highspin = widgets.SimplestNotenameSpinButton(self.get_string('highest_tone'))
        hbox.pack_start(highspin, gtk.FALSE)
        self.m_notenamerange = widgets.nNotenameRangeController(lowspin, highspin,
                           mpd.LOWEST_NOTENAME, mpd.HIGHEST_NOTENAME,
                           self.m_t.m_exname, 'lowest_tone', 'highest_tone')

        self._add_auto_new_question_gui(self.config_box)
        # ----------------------------------
        hbox = gu.bHBox(self.config_box, gtk.FALSE)
        hbox.set_spacing(gnome.uiconsts.PAD_SMALL)
        gu.bLabel(hbox, _("Input interface:"), gtk.FALSE)
        combo = gu.nCombo(self.m_exname, 'inputwidget', _("Buttons"),
                   inputwidgets.inputwidget_names)
        hbox.pack_start(combo, gtk.FALSE)
        combo.entry.connect('changed',
               lambda entry, self=self, inputwidgets=inputwidgets:
               self.use_inputwidget(inputwidgets.name_to_inputwidget(
                           entry.get_text(), self.click_on_intervall)))
        self.config_box.show_all()
        ##############
        # statistics #
        ##############
        self.setup_statisticsviewer(statisticsviewer.StatisticsViewer,
                                    _("Melodic intervall"))
        self.select_inputwidget()
    def give_up(self, _o=None):
        if self.m_t.q_status == const.QSTATUS_WRONG:
            s = utils.int_to_intervallname(self.m_t.m_question[0], 
                       len(self.m_t.m_question) > 1, 1)
            for n in self.m_t.m_question[1:]:
                s = s + "+" + utils.int_to_intervallname(n, 
                       len(self.m_t.m_question) > 1, 1)
            self.g_flashbar.push(_("The answer is: %s") % s)
            self.m_t.give_up()
            self.g_new_intervall.set_sensitive(gtk.TRUE)
            self.g_give_up.set_sensitive(gtk.FALSE)
    def click_on_intervall(self, mouse_button, intervall, midi_int):
        """The key bindings are also directed here.
        """
        if mouse_button != 1:
            return
        if self.m_t.q_status == const.QSTATUS_NO:
            self.g_flashbar.flash(_("Click 'New intervall' to begin."))
            return
        if midi_int:
            # midi_int is only set when we use some of the instrument widgets,
            # not when we use the buttons interface.
            soundcard.play_note(self.get_int('config/preferred_instrument'),
                        4, 0, midi_int,
                        self.get_int('config/preferred_instrument_velocity'))
        if self.m_t.q_status == const.QSTATUS_GIVE_UP:
            return
        if self.m_t.q_status == const.QSTATUS_SOLVED:
            self.g_flashbar.flash(_("You have already identified this intervall"))
            return
        if not (-17 < intervall < 17):
            self.g_flashbar.flash(_("Ignoring intervalls greater than decim."))
            self.g_input.forget_last_tone()
            return
        self.m_answer.append(intervall)
        if not self.msg:
            self.msg = utils.int_to_intervallname(intervall, 1, 1)
        else:
            self.msg = self.msg + "+" + utils.int_to_intervallname(intervall, 1, 1)
        self.g_flashbar.flash(self.msg)
        if len(self.m_answer) == self.m_number_of_intervalls_in_question:
            if self.m_t.guess_answer(self.m_answer,
                                     self.g_input.know_directions()):
                self.g_flashbar.flash(_("Correct"))
                self.g_new_intervall.set_sensitive(gtk.TRUE)
                self.g_give_up.set_sensitive(gtk.FALSE)
            else:
                self.g_flashbar.flash(_("Wrong"))
                if self.get_bool("config/auto_repeat_question_if_wrong_answer"):
                    self.m_t.play_question()
                self.g_give_up.set_sensitive(gtk.TRUE)
            self.m_answer = []
            self.g_input.set_first_note(self.m_t.m_tonika)
            self.msg = ""
    def new_question(self, _o=None):
        self.msg = ""
        self.m_number_of_intervalls_in_question \
               = self.get_int('number_of_intervalls')
        self.m_answer = []
        ret = self.m_t.new_question(self.get_string('lowest_tone'),
                                    self.get_string('highest_tone'))
        if ret == Teacher.ERR_CONFIGURE:
            self.g_flashbar.clear()
            self.g_flashbar.flash(_("The exercise has to be better configured."))
            self.g_repeat.set_sensitive(gtk.FALSE)
        elif ret == Teacher.OK:
            self.g_repeat.set_sensitive(gtk.TRUE)
            self.g_give_up.set_sensitive(gtk.FALSE)
            self.g_input.set_first_note(self.m_t.m_tonika)
            self.m_t.play_question()
            self.g_new_intervall.set_sensitive(
                  not self.get_bool('config/picky_on_new_question'))
            self.g_flashbar.clear()
        elif ret == Teacher.ERR_PICKY:
            self.g_flashbar.flash(_("You have to solve this question first."))
    def on_end_practise(self):
        self.g_new_intervall.set_sensitive(gtk.TRUE)
        self.g_repeat.set_sensitive(gtk.FALSE)
        self.g_give_up.set_sensitive(gtk.FALSE)
        self.g_input.clear()
        self.g_flashbar.clear()
        self.m_t.end_practise()
