"======================================================================
|
|   URL
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================
"

Object subclass: #URL
       instanceVariableNames: 'protocol host port file anchor cachedHash'
       classVariableNames: 'NoPercentEncoding ConnectionClasses DefaultProtocol DefaultFile'
       poolDictionaries: ''
       category: 'Sockets-RFC'
!

!URL class methodsFor: 'encoding URLs'!

decode: anURL
    | result in ch |
    result := WriteStream on: (String new: anURL size).
    in := ReadStream on: anURL.
    [ in atEnd ] whileFalse: [
        (ch := in next) = $+
            ifTrue: [ result nextPut: $  ]
            ifFalse: [
                 ch = $%
                    ifFalse: [ result nextPut: ch ]
                    ifTrue: [
                        ch := in next digitValue * 16 + in next digitValue.
                        result nextPut: ch asCharacter
                    ]
            ]
    ].
    ^result contents
!

encode: anURL
    | result value |
    result := WriteStream on: (String new: anURL size + 10).
    anURL do: [ :each |
        (each = $ )
            ifTrue: [ result nextPut: $+ ]
            ifFalse: [
                value := each value.
                (NoPercentEncoding at: value) = 1
                    ifTrue: [ result nextPut: each ]
                    ifFalse: [
                        result
                            nextPut: $%;
                            nextPut: ('0123456789ABCDEF' at: value // 16 + 1);
                            nextPut: ('0123456789ABCDEF' at: value \\ 16 + 1)
                    ]
            ]
    ].
    ^result contents
!

initialize
    ConnectionClasses := Dictionary new.
    DefaultProtocol := 'http:'.
    DefaultFile := 'index.htm'.

    NoPercentEncoding := ByteArray new: 256.
    'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ -_.*0123456789'
        do: [ :each |
            NoPercentEncoding at: each value put: 1
        ]
! !

!URL class methodsFor: 'accessing'!

defaultFile
    ^DefaultFile
!

defaultFile: file
    DefaultFile := file
!

defaultProtocol
    ^DefaultProtocol
!

defaultProtocol: protocol
    DefaultProtocol := protocol
!

use: aURLConnectionClass forProtocol: aString
    ^ConnectionClasses at: aString put: aURLConnectionClass
! !

!URL class methodsFor: 'private'!

file: file relativeTo: url into: new
    | spec host port first i j |

    spec := file.
    first := spec findFirst: [ :each | each ~= $  ].
    (first + 3 <= spec size)
        ifTrue: [
            (spec copyFrom: first to: first + 3) = 'url:'
                ifTrue: [ first := first + 4 ]
        ].

    "Remove leading and trailing white space"
    spec := spec
        copyFrom: first
        to: (spec findLast: [ :each | each ~= $  ]).

    (spec at: 1) = $# ifTrue: [
         url isNil ifFalse: [ self error: 'missing file in absolute URL' ].
         new anchor: (spec copyFrom: 2 to: spec size).
         new calculateHash.
         ^new
    ].

    "Search for :/ - if present, the protocol is from the start to :"
    i := spec indexOf: $/.
    i > 1 ifTrue: [
        (spec at: i - 1) = $:
            ifTrue: [ new protocol: (spec copyFrom: 1 to: i - 1) ].

        "If the protocol changed, the URL must be absolute"
        new protocol = url protocol ifFalse: [
            new host: nil; port: -1; file: nil; anchor: nil
        ]
    ].
    new protocol isNil ifTrue: [ new protocol: self defaultProtocol ].

    "If <protocol>//<xxx>, <xxx> is the host and, maybe, the port.
        i is here-^						"

    ((i < spec size) and: [ (spec at: i + 1) = $/ ]) ifTrue: [
        j := spec indexOf: $/ startingAt: i + 2 ifAbsent: [ spec size + 1 ].
        host := spec copyFrom: i + 2 to: j - 1.
        i := j.

        new port: -1.
        j := host indexOf: $:.
        j = 0
            ifFalse: [
                port := host copyFrom: j + 1 to: host size.
                host := host copyFrom: 1 to: j - 1.
                port isEmpty ifFalse: [
                    (port allSatisfy: [ :each | each isDigit ])
                        ifFalse: [ self error: 'non-numeric character in port' ]
                        ifTrue: [ new port: port asInteger ]
                ]
            ].

        host isEmpty ifTrue: [ self error: 'empty host name' ]
        new host: host.
    ].

    i >= spec size ifTrue: [
        ^new file: self defaultFile; anchor: nil; calculateHash
    ].

    j := spec indexOf: $# startingAt: i + 1 ifAbsent: [ spec size + 1 ].
    new file: (spec copyFrom: i + 1 to: j - 1).

    j < spec size
        ifTrue: [ new anchor: (spec copyFrom: j + 1 to: spec size) ]
        ifFalse: [ new anchor: nil ].

    ^new calculateHash
! !

!URL class methodsFor: 'instance creation'!

file: spec relativeTo: url
    ^self file: spec relativeTo: url into: url copy
!

fromString: aString
    ^self file: aString relativeTo: nil into: URL basicNew
!

new
    self shouldNotImplement
!

protocol: protocol host: host file: file
    ^self protocol: protocol host: host port: -1 file: file
!

protocol: protocol host: host port: port file: file
    | anchor pos |
    pos := file indexOf: $# ifAbsent: [ file size + 1 ]
    pos < file size
        ifTrue: [ anchor := file copyFrom: pos + 1 to: file size ].

    ^URL basicNew
        protocol: protocol;
        host: host;
        port: port;
        file: (file copyFrom: 1 to: pos - 1);
        anchor: anchor;
        calculateHash
! !

!URL methodsFor: 'accessing'!

= anURL
    self hash = anURL hash ifFalse: [ ^false ].
    self class == anURL class ifFalse: [ ^false ].

    ^self protocol = anURL protocol and: [
     self host = anURL host and: [
     self port = anURL port and: [
     self file = anURL file and: [
     self anchor = anURL anchor ]]]]
!

anchor
    ^anchor
!

contents
    ^self newConnection contents
!

file
    ^file
!

hash
    ^cachedHash
!

host
    ^host
!

newConnection
    ^(ConnectionClasses at: self protocol) on: self
!

port
    ^port = -1
        ifTrue: [ Socket defaultPortAt: self protocol ifAbsent: [ -1 ] ]
        ifFalse: [ port ]
!

protocol
    ^protocol
!

relative: urlSpec
    ^self species file: urlSpec relativeTo: self
!

sameFileAs: anURL
    ^self = anURL or: [ self anchor ~= anURL anchor ]
!

sameHostAs: anURL
    ^self host = anURL host or: [
        (IPAddress byName: self host) = (IPAddress byName: anURL host) ]
! !

!URL methodsFor: 'printing'!

printOn: aStream

    aStream
        nextPutAll: self protocol;
        nextPut: $/;
        nextPut: $/;
        nextPutAll: self host.

    port > 0 ifTrue: [
        aStream
            nextPut: $: ;
            print: self port
    ].
    aStream
        nextPut: $/ ;
        nextPutAll: self file.

    anchor isNil ifFalse: [
        aStream
            nextPut: $# ;
            nextPutAll: self anchor
    ].
! !

!URL methodsFor: 'private'!

calculateHash
    cachedHash := self printString hash
!

protocol: newProtocol
    protocol := newProtocol asLowercase
!

host: newHost
    host := newHost asLowercase
!

file: newFile
    file := newFile
!

port: newPort
    port := newPort asLowercase
!

anchor: newAnchor
    anchor := newAnchor
! !
