"======================================================================
|
|   Smalltalk eight queens
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Object subclass: #NullQueen
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Examples-Classic'!

!NullQueen methodsFor: 'queens'!

move
    "Move the queen so that it is not menaced, backtracking if necessary.
     Answer whether a position can be found.
     If the null queen is asked to advance, the search tree has been walked
     entirely - so return false."
    ^false
!

check: test
    "Answer whether a queen is menaced in the given position by the queens
     up to and including the receiver. The null queen does not menace
     anything."
    ^false
!

result
    "Answer all the queens' rows, up to and including the receiver"
    ^OrderedCollection new
!

next
    "Answer a solution, or nil if there aren't anymore"
    ^self move
        ifTrue: [ self result ]
        ifFalse: [ nil ]
!!

NullQueen subclass: #Queen
    instanceVariableNames: 'row column neighbor rows'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Examples-Classic'!
!!

!Queen class methodsFor: 'initializing'!

test: side
    | lastQueen result line n |
    (line := String new: side * 2 + 1)
        atAll: (1 to: side * 2 + 1 by: 2) put: $|;
        atAll: (2 to: side * 2 + 1 by: 2) put: $_.

    n := 0.
    lastQueen := self board: side.
    [ result := lastQueen next. result notNil ] whileTrue: [
        n := n + 1.
        Transcript
            space;
            next: side * 2 - 1 put: $_;
            nl.

        result do: [:x |
    	    line at: x + x put: $*.
    	    Transcript nextPutAll: line; nl.
    	    line at: x + x put: $_.
        ].
        Transcript nl.
    ].
    Transcript nl.
    ^n!

board: n
    "Answer a Queen which will return results for a chessboard of side n"
    ^(1 to: n) inject: NullQueen new into: [ :neighbor :column |
        self new
	    setColumn: column
	    rows: n
	    neighbor: neighbor
    ]
!!

!Queen methodsFor: 'private'!

setColumn: aNumber rows: n neighbor: aQueen
    "Initialize the receiver to work on column aNumber of a chessboard of
     side n, having aQueen as a neighbor"
    column := aNumber.
    rows := n.
    neighbor := aQueen.
    row := 0.
    
    "Put all the queens but the last in some place where they are safe. The
     last will be requested by sending #next"
    self neighbor move.
    ^self
!

advance
    "Move the receiver one row further if possible, else backtrack and move
     to the first row. Answer whether the neighbor could find a safe position."
    ^row = rows
        ifTrue: [ row := 1. self neighbor move ]
        ifFalse: [ row := row + 1. true ].
!

row
    ^row
!

column
    ^column
!

neighbor
    ^neighbor
!!

!Queen methodsFor: 'inherited'!

check: test
    | columnDifference rowDifference |
    columnDifference := (test column - self column) abs.
    rowDifference := (test row - self row) abs.

    rowDifference = 0                ifTrue: [ ^true ].
    rowDifference = columnDifference ifTrue: [ ^true ].

    ^self neighbor check: test
!

move
    "Here and in #advance is where the search really takes place.
     We advance the queen to the next cell; if the edge has been reached,
     #advance takes care of backtracking by sending #move to the neighbor
     (which in turn could backtrack).
     If the queen is safe there, return true; else advance the queen
     again. Sooner or later every queen will reach the edge and ask its
     neighbor to advance. When the first queen asks the NullQueen to
     advance, all the invocation of #move will answer false and the search
     will end."
    [   self advance ifFalse: [ ^false ].
	self neighbor check: self
    ] whileTrue: [ ].
    ^true
!

result
    ^self neighbor result
	addLast: row;
	yourself
!!

Queen subclass: #Amazon
    instanceVariableNames: 'row column neighbor rows'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Examples-Classic'!
!!

!Amazon methodsFor: 'inherited'!

check: test
    | columnDifference rowDifference |
    columnDifference := (test column - self column) abs.
    rowDifference := (test row - self row) abs.

    rowDifference        = 0                ifTrue: [ ^true ].
    rowDifference        = columnDifference ifTrue: [ ^true ].
    rowDifference * 2    = columnDifference ifTrue: [ ^true ].
    columnDifference * 2 = rowDifference    ifTrue: [ ^true ].

    ^self neighbor check: test
!!

"	EVALUATE THIS:		"	"RESULT	"
"	^Queen test: 3!		"	"0	"
"	^Queen test: 4!		"	"2	"
"	^Queen test: 8!		"	"92	"
"	^Amazon test: 8!	"	"0	"
"	^Amazon test: 10!	"	"4	"
