"======================================================================
|
|   Smalltalk Tk-based GUI building blocks (text widget).
|
|   $Revision: 1.6.2$
|   $Date: 1999/08/31 11:23:18$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1990, 91, 92, 94, 95, 99 Free Software Foundation, Inc.
| Written by Steve Byrne.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

"-------------------------- Main text widget class ------------------"

BViewport subclass: #BText
     instanceVariableNames: 'callback tagInfo'
     classVariableNames: ''
     poolDictionaries: ''
     category: 'Graphics-Windows'
! 

BText comment: '
I represent a text viewer with pretty good formatting options.'!

BText
    "#char, #word, #none"
    defineAccessor: 'wrap' option: 'wrap' fromStringCode: 'asSymbol';
    defineMutator: 'wrap' option: 'wrap' toStringCode: '';
    defineColorProtocolWithActivePrefix: 'select'!

!BText class methodsFor: 'accessing'!

emacsLike
    self tclEval: 'return $tk_strictMotif'.
    ^self tclResult = '1'
!

emacsLike: aBoolean
    self tclEval:
        'set tk_strictMotif ', (aBoolean ifTrue: [ '0' ] ifFalse: [ '1' ]).
! !

!BText class methodsFor: 'instance creation'!

newReadOnly: parent
    | ctl |
    ctl := self new: parent.
    ctl tclEval: ctl connected, ' configure -state disabled'.
    ^ctl
! !

!BText methodsFor: 'accessing'!

callback: aReceiver message: aString
    callback := DirectedMessage
        selector: aString asSymbol
        arguments: #()
        receiver: aReceiver
!

contents
    self tclEval: self connected, ' get 1.0 end-1c'.
    ^self tclResult
!

contents: aString
    self tclEval: self connected, ' delete 1.0 end'.
    self tclEval: self connected, ' insert 1.0 ', aString asTkString.
    self tclEval: self connected, ' see 1.0'.
!

getSelection
    | result |
    self tclEval: ('
        if { [%1 tag ranges sel] == {} } then { return {} }
        return [%1 get sel.first sel.last]' bindWith: self connected).

    result := self tclResult.
    ^result isEmpty ifTrue: [ nil ] ifFalse: [ result ]
! !

!BText methodsFor: 'widget protocol'!

currentPosition
    | stream y |
    self tclEval: 'return [', self connected, ' index insert]'.
    stream := ReadStream on: self tclResult.
    y := (stream upTo: $.) asNumber.
    ^stream upToEnd asNumber + 1 @ y
!

currentPosition: aPoint
    | stream y |
    self tclEval: ('
      %1 mark set insert %2.%3
      %1 see insert' 
        bindWith: self connected
        with: aPoint y printString
        with: (aPoint x + 1) printString).
!

currentColumn
    | stream |
    self tclEval: 'return [', self connected, ' index insert]'.
    stream := ReadStream on: self tclResult.
    stream skipTo: $. .
    ^stream upToEnd asNumber + 1
!

currentLine
    | stream |
    self tclEval: 'return [', self connected, ' index insert]'.
    stream := ReadStream on: self tclResult.
    ^(stream upTo: $.) asNumber
!

gotoLine: line end: aBoolean

    | code |
    code := aBoolean
        ifTrue: [ '%1 mark set insert "%2.0 -1l lineend"' ]
        ifFalse: [ '%1 mark set insert %2.0' ].

    self tclEval: (code bindWith: self connected with: line printString).
    self tclEval: self connected, ' see insert'.
    ^1
!

insertText: aString
    self tclEval: self connected, ' delete sel.first sel.last'.
    self tclEval: self connected, ' insert insert ', aString asTkString.
    self tclEval: self connected, ' see insert'.
!

insertSelectedText: aString
    self tclEval: self connected, ' tag remove sel 1.0 end'.
    self tclEval: self connected, ' insert insert ', aString asTkString, ' { sel }'.
    self tclEval: self connected, ' see insert'.
!

insertTextSelection: aString
    self tclEval: 'catch {', self connected, ' mark set insert sel.last }'.
    self tclEval: self connected, ' tag remove sel 1.0 end'.
    self tclEval: self connected, ' insert insert ', aString asTkString, ' { sel }'.
    self tclEval: self connected, ' see insert'.
!

nextPut: aCharacter
    self insertAtEnd: (String with: aCharacter)
!

nextPutAll: aString
    self insertAtEnd: aString
!

nl
    self insertAtEnd: Character nl asString
!

insertAtEnd: aString
    self tclEval: self connected, ' tag remove sel 1.0 end'.
    self tclEval: self connected, ' insert end ', aString asTkString.
    "self tclEval: self connected, ' mark set insert end'.
    self tclEval: self connected, ' see end'."
!

insertText: aString attribute: attr
    tagInfo isNil ifTrue: [ tagInfo := BTextTags new: self ].
    self tclEval: self connected, ' delete sel.first sel.last'.
    self tclEval: self connected, ' insert insert ', aString asTkString, (attr tags: tagInfo).
    self tclEval: self connected, ' see insert'.
!

insertAtEnd: aString attribute: attr
    tagInfo isNil ifTrue: [ tagInfo := BTextTags new: self ].
    self tclEval: self connected, ' tag remove sel 1.0 end'.
    self tclEval: self connected, ' insert end ', aString asTkString, (attr tags: tagInfo).
    "self tclEval: self connected, ' mark set insert end'.
    self tclEval: self connected, ' see end'."
!


refuseTabs
    self tclEval: ('
        bind %1 <Tab> {
            focus [tk_focusNext %W]
            break
        }
        bind %1 <Shift-Tab> {
            focus [tk_focusPrev %W]
            break
        }' bindWith: self connected)
!

replaceSelection: aString
    self tclEval: 'catch { ', self connected, ' delete sel.first sel.last }'.
    self tclEval: self connected, ' insert insert ', aString asTkString, ' { sel }'.
    self tclEval: self connected, ' see insert'.
!

searchString: aString
    | result |
    self tclEval: 'return [', self connected, ' search ',
        aString asTkString, ' 1.0 end]'.

    result := self tclResult.
    result isEmpty ifTrue: [ ^0 ].
    
    self tclEval: self connected, ' mark set insert ', result.
    self tclEval: self connected, ' see insert'.

    "Sending truncated removes the column"
    ^result asNumber truncated
!

space
    self insertAtEnd: ' '
! !

!BText methodsFor: 'private'!

addChild: child
    self tclEval: self connected, ' window create end -window ', child container.
    ^self basicAddChild: child
!

callback
    ^callback
!

create
    self create:
        ' -bg white -wrap word -state normal -takefocus 1'.

    "Tcl hack to get the callback upon insert or delete
     See Tk faq by Jeffrey Hobbs (jeff.hobbs@acm.org)"

    self tclEval: ('
      rename %1 .%1
      proc %1 args {
        if [regexp {^(ins|del).*} [lindex $args 0]] { callback %2 invokeCallback }
        uplevel .%1 $args
      }' bindWith: self connected with: self asOop printString).

!

defineTag: name as: options
    self tclEval: self connected, ' tag configure ', name, options
!

tag: name bind: event to: aSymbol of: anObject parameters: params

    self
        bind: event
        to: aSymbol
        of: anObject
        parameters: params
        prefix: self connected, ' tag bind ', name
!

invokeCallback
    self callback isNil ifFalse: [ self callback send ]
!

widgetType
    ^'text '
! !

"--------------------------- BTextBindings --------------------------"

BEventTarget subclass: #BTextBindings
       instanceVariableNames: 'list tagName'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'!

!BTextBindings class methodsFor: 'instance creation'!

new
    ^self basicNew initialize
! !

!BTextBindings methodsFor: 'BTextTags protocol'!

defineTagFor: aBText
    list do: [ :each | each sendTo: aBText ]
!

tagName
    ^tagName at: 1
! !

!BTextBindings methodsFor: 'private'!

initialize
    tagName := Blox uniqueName.
    tagName := tagName copyFrom: 2 to: tagName size.  "without the initial dot"
    tagName := Array with: tagName.

    list := OrderedCollection new.
!

primBind: event to: aSymbol of: anObject parameters: params
    list add: (Message
        selector: #tag:bind:to:of:parameters:
        arguments: tagName, (Array with: event with: aSymbol with: anObject with: params))
! !


"-------------------------- BTextAttributes -------------------------"

Object subclass: #BTextAttributes
       instanceVariableNames: 'bgColor fgColor font styles events'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'!

BTextAttributes comment: '
I help you creating wonderful, colorful BTexts.'!


!BTextAttributes class methodsFor: 'instance-creation shortcuts'!

backgroundColor: color
    ^self new backgroundColor: color
!

foregroundColor: color
    ^self new foregroundColor: color
!

events: aBTextBindings
    ^self new events: aBTextBindings
!

font: font
    ^self new font: font
!

strikeout
    ^self new strikeout
!

underline
    ^self new underline
!

center
    ^self new center
!

red
    ^self new foregroundColor: 'red'
!

blue
    ^self new foregroundColor: 'blue'
!

darkGreen
    ^self new foregroundColor: 'PureDarkGreen'
!

green
    ^self new foregroundColor: 'green'
!

darkCyan
    ^self new foregroundColor: 'PureDarkCyan'
!

cyan
    ^self new foregroundColor: 'cyan'
!

darkMagenta
    ^self new foregroundColor: 'PureDarkMagenta'
!

magenta
    ^self new foregroundColor: 'magenta'
!

yellow
    ^self new foregroundColor: 'yellow'
!

white
    ^self new foregroundColor: 'white'
!

black
    ^self new foregroundColor: 'black'
!!

!BTextAttributes methodsFor: 'setting attributes'!

backgroundColor
    ^bgColor
!

backgroundColor: color
    bgColor := color
!

foregroundColor
    ^bgColor
!

foregroundColor: color
    fgColor := color
!

events
    ^events
!

events: aBTextBindings
    events := aBTextBindings
!

font
    ^font
!

font: fontName
    font := fontName
!

isStruckout
    ^self hasStyle: #STYLEstrikeout
!

strikeout
    self style: #STYLEstrikeout
!

isUnderlined
    ^self hasStyle: #STYLEunderline
!

underline
    self style: #STYLEunderline
!

isCentered
    self hasStyle: #STYLEcenter
!

center
    self style: #STYLEcenter
!!

!BTextAttributes methodsFor: 'colors'!

red
    self foregroundColor: 'red'
!

blue
    self foregroundColor: 'blue'
!

darkGreen
    self foregroundColor: 'PureDarkGreen'
!

green
    self foregroundColor: 'green'
!

darkCyan
    self foregroundColor: 'PureDarkCyan'
!

cyan
    self foregroundColor: 'cyan'
!

darkMagenta
    self foregroundColor: 'PureDarkMagenta'
!

magenta
    self foregroundColor: 'magenta'
!

yellow
    self foregroundColor: 'yellow'
!

white
    self foregroundColor: 'white'
!

black
    self foregroundColor: 'black'
!!

!BTextAttributes methodsFor: 'private'!

hasStyle: aSymbol
    ^styles notNil and: [ styles includes: aSymbol ]
!

style: aSymbol
    styles isNil ifTrue: [ styles := Set new ].
    styles add: aSymbol
!

tags: aBTextTags
    | s |
    s := WriteStream on: (String new: 20).
    s nextPutAll: ' {'.

    fgColor isNil ifFalse: [ s nextPutAll: (aBTextTags fgColor: fgColor) ].
    bgColor isNil ifFalse: [ s nextPutAll: (aBTextTags bgColor: bgColor) ].
    font    isNil ifFalse: [ s nextPutAll: (aBTextTags font:    font)    ].
    events  isNil ifFalse: [ s nextPutAll: (aBTextTags events:  events)  ].

    styles isNil ifFalse: [
        styles do: [ :each | s nextPut: $ ; nextPutAll: each ]
    ].
    s nextPut: $}.
    ^s contents
!!

"-------------------------- BTextTags - private ---------------------"

Object subclass: #BTextTags
       instanceVariableNames: 'client tags'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Graphics-Windows'!

BTextTags comment: '
I am a private class. I sit between a BText and BTextAttributes, helping
the latter in telling the former which attributes to use.'!


!BTextTags class methodsFor: 'instance creation'!

new
    self shouldNotImplement
!

new: client
    ^super new initialize: client
!!

!BTextTags methodsFor: 'BTextAttributes protocol'!

bgColor: color
    ^' b_', (self color: color)
!

fgColor: color
    ^' f_', (self color: color)
!

events: aBTextBindings
    | tagName |
    tagName := aBTextBindings tagName.
    (tags includes: tagName) ifFalse: [
         tags add: tagName.
         aBTextBindings defineTagFor: client.
    ].
    ^' ', tagName
!

font: font
    | tagName |
    tagName := WriteStream on: (String new: 20).
    font substrings do: [ :each | tagName nextPutAll: each; nextPut: $_ ].
    tagName := tagName contents.

    (tags includes: tagName)
        ifFalse: [
            tags add: tagName.
            client defineTag: tagName as: ' -font {', font, '}'
        ].

    ^' ', tagName
!!

!BTextTags methodsFor: 'private'!

color: color
    | tagName |
    tagName := (color at: 1) = $#
        ifTrue: [ color copy at: 1 put: $_; yourself ]
        ifFalse: [ color asLowercase ].

    (tags includes: tagName)
        ifFalse: [
            tags add: tagName.
            client defineTag: 'f_', tagName as: ' -foreground ', color.
            client defineTag: 'b_', tagName as: ' -background ', color
        ].
    ^tagName
!


initialize: clientBText
    client := clientBText.
    tags := Set new.
    client defineTag: 'STYLEstrikeout' as: ' -overstrike 1'.
    client defineTag: 'STYLEunderline' as: ' -underline 1'.
    client defineTag: 'STYLEcenter' as: ' -justify center'.
!!
