/* menus.c
 * Written by David Allen <s2mdalle@titan.vcu.edu>
 * http://opop.nols.com/
 *
 * This file handles menu generation for GTKeyboard's main window.
 */
/* GTKeyboard - A Graphical Keyboard For X
 * Copyright (C) 1999, 2000 David Allen  
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */

#define MENUS_C
#include "master.h"

GtkAccelGroup *ac = NULL;
GList *bindings = NULL;

/* This structure specifies the binding between a given GtkWidget,
 * (which for our purposes is always GtkCheckMenuItem)
 * and an integer slot which holds what the state of the item should be.
 */
typedef struct {
     GtkWidget *widget;
     int *boolean;
     char *description;
} menubinding;

/* Clean up one node in the bindings doubly linked list */
static void cleanup_menu_binding(gpointer data, gpointer user_data)
{
     menubinding *foo = (menubinding *)data;
     if(foo && foo->description)
     {
          g_free_(foo->description);
     } /* End if */

     g_free_(foo);
} /* End cleanup_menu_item() */

/* call from gtkeyboard_mem_cleanup to clean up all menu specific garbage
 * like the bindings
 */
void menus_cleanup(void)
{
     if(bindings)
     {
          g_list_foreach(bindings, (GFunc)cleanup_menu_binding, NULL);
          g_list_free(bindings);
     } /* End if */
} /* End menus_cleanup() */

/* This creates a binding between an integer and a specific GtkCheckMenuItem
 * in the menu system.  Some are supposed to be connected, i.e. when the
 * boolean is false, the checkmenuitem should always be up, and vice versa,
 * etc.  This establishes that binding, so that whenever one of the booleans
 * changes, the changer only needs to call a function, and all menus will
 * be updated
 */
static void bind_variable_to_boolean(GtkWidget *widget, int *boolptr, char *d)
{
     menubinding *toinsert = g_new0_(menubinding, 1);
     
     toinsert->widget      = widget;
     toinsert->boolean     = boolptr;
     toinsert->description = d;
     bindings = g_list_append(bindings, toinsert);
} /* End bind_variable_to_pointer() */

/* Iterator function over bindings for update_menus() */
static void update_single_item(gpointer data, gpointer user_data)
{
     /* Each element in the GList is a menubinding */
     menubinding *bound = (menubinding *)data;
     GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM(bound->widget);
     
     if(item->active != *bound->boolean)
     {
          gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), 
                                         *bound->boolean);
#ifdef MENU_DEBUGGING
          fprintf(Q,"update_single_item: toggled item \"%s\".\n",
                  bound->description ? bound->description : "unknown");
          fflush(Q);
#endif /* MENU_DEBUGGING */
     } /* End if */
} /* End update_single_item */

void update_menus(void)
{
     /* Iterate over each item in the doubly linked list updating
      * the booleans on it.
      */
     g_list_foreach(bindings, (GFunc)update_single_item, NULL);
} /* End update_menus() */

static GtkWidget *menu_hseparator_new(void)
{
     GtkWidget *item = gtk_menu_item_new();
     GtkWidget *sep = gtk_hseparator_new();
     gtk_container_add(GTK_CONTAINER(item), sep);
     return(item);
}

static GtkWidget *named_menu_separator_new(char *name)
{
     GtkWidget *hbox     = gtk_hbox_new(FALSE, 0);
     GtkWidget *label    = gtk_label_new(name);
     GtkWidget *hleft    = gtk_hseparator_new();
     GtkWidget *hright   = gtk_hseparator_new();
     GtkWidget *menuitem = gtk_menu_item_new();

     gtk_box_pack_start(GTK_BOX(hbox), hleft, TRUE, TRUE, 0);
     gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
     gtk_box_pack_start(GTK_BOX(hbox), hright, TRUE, TRUE, 0);
     gtk_container_add(GTK_CONTAINER(menuitem), hbox);
     return(menuitem);
} /* End named_menu_separator_new() */

static GtkWidget *create_checked_item(char *name, GtkSignalFunc func,
                                      gpointer data, gboolean initial)
{
     GtkWidget *menuitem = gtk_check_menu_item_new_with_label(name);
     gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem), initial);   
     gtk_signal_connect(GTK_OBJECT(menuitem), "toggled", 
                        GTK_SIGNAL_FUNC(func), data);
     gtk_check_menu_item_set_show_toggle(GTK_CHECK_MENU_ITEM(menuitem), TRUE);
     gtk_widget_show(menuitem);
     return(menuitem);
} /* End create_checked_item() */

typedef struct {
     GtkSignalFunc show;
     GtkSignalFunc hide;
     gpointer showdata;
     gpointer hidedata;
} MenuHandler;

/* Takes a MenuHandler pointer through
 * the data argument and executes one of the two choice
 * functions depending on the state of the check button
 */
static void hide_show_callback(GtkWidget *emitter, gpointer data)
{
     MenuHandler *foo = data;
     GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM(emitter);
     
     if(item->active)
     {
          foo->show(emitter, GPOINTER_TO_INT(foo->showdata));
     } /* End if */
     else 
     {
          foo->hide(emitter, GPOINTER_TO_INT(foo->hidedata));
     } /* End else */

     return;
} /* End hide_show_callback() */

static GtkWidget *create_hide_show_toggle_button(char *name, GtkSignalFunc
                                                 hide, gpointer hidedata,
                                                 GtkSignalFunc show, 
                                                 gpointer showdata, gboolean
                                                 initial)
{
     MenuHandler *ptr;
     GtkWidget *item;

     ptr = g_new0_(MenuHandler, 1);
     ptr->show = show;
     ptr->hide = hide;
     ptr->showdata = showdata;
     ptr->hidedata = hidedata;
     
     item = create_checked_item(name, GTK_SIGNAL_FUNC(hide_show_callback),
                                ptr, initial);
     return(item);
} /* End create_hide_show_toggle_button() */
                                                 
static GtkWidget *create_menu_item(char *label, char *accel, char *tooltip,
                                   GtkSignalFunc func, gpointer data)
{
     GtkWidget *menuitem = NULL;
     
     if(label && strlen(label))
     {
          menuitem = gtk_menu_item_new_with_label(label);

          if(func)
               gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                                  GTK_SIGNAL_FUNC(func), data);
     } /* End if */
     else 
     {
          menuitem = gtk_menu_item_new();
          if(func)
               gtk_signal_connect(GTK_OBJECT(menuitem), "activate",
                                  GTK_SIGNAL_FUNC(func), data);
     }

     if(accel && accel[0] == '^')
     {
          gtk_widget_add_accelerator(menuitem, "activate", ac,
                                     accel[1], GDK_CONTROL_MASK,
                                     GTK_ACCEL_VISIBLE);
     } /* End if */

     if(tooltip && strlen(tooltip))
     {
          add_tooltip(menuitem, tooltip);
     } /* End if */

     gtk_widget_show(menuitem);
     return(menuitem);
} /* End create_menu_item() */

static GtkWidget *create_file_menu(void)
{
     GtkWidget *menu    = gtk_menu_new();
     GtkWidget *new, *open, *save, *save_as, *quit;

     ac  = gtk_accel_group_new();

     new     = create_menu_item("New", "^N", NULL, 
                                GTK_SIGNAL_FUNC(NewFile), NULL);
     open    = create_menu_item("Open", "^O", NULL,
                                GTK_SIGNAL_FUNC(OpenFile), NULL);
     save    = create_menu_item("Save", "^S", NULL,
                                GTK_SIGNAL_FUNC(save_output_text), NULL);
     save_as = create_menu_item("Save As", "^A", NULL,
                                GTK_SIGNAL_FUNC(save_file_as), NULL);
     quit    = create_menu_item("Quit", "^Q", NULL,
                                GTK_SIGNAL_FUNC(final_check), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), new);
     gtk_menu_append(GTK_MENU(menu), open);
     gtk_menu_append(GTK_MENU(menu), save);
     gtk_menu_append(GTK_MENU(menu), save_as);
     gtk_menu_append(GTK_MENU(menu), quit);
     
     return(menu);
} /* ENd create_file_menu() */

static GtkWidget *create_edit_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *cut, *copy, *paste, *find;

     cut   = create_menu_item("Cut", "^X", NULL,
                              GTK_SIGNAL_FUNC(CutText), NULL);
     copy  = create_menu_item("Copy", "^C", NULL,
                              GTK_SIGNAL_FUNC(CopyText), NULL);
     paste = create_menu_item("Paste", "^V", NULL,
                              GTK_SIGNAL_FUNC(PasteText), NULL);
     find  = create_menu_item("Find", "^F", NULL,
                              GTK_SIGNAL_FUNC(find_dialog), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), cut);
     gtk_menu_append(GTK_MENU(menu), copy);
     gtk_menu_append(GTK_MENU(menu), paste);
     gtk_menu_append(GTK_MENU(menu), find);
     return(menu);
} /* End create_edit_menu() */

typedef struct {
     char *text;
     GtkSignalFunc function;
     gpointer data;
} SimpleCallback;

static GtkWidget *menu_generator(SimpleCallback *data, int len)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *item = NULL;
     int x=0;

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     
     while(x < len)
     {
          item = create_menu_item(data[x].text, NULL, NULL, 
                                  data[x].function, data[x].data);
          gtk_menu_append(GTK_MENU(menu), item);
          x++;
     } /* End while */

     return(menu);
} /* End menu_generator() */

static GtkWidget *create_HTML_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *document=NULL, *body=NULL, *text=NULL, *headers=NULL, 
          *lists=NULL, *tables=NULL, *protocols=NULL, *internal=NULL,
          *misc=NULL, *paragraph=NULL, *image=NULL, *link=NULL, *template=NULL;
     GtkWidget *mdocument=NULL, *mbody=NULL, *mtext=NULL, *mheaders=NULL,
          *mlists=NULL, *mtables=NULL, *mprotocols=NULL, *minternals=NULL,
          *mmisc;
     
#define GSF(x)                GTK_SIGNAL_FUNC(x)
     SimpleCallback cdocument [] = {
          { "Doctype",        GSF(html_doctype),    NULL },
          { "Open Document",  GSF(html_html),       NULL },
          { "Close Document", GSF(html_html_close), NULL },
          { "Head",           GSF(html_head),       NULL },
          { "Close Head",     GSF(html_head_close), NULL },
          { "Title",          GSF(html_title),      NULL } };
     SimpleCallback cbody [] = {
          { "Open Body",      GSF(html_body),       NULL },
          { "Close Body",     GSF(html_body_close), NULL } };
     SimpleCallback ctext [] = {
          { "Bold",           GSF(html_bold),       NULL },
          { "Strong",         GSF(html_strong),     NULL },
          { "Italics",        GSF(html_italics),    NULL },
          { "Emphasis",       GSF(html_em),         NULL },
          { "Center",         GSF(html_center),     NULL },
          { "Preformatted",   GSF(html_pre),        NULL },
          { "Font Color",     GSF(html_font_color), NULL },
          { "Font Size",      GSF(html_font_size),  NULL },
          { "Strikeout",      GSF(html_strike),     NULL },
          { "Subscript",      GSF(html_sub),        NULL },
          { "Superscript",    GSF(html_super),      NULL },
          { "Space",          GSF(html_nbsp),       NULL },
          { "Blink (Evil)",   GSF(html_blink),      NULL } };
     SimpleCallback cheaders [] = {
          { "H1",             GSF(html_header),     GINT_TO_POINTER(H1) },
          { "H2",             GSF(html_header),     GINT_TO_POINTER(H2) },
          { "H3",             GSF(html_header),     GINT_TO_POINTER(H3) },
          { "H4",             GSF(html_header),     GINT_TO_POINTER(H4) },
          { "H5",             GSF(html_header),     GINT_TO_POINTER(H5) },
          { "H6",             GSF(html_header),     GINT_TO_POINTER(H6) } };
     SimpleCallback clists [] = {
          { "New List",       GSF(html_list),       NULL },
          { "List Item",      GSF(html_list_item),  NULL } };
     SimpleCallback ctables[] = {
          { "New Table",      GSF(html_table),      NULL },
          { "Table Row",      GSF(html_table_row),  NULL },
          { "Table Data",     GSF(html_table_data), NULL } };
     SimpleCallback cprotocols[] = {
          { "HTTP",           GSF(html_http),       NULL },
          { "FTP",            GSF(html_ftp),        NULL },
          { "News",           GSF(html_news),       NULL },
          { "Telnet",         GSF(html_telnet),     NULL },
          { "Gopher",         GSF(html_gopher),     NULL },
          { "WWW",            GSF(html_www),        NULL } };
     SimpleCallback cinternals [] = {
          { "BGCOLOR", GSF(html_internal), GINT_TO_POINTER(HTML_BGCOLOR) },
          { "TEXT",    GSF(html_internal), GINT_TO_POINTER(HTML_TEXT),   },
          { "LINK",    GSF(html_internal), GINT_TO_POINTER(HTML_LINK),   } };
     SimpleCallback cmisc [] = {
          { "SSI",           GSF(html_ssi),           NULL },
          { "HR",            GSF(html_hr),            NULL },
          { "Linebreak",     GSF(html_linebreak),     NULL },
          { "Mailto",        GSF(html_mailto),        NULL },
          { "Comment",       GSF(html_comment),       NULL },
          { "Close Unnamed", GSF(html_close_unnamed), NULL } };

     document  = gtk_menu_item_new_with_label("Document");
     body      = gtk_menu_item_new_with_label("Body");
     text      = gtk_menu_item_new_with_label("Text");
     headers   = gtk_menu_item_new_with_label("Headers");
     lists     = gtk_menu_item_new_with_label("Lists");
     tables    = gtk_menu_item_new_with_label("Tables");
     protocols = gtk_menu_item_new_with_label("Protocols");
     internal  = gtk_menu_item_new_with_label("Internal");
     misc      = gtk_menu_item_new_with_label("Miscellaneous");

     mdocument  = menu_generator(&cdocument[0], 6);
     mbody      = menu_generator(&cbody[0], 2);
     mtext      = menu_generator(&ctext[0], 13);
     mheaders   = menu_generator(&cheaders[0], 6);
     mprotocols = menu_generator(&cprotocols[0], 6);
     mlists     = menu_generator(&clists[0], 2);
     mtables    = menu_generator(&ctables[0], 3);
     minternals = menu_generator(&cinternals[0], 3);
     mmisc      = menu_generator(&cmisc[0], 6);
     
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(document), mdocument);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(body), mbody);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(text), mtext);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(headers), mheaders);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(protocols), mprotocols);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(tables), mtables);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(lists), mlists);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(internal), minternals);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(misc), mmisc);

     paragraph = create_menu_item("Paragraph", NULL, NULL, 
                                  GTK_SIGNAL_FUNC(html_paragraph), NULL);
     image     = create_menu_item("Image", NULL, NULL,
                                  GTK_SIGNAL_FUNC(html_image), NULL);
     link      = create_menu_item("Link", NULL, NULL,
                                  GTK_SIGNAL_FUNC(html_link), NULL);
     template  = create_menu_item("Full HTML Template", NULL, NULL,
                                  GTK_SIGNAL_FUNC(html_template), NULL);
     
     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), document);
     gtk_menu_append(GTK_MENU(menu), body);
     gtk_menu_append(GTK_MENU(menu), text);
     gtk_menu_append(GTK_MENU(menu), headers);
     gtk_menu_append(GTK_MENU(menu), lists);
     gtk_menu_append(GTK_MENU(menu), tables);
     gtk_menu_append(GTK_MENU(menu), protocols);
     gtk_menu_append(GTK_MENU(menu), internal);
     gtk_menu_append(GTK_MENU(menu), misc);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), paragraph);
     gtk_menu_append(GTK_MENU(menu), image);
     gtk_menu_append(GTK_MENU(menu), link);
     gtk_menu_append(GTK_MENU(menu), template);

     return(menu);
} /* End create_HTML_menu() */

static GtkWidget *create_shortcuts_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *userdef=NULL, *ext=NULL, *quot=NULL, *group=NULL, *emot=NULL,
          *count = NULL, *delline =NULL, *pseudo=NULL, *em=NULL, 
          *randomstr=NULL;
     GtkWidget *muserdef=NULL, *mext=NULL, *mquot=NULL, *mgroup=NULL, 
          *memot=NULL;

     SimpleCallback cuserdef [] = {
          { "Define",      GSF(define_shortcuts),    NULL },
          { "#1",          GSF(do_shortcut),         GINT_TO_POINTER(0) },
          { "#2",          GSF(do_shortcut),         GINT_TO_POINTER(1) },
          { "#3",          GSF(do_shortcut),         GINT_TO_POINTER(2) },
          { "#4",          GSF(do_shortcut),         GINT_TO_POINTER(3) },
          { "#5",          GSF(do_shortcut),         GINT_TO_POINTER(4) },
          { "#6",          GSF(do_shortcut),         GINT_TO_POINTER(5) } };
     SimpleCallback cext [] = {
          { "RXVT",        GSF(EXT_callprog),  GINT_TO_POINTER(PROG_RXVT)   },
          { "Telnet",      GSF(EXT_callprog),  GINT_TO_POINTER(PROG_TELNET) },
          { "FTP",         GSF(EXT_callprog),  GINT_TO_POINTER(PROG_FTP)    },
          { "Lynx",        GSF(EXT_callprog),  GINT_TO_POINTER(PROG_LYNX)   }};
     SimpleCallback cquot[] = {
          { "Quote",           GSF(SC_quote),        NULL },
          { "Single Quote",    GSF(SC_single_quote), NULL },
          { "Book Quote",      GSF(SC_book_quote),   NULL },
          { "Back Quote",      GSF(SC_backquote),    NULL },
          { "Paragraph",       GSF(SC_paragraph),    NULL },
          { "Surround",        GSF(SC_surround),     NULL } };
     SimpleCallback cgroup[] = {
          { "Parentheses",     GSF(SC_parens),       NULL },
          { "Angle Bracket",   GSF(SC_angle),        NULL },
          { "Double Angle",    GSF(SC_double_angle), NULL },
          { "Hi-Hat",          GSF(SC_hihat),        NULL } };
     SimpleCallback cemot [] = {
          { "Smiley",          GSF(SC_smiley),       NULL },
          { "Winkey",          GSF(SC_winky),       NULL },
          { "Rose",            GSF(SC_rose),         NULL } };

     userdef = gtk_menu_item_new_with_label("User Defined");
     ext     = gtk_menu_item_new_with_label("External Programs");
     quot    = gtk_menu_item_new_with_label("Quoting");
     group   = gtk_menu_item_new_with_label("Group");
     emot    = gtk_menu_item_new_with_label("Emoticons");
     
     muserdef = menu_generator(&cuserdef[0], 7);
     mext     = menu_generator(&cext[0], 4);
     mquot    = menu_generator(&cquot[0], 6);
     mgroup   = menu_generator(&cgroup[0], 4);
     memot    = menu_generator(&cemot[0], 3);

     gtk_menu_item_set_submenu(GTK_MENU_ITEM(userdef), muserdef);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(ext), mext);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(quot), mquot);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(group), mgroup);
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(emot), memot);

     count     = create_menu_item("Character Count", NULL, NULL,
                                  GTK_SIGNAL_FUNC(SC_report_length), NULL);
     delline   = create_menu_item("Delete Line",      NULL, NULL,
                                  GTK_SIGNAL_FUNC(delete_line), NULL);
     pseudo    = create_menu_item("Pseudo-Underline", NULL, NULL,
                                  GTK_SIGNAL_FUNC(SC_pseudo_underline), NULL);
     em        = create_menu_item("Emphasize", NULL, NULL,
                                  GTK_SIGNAL_FUNC(SC_emphasize), NULL);
     randomstr = create_menu_item("Random String", NULL, NULL,
                                  GTK_SIGNAL_FUNC(SC_random_string), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), userdef);
     gtk_menu_append(GTK_MENU(menu), ext);
     gtk_menu_append(GTK_MENU(menu), quot);
     gtk_menu_append(GTK_MENU(menu), group);
     gtk_menu_append(GTK_MENU(menu), emot);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), count);
     gtk_menu_append(GTK_MENU(menu), delline);
     gtk_menu_append(GTK_MENU(menu), pseudo);
     gtk_menu_append(GTK_MENU(menu), em);
     gtk_menu_append(GTK_MENU(menu), randomstr);

     return(menu);
} /* End create_shortcuts_menu() */

static GtkWidget *create_move_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *remote=NULL, *origin=NULL, *center=NULL, *br=NULL, *bl=NULL,
          *ur=NULL, *ul=NULL, *l, *r, *u, *d;

     remote = create_menu_item("Remote Control", "^R", NULL,
                               GTK_SIGNAL_FUNC(gtkeyboard_remote_control),
                               NULL);
     origin = create_menu_item("Origin", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window), 
                               GINT_TO_POINTER(ORIGIN));
     center = create_menu_item("Center", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(WIN_CENTER));
     br     = create_menu_item("Bottom Right", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(BR_CORNER));
     bl     = create_menu_item("Bottom Left", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(BL_CORNER));
     ul     = create_menu_item("Upper Left", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(UL_CORNER));
     ur     = create_menu_item("Upper Right", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(UR_CORNER));
     u      = create_menu_item("Top Centered", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(T_CENTER));
     d      = create_menu_item("Bottom Centered", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(B_CENTER));
     l      = create_menu_item("Left Centered", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(L_CENTER));
     r      = create_menu_item("Right Centered", NULL, NULL,
                               GTK_SIGNAL_FUNC(move_window),
                               GINT_TO_POINTER(R_CENTER));
     
     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), remote);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), origin);
     gtk_menu_append(GTK_MENU(menu), center);
     gtk_menu_append(GTK_MENU(menu), br);
     gtk_menu_append(GTK_MENU(menu), bl);
     gtk_menu_append(GTK_MENU(menu), ur);
     gtk_menu_append(GTK_MENU(menu), ul);
     gtk_menu_append(GTK_MENU(menu), u);
     gtk_menu_append(GTK_MENU(menu), d);
     gtk_menu_append(GTK_MENU(menu), r);     
     gtk_menu_append(GTK_MENU(menu), l);

     return(menu);
} /* End create_move_menu() */

static GtkWidget *create_color_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *choose=NULL, *reset=NULL;

     reset = create_menu_item("Reset Style", NULL, NULL,
                              GTK_SIGNAL_FUNC(reset_application_style), NULL);
     choose = create_menu_item("Set Colors", NULL, NULL,
                               GTK_SIGNAL_FUNC(color_button_clicked), NULL);
     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), choose);
     gtk_menu_append(GTK_MENU(menu), reset);
     return(menu);
} /* End create_color_menu() */

static GtkWidget *create_font_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *text = NULL, *keyb= NULL;
     
     text = create_menu_item("Choose Text Font", NULL, NULL, 
                             GTK_SIGNAL_FUNC(FontSelect), NULL);
     keyb = create_menu_item("Choose Keyboard Font", NULL, NULL,
                             GTK_SIGNAL_FUNC(KeyboardFontSelect), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), text);
     gtk_menu_append(GTK_MENU(menu), keyb);
     return(menu);
} /* End create_font_menu() */

static GtkWidget *create_looks_menu(void)
{
 GtkWidget *menu = gtk_menu_new();
     GtkWidget *colors=NULL, *fonts=NULL, *text=NULL, *smenu=NULL, 
          *status=NULL, *toolbar=NULL, *buttonrow=NULL,
          *allbutkb=NULL, *showall=NULL, *savelayout=NULL, *forget=NULL,
          *raise=NULL, *lower=NULL, *numberpad=NULL, *fkeys=NULL, 
          *cursors=NULL, *mainkbd=NULL;
     
     colors    = gtk_menu_item_new_with_label("Colors");
     fonts     = gtk_menu_item_new_with_label("Fonts");
     
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(colors), create_color_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(fonts), create_font_menu());

     mainkbd   = 
          create_hide_show_toggle_button("Main Keyboard",
                                         GTK_SIGNAL_FUNC(hide_widget),
                                         GINT_TO_POINTER(HIDE_KBD_KBD),
                                         GTK_SIGNAL_FUNC(unhide_widget),
                                         GINT_TO_POINTER(SHOW_KBD_KBD),
                                         GUI.keyboard_elements.show_keyboard);
     bind_variable_to_boolean(mainkbd, &GUI.keyboard_elements.show_keyboard,
                              g_strdup("Main keyboard"));
     cursors = 
          create_hide_show_toggle_button("Cursor Keys",
                                         GTK_SIGNAL_FUNC(hide_widget),
                                         GINT_TO_POINTER(HIDE_KBD_CURSOR_PAD),
                                         GTK_SIGNAL_FUNC(unhide_widget),
                                         GINT_TO_POINTER(SHOW_KBD_CURSOR_PAD),
                                         GUI.keyboard_elements.show_cursor_keys);

     bind_variable_to_boolean(cursors, 
                              &GUI.keyboard_elements.show_cursor_keys,
                              g_strdup("cursor keys"));
     fkeys =
          create_hide_show_toggle_button("F-Keys",
                                         GTK_SIGNAL_FUNC(hide_widget),
                                         GINT_TO_POINTER(HIDE_KBD_FKEYS),
                                         GTK_SIGNAL_FUNC(unhide_widget),
                                         GINT_TO_POINTER(SHOW_KBD_FKEYS),
                                         GUI.keyboard_elements.show_f_keys);
     bind_variable_to_boolean(fkeys, &GUI.keyboard_elements.show_f_keys,
                              g_strdup("f keys"));
     numberpad = 
          create_hide_show_toggle_button("Number Pad",
                                         GTK_SIGNAL_FUNC(hide_widget),
                                         GINT_TO_POINTER(HIDE_KBD_NUMBERPAD),
                                         GTK_SIGNAL_FUNC(unhide_widget),
                                         GINT_TO_POINTER(SHOW_KBD_NUMBERPAD),
                                         GUI.keyboard_elements.show_number_pad
               );
     bind_variable_to_boolean(numberpad, 
                              &GUI.keyboard_elements.show_number_pad,
                              g_strdup("number pad"));
     smenu     = create_hide_show_toggle_button("Menus",
                                                GTK_SIGNAL_FUNC(hide_widget),
                                                GINT_TO_POINTER(HIDE_MENUS),
                                                GTK_SIGNAL_FUNC(unhide_widget),
                                                GINT_TO_POINTER(SHOW_MENUS),
                                                ELEMENTS.menubar);
     bind_variable_to_boolean(smenu, &ELEMENTS.menubar,
                              g_strdup("menubar"));
     text      = create_hide_show_toggle_button("Text Box",
                                                GTK_SIGNAL_FUNC(hide_widget),
                                                GINT_TO_POINTER(HIDE_OUTPUT),
                                                GTK_SIGNAL_FUNC(unhide_widget),
                                                GINT_TO_POINTER(SHOW_OUTPUT),
                                                ELEMENTS.text);
     bind_variable_to_boolean(text, &ELEMENTS.text, g_strdup("Text"));
     status    = create_hide_show_toggle_button("Status",
                                                GTK_SIGNAL_FUNC(hide_widget),
                                                GINT_TO_POINTER(HIDE_STATUS),
                                                GTK_SIGNAL_FUNC(unhide_widget),
                                                GINT_TO_POINTER(SHOW_STATUS),
                                                ELEMENTS.status);
     bind_variable_to_boolean(status, &ELEMENTS.status, 
                              g_strdup("status"));
     toolbar   = create_hide_show_toggle_button("Toolbar",
                                                GTK_SIGNAL_FUNC(hide_widget),
                                                GINT_TO_POINTER(HIDE_TOOLBAR),
                                                GTK_SIGNAL_FUNC(unhide_widget),
                                                GINT_TO_POINTER(SHOW_TOOLBAR),
                                                ELEMENTS.toolbar);
     bind_variable_to_boolean(toolbar, &ELEMENTS.toolbar,
                              g_strdup("toolbar"));
     buttonrow = create_hide_show_toggle_button("Bottom Row Buttons",
                                                GTK_SIGNAL_FUNC(hide_widget),
                                                GINT_TO_POINTER(
                                                     HIDE_BOTTOM_ROW_BUTTONS),
                                                GTK_SIGNAL_FUNC(unhide_widget),
                                                GINT_TO_POINTER(
                                                     SHOW_BOTTOM_ROW_BUTTONS),
                                                ELEMENTS.buttonbar);
     bind_variable_to_boolean(buttonrow, &ELEMENTS.buttonbar,
                              g_strdup("buttonrow"));
     allbutkb   = create_menu_item("Hide all but Keyboard", NULL, NULL,
                                   GTK_SIGNAL_FUNC(HideAllButKeyboard), NULL);
     showall    = create_menu_item("Show All", NULL, NULL,
                                   GTK_SIGNAL_FUNC(ShowAll), NULL);
     savelayout = create_menu_item("Save Layout Settings", NULL, NULL, 
                                   GTK_SIGNAL_FUNC(save_layout_settings), 
                                   NULL);
     forget     = create_menu_item("Forget Layout Settings", NULL, NULL,
                                   GTK_SIGNAL_FUNC(forget_layout_settings),
                                   NULL);
     raise      = create_menu_item("Raise Window", NULL, NULL,
                                   GTK_SIGNAL_FUNC(gtkeyboard_raise), NULL);
     lower      = create_menu_item("Lower Window", NULL, NULL,
                                   GTK_SIGNAL_FUNC(gtkeyboard_lower), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), colors);
     gtk_menu_append(GTK_MENU(menu), fonts);
     gtk_menu_append(GTK_MENU(menu), named_menu_separator_new("Keyboard"));
     gtk_menu_append(GTK_MENU(menu), mainkbd);
     gtk_menu_append(GTK_MENU(menu), numberpad);
     gtk_menu_append(GTK_MENU(menu), cursors);
     gtk_menu_append(GTK_MENU(menu), fkeys);
     gtk_menu_append(GTK_MENU(menu), named_menu_separator_new("Other Layout"));
     gtk_menu_append(GTK_MENU(menu), text);
     gtk_menu_append(GTK_MENU(menu), smenu);
     gtk_menu_append(GTK_MENU(menu), status);
     gtk_menu_append(GTK_MENU(menu), toolbar);
     gtk_menu_append(GTK_MENU(menu), buttonrow);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), allbutkb);
     gtk_menu_append(GTK_MENU(menu), showall);
     gtk_menu_append(GTK_MENU(menu), savelayout);
     gtk_menu_append(GTK_MENU(menu), forget);
     gtk_menu_append(GTK_MENU(menu), raise);
     gtk_menu_append(GTK_MENU(menu), lower);
     return(menu);
} /* End create_looks_menu() */

static GtkWidget *create_output_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *doutput=NULL, *cstatus=NULL, *ctext=NULL, *console=NULL, 
          *printer=NULL, *tmp=NULL;

     doutput  = create_checked_item("Double Output",
                                    GTK_SIGNAL_FUNC(toggle_double_data), NULL,
                                    options.SEND_TO_BOTH_WINDOWS);
     cstatus = create_menu_item("Clear Status", NULL, NULL,
                                GTK_SIGNAL_FUNC(clear_wrapper),
                                GINT_TO_POINTER(CLEAR_STATUS));
     ctext   = create_menu_item("Clear Text", NULL, NULL,
                                GTK_SIGNAL_FUNC(clear_wrapper),
                                GINT_TO_POINTER(CLEAR_TEXT));
     console = create_menu_item("System Console", NULL, NULL,
                                GTK_SIGNAL_FUNC(change_output),
                                GINT_TO_POINTER(OUTPUT_CONSOLE));
     printer = create_menu_item("Printer", NULL, NULL,
                                GTK_SIGNAL_FUNC(change_output),
                                GINT_TO_POINTER(OUTPUT_PRINTER));
     tmp     = create_menu_item("Temp File", NULL, NULL,
                                GTK_SIGNAL_FUNC(change_output),
                                GINT_TO_POINTER(OUTPUT_TEMPFILE));

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), doutput);
     gtk_menu_append(GTK_MENU(menu), cstatus);
     gtk_menu_append(GTK_MENU(menu), ctext);
     gtk_menu_append(GTK_MENU(menu), console);
     gtk_menu_append(GTK_MENU(menu), printer);
     gtk_menu_append(GTK_MENU(menu), tmp);
     return(menu);
} /* End create_output_menu() */

static GtkWidget *create_redirection_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *ex=NULL, *im=NULL;

     ex = create_menu_item("Explicit", NULL, NULL,
                           GTK_SIGNAL_FUNC(redirect_explicit), NULL);
     im = create_menu_item("Implicit", NULL, NULL,
                           GTK_SIGNAL_FUNC(redirect_implicit), NULL);
     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), ex);
     gtk_menu_append(GTK_MENU(menu), im);
     return(menu);
} /* End create_redirection_menu() */

static GtkWidget *create_options_menu(void)
{
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *move=NULL, *looks=NULL, *output=NULL, *redirection=NULL,
          *layout=NULL, *redirect=NULL, *ignore=NULL, *define=NULL,
          *toggleww=NULL, *togglesoe=NULL, *togglesl=NULL, *reset=NULL, 
          *disable=NULL, *enable=NULL, *remote=NULL;

     move        = gtk_menu_item_new_with_label("Move To...");
     looks       = gtk_menu_item_new_with_label("Looks");
     output      = gtk_menu_item_new_with_label("Output");
     redirection = gtk_menu_item_new_with_label("Redirection");

     gtk_menu_item_set_submenu(GTK_MENU_ITEM(move), create_move_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(looks), create_looks_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(output), create_output_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(redirection), 
                               create_redirection_menu());

     layout    = create_menu_item("Choose a Layout File", NULL, NULL,
                                  GTK_SIGNAL_FUNC(custom_keyboard_file_dialog),
                                  NULL);
     redirect  = create_menu_item("Redirect to a Window", NULL, NULL,
                                  GTK_SIGNAL_FUNC(grab_a_window), NULL);
     ignore    = create_menu_item("Ignore a Window", NULL, NULL,
                                  GTK_SIGNAL_FUNC(ignore_window), NULL);
     define    = create_menu_item("Define Shortcuts", NULL, NULL,
                                  GTK_SIGNAL_FUNC(define_shortcuts), NULL);
     toggleww  = create_checked_item("Word Wrap", 
                                     GTK_SIGNAL_FUNC(toggle_wordwrap), NULL,
                                     options.WORD_WRAP);
     togglesoe = create_checked_item("Save on Exit",
                                     GTK_SIGNAL_FUNC(toggle_ask_save_on_exit),
                                     NULL, options.ASK_SAVE_ON_EXIT);
     togglesl  = create_checked_item("Status Logging",
                                     GTK_SIGNAL_FUNC(status_logging_toggle),
                                     NULL, options.STATUS_LOGGING);
     
     reset   = create_menu_item("Reset Looks", NULL, NULL,
                                GTK_SIGNAL_FUNC(reset_application_style),
                                NULL);
     disable = create_menu_item("Disable ToolTips", NULL, NULL,
                                GTK_SIGNAL_FUNC(disable_tooltips), NULL);
     enable  = create_menu_item("Enable ToolTips", NULL, NULL,
                                GTK_SIGNAL_FUNC(enable_tooltips), NULL);
     remote  = create_menu_item("Remote Control", "^R", NULL,
                                GTK_SIGNAL_FUNC(gtkeyboard_remote_control),
                                NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());

     gtk_menu_append(GTK_MENU(menu), move);
     gtk_menu_append(GTK_MENU(menu), looks);
     gtk_menu_append(GTK_MENU(menu), output);
     gtk_menu_append(GTK_MENU(menu), redirection);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), toggleww);
     gtk_menu_append(GTK_MENU(menu), togglesoe);
     gtk_menu_append(GTK_MENU(menu), togglesl);
     gtk_menu_append(GTK_MENU(menu), menu_hseparator_new());
     gtk_menu_append(GTK_MENU(menu), layout);
     gtk_menu_append(GTK_MENU(menu), redirect);
     gtk_menu_append(GTK_MENU(menu), ignore);
     gtk_menu_append(GTK_MENU(menu), define);
     gtk_menu_append(GTK_MENU(menu), reset);
     gtk_menu_append(GTK_MENU(menu), disable);
     gtk_menu_append(GTK_MENU(menu), enable);
     gtk_menu_append(GTK_MENU(menu), remote);     
     return(menu);
} /* End create_options_menu() */

static GtkWidget *create_misc_menu(void)
{     
     GtkWidget *menu = gtk_menu_new();
     GtkWidget *completer = NULL, *timestamp = NULL, *status = NULL,
          *bol = NULL, *eol = NULL, *bof = NULL, *eof = NULL, *coord = NULL,
          *remap = NULL;
     
     completer = create_menu_item("New Completer Window", "^W", NULL,
                                  GTK_SIGNAL_FUNC(completer_window_new), NULL);
     timestamp = create_menu_item("Timestamp", NULL, NULL,
                                  GTK_SIGNAL_FUNC(SC_timestamp), NULL);
     status    = create_menu_item("Status Report", NULL, NULL,
                                  GTK_SIGNAL_FUNC(status_report), NULL);
     bol       = create_menu_item("Beginning of Line", NULL, NULL,
                                  GTK_SIGNAL_FUNC(goto_bol), NULL);
     eol       = create_menu_item("End of Line", NULL, NULL,
                                  GTK_SIGNAL_FUNC(goto_eol), NULL);
     bof       = create_menu_item("Beginning of File", NULL, NULL,
                                  GTK_SIGNAL_FUNC(goto_bof), NULL);
     eof       = create_menu_item("End of File", NULL, NULL,
                                  GTK_SIGNAL_FUNC(goto_eof), NULL);
     coord     = create_menu_item("Show Coordinates", NULL, NULL,
                                  GTK_SIGNAL_FUNC(gtkeyboard_print_coords), 
                                  NULL);
     remap     = create_menu_item("Remap Keyboard", "^M", NULL,
                                  GTK_SIGNAL_FUNC(remap_keyboard_really),
                                  NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), completer);
     gtk_menu_append(GTK_MENU(menu), timestamp);
     gtk_menu_append(GTK_MENU(menu), status);
     gtk_menu_append(GTK_MENU(menu), bol);
     gtk_menu_append(GTK_MENU(menu), eol);
     gtk_menu_append(GTK_MENU(menu), bof);
     gtk_menu_append(GTK_MENU(menu), eof);
     gtk_menu_append(GTK_MENU(menu), coord);
     gtk_menu_append(GTK_MENU(menu), remap);
     return(menu);
} /* End create_misc_menu() */

static GtkWidget *create_help_menu(void)
{
     GtkWidget *menu = gtk_menu_new();

     GtkWidget *about = NULL, *license=NULL, *new=NULL, *splash_item=NULL;

     about   = create_menu_item("About", NULL, NULL,
                                GTK_SIGNAL_FUNC(program_info),
                                (gpointer)INFO_ABOUT);
     license = create_menu_item("License", NULL, NULL,
                                GTK_SIGNAL_FUNC(program_info),
                                (gpointer)INFO_LICENSE);
     new     = create_menu_item("What's New", NULL, NULL,
                                GTK_SIGNAL_FUNC(program_info),
                                (gpointer)INFO_CHANGELOG);
     splash_item  = create_menu_item("Splash", NULL, NULL,
                                     GTK_SIGNAL_FUNC(splash), NULL);

     gtk_menu_append(GTK_MENU(menu), gtk_tearoff_menu_item_new());
     gtk_menu_append(GTK_MENU(menu), about);
     gtk_menu_append(GTK_MENU(menu), license);
     gtk_menu_append(GTK_MENU(menu), new);
     gtk_menu_append(GTK_MENU(menu), splash_item);

     gtk_widget_show(menu);
     return(menu);
} /* End create_help_menu() */

/* The only real difference between this function and the next one, 
 * create_menus() is that this one returns something that is a GtkMenu
 * and create_menus() returns a GtkMenubar for display as regular pulldown
 * menus
 */
GtkWidget *create_context_popup_menu(GtkWidget *emitter, gpointer data)
{
     GtkWidget *popup_menu = gtk_menu_new();
     GtkWidget *file       = gtk_menu_item_new_with_label("File");
     GtkWidget *edit       = gtk_menu_item_new_with_label("Edit");
     GtkWidget *HTML       = gtk_menu_item_new_with_label("HTML");
     GtkWidget *shortcuts  = gtk_menu_item_new_with_label("Shortcuts");
     GtkWidget *options    = gtk_menu_item_new_with_label("Options");
     GtkWidget *misc       = gtk_menu_item_new_with_label("Misc");
     GtkWidget *help       = gtk_menu_item_new_with_label("Help");
     
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(file), create_file_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(edit), create_edit_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(HTML), create_HTML_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(shortcuts), 
                               create_shortcuts_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(options), create_options_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(misc), create_misc_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(help), create_help_menu());
                             
     gtk_menu_append(GTK_MENU(popup_menu), file);
     gtk_menu_append(GTK_MENU(popup_menu), edit);
     gtk_menu_append(GTK_MENU(popup_menu), HTML);
     gtk_menu_append(GTK_MENU(popup_menu), shortcuts);
     gtk_menu_append(GTK_MENU(popup_menu), options);
     gtk_menu_append(GTK_MENU(popup_menu), misc);
     gtk_menu_append(GTK_MENU(popup_menu), help);
     
     return(popup_menu);
} /* End create_context_popup_menu() */

GtkWidget *create_menus(GtkWidget *emitter, gpointer data)
{
     GtkWidget *menubar   = gtk_menu_bar_new();
     GtkWidget *file      = gtk_menu_item_new_with_label("File");
     GtkWidget *edit      = gtk_menu_item_new_with_label("Edit");
     GtkWidget *HTML      = gtk_menu_item_new_with_label("HTML");
     GtkWidget *shortcuts = gtk_menu_item_new_with_label("Shortcuts");
     GtkWidget *options   = gtk_menu_item_new_with_label("Options");
     GtkWidget *misc      = gtk_menu_item_new_with_label("Misc");
     GtkWidget *help      = gtk_menu_item_new_with_label("Help");
     
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(file), create_file_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(edit), create_edit_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(HTML), create_HTML_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(shortcuts), 
                               create_shortcuts_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(options), create_options_menu());
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(misc), create_misc_menu());
     gtk_menu_item_right_justify(GTK_MENU_ITEM(help));
     gtk_menu_item_set_submenu(GTK_MENU_ITEM(help), create_help_menu());
                             
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), file);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), edit);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), HTML);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), shortcuts);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), options);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), misc);
     gtk_menu_bar_append(GTK_MENU_BAR(menubar), help);
     
     return(menubar);
} /* End create_menu() */






