/* -*- C++ -*-
   XGContextEvent - Window/Event code for X11 backends.

   Copyright (C) 1998,1999 Free Software Foundation, Inc.

   Written by:  Adam Fedor <fedor@boulder.colorado.edu>
   Date: Nov 1998
   
   This file is part of the GNU Objective C User Interface Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   */
#include <AppKit/AppKitExceptions.h>
#include <AppKit/NSAffineTransform.h>
#include <AppKit/NSView.h>
#include <AppKit/NSGraphics.h>
#include <AppKit/NSWindow.h>
#include <Foundation/NSException.h>
#include <Foundation/NSArray.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSData.h>
#include <Foundation/NSValue.h>
#include <Foundation/NSString.h>
#include <Foundation/NSUserDefaults.h>
#include <Foundation/NSDebug.h>

#ifdef XDPS_BACKEND_LIBRARY
#define XGContext       NSDPSContext
#define XGDrawingEngine PXKDrawingEngine
#define XGRawWindow	PXKRawWindow
#define XGDragView	PXKDragView
#define XGBitmapImageRep PXKBitmapImageRep
#include "gnustep/xdps/NSDPSContextWindow.h"
#include "PXKDrawingEngine.h"
#else
#include "gnustep/xgps/XGContextWindow.h"
#include "gnustep/xgps/XGContextPrivate.h"
#include "XGDrawingEngine.h"
#endif
#include "SharedX/XGGeneric.h"
#include "SharedX/xrtools.h"
#include "SharedX/xdnd.h"

#include "math.h"
#include <X11/keysym.h>
#include <X11/Xproto.h>
#include <X11/extensions/shape.h>

#if LIB_FOUNDATION_LIBRARY
# include <Foundation/NSPosixFileDescriptor.h>
#elif defined(NeXT_PDO)
# include <Foundation/NSFileHandle.h>
# include <Foundation/NSNotification.h>
#endif

@class	NSIconWindow;

extern Atom     WM_STATE;

static BOOL     xDndInitialized = NO;
static DndClass dnd;

// NumLock's mask (it depends on the keyboard mapping)
static unsigned int _num_lock_mask;
// Modifier state
static BOOL _control_pressed = NO;
static BOOL _command_pressed = NO;
static BOOL _alt_pressed = NO;
// Keys used for the modifiers (you may set them with user preferences)
static KeyCode _control_keycodes[2];
static KeyCode _command_keycodes[2];
static KeyCode _alt_keycodes[2];

static BOOL _is_keyboard_initialized = NO;

@interface XGContext (Private)
- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode;
- (void) setupRunLoopInputSourcesForMode: (NSString*)mode; 
- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode;
- (int) XGErrorHandler: (Display*)display : (XErrorEvent*)err;
@end

DndClass xdnd(void)
{
  return dnd;			// FIX ME rename with private desig
}

Atom 
GSActionForDragOperation(NSDragOperation op)
{
  Atom xaction;
  switch (op)
    {
      case NSDragOperationNone: xaction = None; break;
      case NSDragOperationCopy: xaction = dnd.XdndActionCopy; break;
      case NSDragOperationLink: xaction = dnd.XdndActionLink; break;
      case NSDragOperationGeneric: xaction = dnd.XdndActionCopy; break;
      case NSDragOperationPrivate: xaction = dnd.XdndActionPrivate; break;
      case NSDragOperationAll: xaction = dnd.XdndActionPrivate; break;
    }
  return xaction;
}

NSDragOperation
GSDragOperationForAction(Atom xaction)
{
  NSDragOperation action;
  if (xaction == dnd.XdndActionCopy)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionMove)
    action = NSDragOperationCopy;
  else if (xaction == dnd.XdndActionLink) 
    action = NSDragOperationLink;
  else if (xaction == dnd.XdndActionAsk) 
    action = NSDragOperationGeneric;
  else if (xaction == dnd.XdndActionPrivate) 
    action = NSDragOperationPrivate;
  else
    action = NSDragOperationNone;
  return action;
}

int
XGErrorHandler(Display *display, XErrorEvent *err)
{
  XGContext	*ctxt = (XGContext*)GSCurrentContext();

  return [ctxt XGErrorHandler: display : err];
}

static NSEvent*
process_key_event(XEvent* xEvent, XGContext* ctxt, NSEventType eventType);

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, unsigned eventModifierFlags);

static unsigned
process_modifier_flags(unsigned int state);

static void
initialize_keyboard (void);

static void
set_up_num_lock (void);

static inline int 
check_modifier (XEvent *xEvent, KeyCode key_code) 
{
  return (xEvent->xkeymap.key_vector[key_code / 8] & (1 << (key_code % 8)));  
}

@interface NSCursor (BackendPrivate)
- (void *)_cid;
@end

#define XDPY (((RContext *)context)->dpy)
#define XSCR (((RContext *)context)->screen_number)

@implementation XGContext (X11Methods)

- (int) XGErrorHandler: (Display*)display : (XErrorEvent*)err
{
  int length = 1024;
  char buffer[length+1];

  /*
   * Ignore attempts to set input focus to unmapped window, except for noting
   * if the most recent request failed (mark the request serial number to 0)
   * in which case we should repeat the request when the window becomes
   * mapped again.
   */
  if (err->error_code == BadMatch && err->request_code == X_SetInputFocus)
    {
      if (err->serial == generic.focusRequestNumber)
	{
	  generic.focusRequestNumber = 0;
	}
      return 0;
    }

  XGetErrorText(display, err->error_code, buffer, length),
  [NSException raise: NSWindowServerCommunicationException
    format: @"X-Windows error - %s\n\
          on display: %s\n\
    		type: %d\n\
       serial number: %d\n\
	request code: %d\n",
	buffer,
    	XDisplayName(DisplayString(display)),
	err->type, err->serial, err->request_code];
  return 0;
}

- (void) setupRunLoopInputSourcesForMode: (NSString*)mode
{
  int		xEventQueueFd = XConnectionNumber(XDPY);
  NSRunLoop	*currentRunLoop = [NSRunLoop currentRunLoop];

#if defined(LIB_FOUNDATION_LIBRARY)
  {
    id fileDescriptor = [[[NSPosixFileDescriptor alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    // Invoke limitDateForMode: to setup the current
    // mode of the run loop (the doc says that this
    // method and acceptInputForMode: beforeDate: are
    // the only ones that setup the current mode).

    [currentRunLoop limitDateForMode: mode];

    [fileDescriptor setDelegate: self];
    [fileDescriptor monitorFileActivity: NSPosixReadableActivity];
  }
#elif defined(NeXT_PDO)
  {
    id fileDescriptor = [[[NSFileHandle alloc]
	initWithFileDescriptor: xEventQueueFd]
	autorelease];

    [[NSNotificationCenter defaultCenter] addObserver: self
	selector: @selector(activityOnFileHandle: )
	name: NSFileHandleDataAvailableNotification
	object: fileDescriptor];
    [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: mode]];
  }
#else
  [currentRunLoop addEvent: (void*)(gsaddr)xEventQueueFd
		      type: ET_RDESC
		   watcher: (id<RunLoopEvents>)self
		   forMode: mode];
#endif
}

#if LIB_FOUNDATION_LIBRARY
- (void) activity: (NSPosixFileActivities)activity
		posixFileDescriptor: (NSPosixFileDescriptor*)fileDescriptor
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#elif defined(NeXT_PDO)
- (void) activityOnFileHandle: (NSNotification*)notification
{
  id fileDescriptor = [notification object];
  id runLoopMode = [[NSRunLoop currentRunLoop] currentMode];

  [fileDescriptor waitForDataInBackgroundAndNotifyForModes:
	[NSArray arrayWithObject: runLoopMode]];
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
}
#endif

- (void) receivedEvent: (void*)data
                  type: (RunLoopEventType)type
                 extra: (void*)extra
               forMode: (NSString*)mode
{
  static int		clickCount = 1;
  static unsigned int	eventFlags;
  static unsigned int	last_XEvent_state = -1;
  static gswindow_device_t *window = NULL;
  NSEvent		*e = nil;
  XEvent		xEvent;
  NSPoint		eventLocation;
  NSWindow              *nswin;
  Window		xWin;
  NSEventType		eventType;
  int			count;

  while ((count = XPending(XDPY)) > 0)
    {
      // loop and grab all of the events from the X queue
      while (count-- > 0)
	{
	  //NSDebugLLog(@"NSEvent", @"Get next XWindows event\n");
	  XNextEvent(XDPY, &xEvent);
	  switch (xEvent.type)
	    {
	      // mouse button events
	      case ButtonPress:
		NSDebugLLog(@"NSEvent", @"%d ButtonPress: \
		  xEvent.xbutton.time %u timeOfLastClick %u \n",
		  xEvent.xbutton.window, xEvent.xbutton.time,
		  generic.lastClick);
	      // hardwired test for a double click
	      // default of 300 should be user set;
	      // under NS the windowserver does this
	      if (xEvent.xbutton.time
		< (unsigned long)(generic.lastClick +300))
		clickCount++;
	      else
		clickCount = 1;		// reset the click count
	      generic.lastClick = xEvent.xbutton.time;
	      generic.lastTime = generic.lastClick;

	      // Count any button other than 1 as right
	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseDown : NSRightMouseDown);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (window == 0 || xWin != window->ident)
		window = [XGContext _windowForXWindow: xWin];
	      if (window == 0)
		break;
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = NSHeight(window->xframe)-xEvent.xbutton.y;

	      // create NSEvent
	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)generic.lastClick
				 windowNumber: window->number
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case ButtonRelease:
	      NSDebugLLog(@"NSEvent", @"%d ButtonRelease\n",
		xEvent.xbutton.window);
	      generic.lastTime = xEvent.xbutton.time;
	      eventType = (xEvent.xbutton.button == Button1 ?
			    NSLeftMouseUp : NSRightMouseUp);
	      if (xEvent.xbutton.state != last_XEvent_state)
		{
		  eventFlags = process_modifier_flags(xEvent.xbutton.state);
		  last_XEvent_state = xEvent.xbutton.state;
		}
	      // if pointer is grabbed use grab window
	      xWin = (grabWindow == 0) ? xEvent.xbutton.window : grabWindow;
	      if (window == 0 || xWin != window->ident)
		window = [XGContext _windowForXWindow: xWin];
	      if (window == 0)
		break;
	      eventLocation.x = xEvent.xbutton.x;
	      eventLocation.y = NSHeight(window->xframe)-xEvent.xbutton.y;

	      e = [NSEvent mouseEventWithType: eventType
				     location: eventLocation
				modifierFlags: eventFlags
				    timestamp: (NSTimeInterval)generic.lastTime
				 windowNumber: window->number
				      context: self
				  eventNumber: xEvent.xbutton.serial
				   clickCount: clickCount
				     pressure: 1.0];
	      break;

	    case CirculateNotify:
	      NSDebugLLog(@"NSEvent", @"%d CirculateNotify\n",
		xEvent.xcirculate.window);
	      break;

	    case CirculateRequest:
	      NSDebugLLog(@"NSEvent", @"%d CirculateRequest\n",
		xEvent.xcirculaterequest.window);
	      break;

	    case ClientMessage:
	      {
		NSTimeInterval time;

		NSDebugLLog(@"NSEvent", @"%d ClientMessage\n",
		  xEvent.xclient.window);
		window = [XGContext _windowForXWindow: xEvent.xclient.window];
		if (window == 0)
		  break;
		if (xEvent.xclient.message_type == generic.protocols_atom)
		  {
		    generic.lastTime = (Time)xEvent.xclient.data.l[1];
		    NSDebugLLog(@"NSEvent", @"WM Protocol - %s\n",
		      XGetAtomName(XDPY, xEvent.xclient.data.l[0]));

		    if (xEvent.xclient.data.l[0] == generic.delete_win_atom)
		      {
			/*
			 * WM is asking us to close a window
			 */
			eventLocation = NSMakePoint(0,0);
			e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitWindowClose
			        data1: 0
			        data2: 0];
		      }
		    else if (xEvent.xclient.data.l[0]
		      == generic.miniaturize_atom)
		      {
			nswin = GSWindowWithNumber(window->number);
		
			[nswin performMiniaturize: nswin];
		      }
		    else if (xEvent.xclient.data.l[0]
		      == generic.take_focus_atom)
		      {
			/*
			 * WM is asking us to take the keyboard focus
			 */
			if (window == 0)
			  {
			    NSLog(@"message for window we don't own");
			    break;
			  }
			NSDebugLLog(@"Focus", @"take focus: %d",
			  window->number);
			nswin = [NSApp keyWindow];
			if (nswin == nil
			  || [nswin windowNumber] != window->number)
			  {
			    Window	rootWin;
			    Window	childWin = 0;
			    int		currentX;
			    int		currentY;
			    int		winX;
			    int		winY;
			    unsigned	mask;

			    XQueryPointer (XDPY, [self xDisplayRootWindow],
			      &rootWin, &childWin, &currentX, &currentY,
			      &winX, &winY, &mask);
			    generic.desiredFocusWindow = 0;
			    generic.focusRequestNumber = 0;
			    /*
			     * If the mouse pointer is actually in the
			     * parent of our window, then we assume that
			     * the window manager has sent us the
			     * take-focus because the user clicked in
			     * the decorative border, so we make the
			     * new window the key window if possible.
			     * FIXME - really, we should simulate a
			     * mouse click or something so that the
			     * window can make itsself key.
			     */
			    if (childWin == window->parent)
			      {
				nswin = GSWindowWithNumber(window->number);
				if ([nswin canBecomeKeyWindow] == YES)
				  {
				    [nswin makeKeyAndOrderFront: self];
				    [NSApp activateIgnoringOtherApps: YES];
				  }
			      }
			  }
			if (nswin != nil)
			  {
			    /*
			     * We reassert our desire to have input
			     * focus in our existing key window.
			     */
			    [self DPSsetinputstate: window->number 
			                          : GSTitleBarKey];
			    [self DPSsetinputfocus: [nswin windowNumber]];
			  }
		      }
		  }
		else if (xEvent.xclient.message_type == dnd.XdndEnter)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndEnter message\n");
		    source = XDND_ENTER_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0,0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingEnter
			        data1: source
			        data2: 0];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([XGContext _windowForXWindow: source] == NULL)
		      [[XGDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndPosition)
		  {
		    Window source;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndPosition message\n");
		    source = XDND_POSITION_SOURCE_WIN(&xEvent);
		    eventLocation.x = XDND_POSITION_ROOT_X(&xEvent)
		      - NSMinX(window->xframe);
		    eventLocation.y = XDND_POSITION_ROOT_Y(&xEvent)
		      - NSMinY(window->xframe);
		    eventLocation.y = NSHeight(window->xframe) 
		      - eventLocation.y;
		    time = XDND_POSITION_TIME(&xEvent);
		    action = XDND_POSITION_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingUpdate
			        data1: source
			        data2: GSDragOperationForAction(action)];
		    /* If this is a non-local drag, set the dragInfo */
		    if ([XGContext _windowForXWindow: source] == NULL)
		      [[XGDragView sharedDragView] setDragInfoFromEvent: e];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndStatus)
		  {
		    Window target;
		    Atom action;
		    NSDebugLLog(@"NSDragging", @"  XdndStatus message\n");
		    target = XDND_STATUS_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    action = XDND_STATUS_ACTION(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingStatus
			        data1: target
			        data2: GSDragOperationForAction(action)];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndLeave)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndLeave message\n");
		    source = XDND_LEAVE_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingExit
			        data1: 0
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndDrop)
		  {
		    Window source;
		    NSDebugLLog(@"NSDragging", @"  XdndDrop message\n");
		    source = XDND_DROP_SOURCE_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    time = XDND_DROP_TIME(&xEvent);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: time
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingDrop
			        data1: source
			        data2: 0];
		  }
		else if (xEvent.xclient.message_type == dnd.XdndFinished)
		  {
		    Window target;
		    NSDebugLLog(@"NSDragging", @"  XdndFinished message\n");
		    target = XDND_FINISHED_TARGET_WIN(&xEvent);
		    eventLocation = NSMakePoint(0, 0);
		    e = [NSEvent otherEventWithType: NSAppKitDefined
			        location: eventLocation
			        modifierFlags: 0
			        timestamp: 0
			        windowNumber: window->number
			        context: self
			        subtype: GSAppKitDraggingFinished
			        data1: target
			        data2: 0];
		  }
	
	      }
	      break;

	      case ColormapNotify:
		// colormap attribute
		NSDebugLLog(@"NSEvent", @"%d ColormapNotify\n",
		  xEvent.xcolormap.window);
		break;

	      // the window has been resized, change the width and height
	      // and update the window so the changes get displayed
	      case ConfigureNotify:
		NSDebugLLog(@"NSEvent", @"%d ConfigureNotify\n"
		  @"x:%d y:%d w:%d h:%d b:%d %c", xEvent.xconfigure.window,
		  xEvent.xconfigure.x, xEvent.xconfigure.y,
		  xEvent.xconfigure.width, xEvent.xconfigure.height,
		  xEvent.xconfigure.border_width,
		  xEvent.xconfigure.send_event ? 'T' : 'F');
		window = [XGContext _windowForXWindow:xEvent.xconfigure.window];
		/*
		 * Ignore events for unmapped windows.
		 */
		if (window != 0 && window->map_state == IsViewable)
		  {
		    NSTimeInterval	ts = (NSTimeInterval)generic.lastMotion;
		    NSRect		r;
		    NSRect		x;
		    NSRect		n;

		    /*
		     * Get OpenStep frame coordinates from X frame.
		     * If it's not from the window mmanager, ignore x and y.
		     */
		    r = window->xframe;
		    if (xEvent.xconfigure.send_event == 0)
		      {
			x = NSMakeRect(r.origin.x, r.origin.y,
			  xEvent.xconfigure.width, xEvent.xconfigure.height);
		      }
		    else
		      {
			x = NSMakeRect(xEvent.xconfigure.x, xEvent.xconfigure.y,
			  xEvent.xconfigure.width, xEvent.xconfigure.height);
		      }
		    window->xframe.origin = x.origin;
		    n = [self _XFrameToOSFrame: x for: window];
		    NSDebugLLog(@"Moving", @"Config %d - o:%@, x:%@",
		      window->number, NSStringFromRect(n), NSStringFromRect(x));

		    /*
		     * Set size hints info to be up to date with new size.
		     */
		    x = [self _OSFrameToXHints: n for: window];
		    window->siz_hints.width = x.size.width;
		    window->siz_hints.height = x.size.height;
		    window->siz_hints.x = x.origin.x;
		    window->siz_hints.y = x.origin.y;

		    /*
		     * create GNUstep event(s)
		     */
		    if (!NSEqualSizes(r.size, n.size))
		      {
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: ts
					   windowNumber: window->number
						context: self
						subtype: GSAppKitWindowResized
						  data1: n.size.width
						  data2: n.size.height];
		      }
		    if (!NSEqualPoints(r.origin, n.origin))
		      {
			if (e != nil)
			  {
			    [event_queue addObject: e];
			  }
			e = [NSEvent otherEventWithType: NSAppKitDefined
					       location: eventLocation
					  modifierFlags: eventFlags
					      timestamp: ts
					   windowNumber: window->number
						context: self
						subtype: GSAppKitWindowMoved
						  data1: n.origin.x
						  data2: n.origin.y];
		      }
		  }
		break;

	      // same as ConfigureNotify but we get this event
	      // before the change has actually occurred
	      case ConfigureRequest:
		NSDebugLLog(@"NSEvent", @"%d ConfigureRequest\n",
		  xEvent.xconfigurerequest.window);
		break;

	      // a window has been created
	      case CreateNotify:
		NSDebugLLog(@"NSEvent", @"%d CreateNotify\n",
		  xEvent.xcreatewindow.window);
		break;

	      // a window has been destroyed
	      case DestroyNotify:
		NSDebugLLog(@"NSEvent", @"%d DestroyNotify\n",
		  xEvent.xdestroywindow.window);
		break;

	      // when the pointer enters a window
	      case EnterNotify:
		NSDebugLLog(@"NSEvent", @"%d EnterNotify\n",
		  xEvent.xcrossing.window);
		break;

	      // when the pointer leaves a window
	      case LeaveNotify:
		NSDebugLLog(@"NSEvent", @"%d LeaveNotify\n",
		  xEvent.xcrossing.window);
		break;

	      // the visibility of a window has changed
	      case VisibilityNotify:
		NSDebugLLog(@"NSEvent", @"%d VisibilityNotify %d\n", 
		  xEvent.xvisibility.window, xEvent.xvisibility.state);
		break;

	      // a portion of the window has become visible and
	      // we must redisplay it
	      case Expose:
		NSDebugLLog(@"NSEvent", @"%d Expose\n",
		  xEvent.xexpose.window);
		{
		  XRectangle rectangle;
		  static gswindow_device_t *exposeWin = NULL;

		  if (!exposeWin || xEvent.xbutton.window != exposeWin->ident)
		    exposeWin = [XGContext _windowForXWindow: xEvent.xbutton.window ];
		  rectangle.x = xEvent.xexpose.x;
		  rectangle.y = xEvent.xexpose.y;
		  rectangle.width = xEvent.xexpose.width;
		  rectangle.height = xEvent.xexpose.height;
		  NSDebugLLog(@"NSEvent", @"Expose frame %d %d %d %d\n",
			    rectangle.x, rectangle.y,
			    rectangle.width, rectangle.height);
		  [self _addExposedRectangle: rectangle : exposeWin->number];

		  if (xEvent.xexpose.count == 0)
		    [self _processExposedRectangles: exposeWin->number];
		  break;
		}

	      // keyboard focus entered a window
	      case FocusIn:
		NSDebugLLog(@"NSEvent", @"%d FocusIn\n",
		  xEvent.xfocus.window);
		window = [XGContext _windowForXWindow: xEvent.xfocus.window];
		if (window == 0)
		  break;
		generic.currentFocusWindow = window->number;
		if (xEvent.xfocus.serial == generic.focusRequestNumber)
		  {
		    /*
		     * This is a response to our own request - so we mark the
		     * request as complete.
		     */
		    generic.desiredFocusWindow = 0;
		    generic.focusRequestNumber = 0;
		  }
		break;

	      // keyboard focus left a window
	      case FocusOut:
		{
		  Window	fw;
		  int		rev;

		  /*
		   * See where the focus has moved to -
		   * If it has gone to 'none' or 'PointerRoot' then 
		   * it's not one of ours.
		   * If it has gone to our root window - use the icon window.
		   * If it has gone to a window - we see if it is one of ours.
		   */
		  XGetInputFocus(xEvent.xfocus.display, &fw, &rev);
		  NSDebugLLog(@"NSEvent", @"%d FocusOut\n",
		    xEvent.xfocus.window);
		  window = [XGContext _windowForXWindow: fw];
		  if (window == 0)
		    {
		      nswin = nil;
		    }
		  else
		    {
		      nswin = GSWindowWithNumber(window->number);
		    }
		  if (nswin == nil)
		    {
		      [NSApp deactivate]; 
		    }
		}
		break;

	      case GraphicsExpose:
		NSDebugLLog(@"NSEvent", @"%d GraphicsExpose\n",
		  xEvent.xexpose.window);
		break;

	      case NoExpose:
		NSDebugLLog(@"NSEvent", @"NoExpose\n");
		break;

	      // window is moved because of a change in the size of its parent
	      case GravityNotify:
		NSDebugLLog(@"NSEvent", @"%d GravityNotify\n",
		  xEvent.xgravity.window);
		break;

	      // a key has been pressed
	      case KeyPress:
		NSDebugLLog(@"NSEvent", @"%d KeyPress\n",
		  xEvent.xkey.window);
		generic.lastTime = xEvent.xkey.time;
		e = process_key_event (&xEvent, self, NSKeyDown);
		break;

	      // a key has been released
	      case KeyRelease:
		NSDebugLLog(@"NSEvent", @"%d KeyRelease\n",
		  xEvent.xkey.window);
		generic.lastTime = xEvent.xkey.time;
		e = process_key_event (&xEvent, self, NSKeyUp);
		break;

	      // reports the state of the keyboard when pointer or
	      // focus enters a window
              case KeymapNotify:
		NSDebugLLog(@"NSEvent", @"%d KeymapNotify\n",
		  xEvent.xkeymap.window);
		// Check if control is pressed 
		if (_control_keycodes[1] 
		    && check_modifier (&xEvent, _control_keycodes[1]))
		  _control_pressed = YES;
		else if (_control_keycodes[2] 
			 && check_modifier (&xEvent, _control_keycodes[2]))
		  _control_pressed = YES;
		else 
		  _control_pressed = NO;
		// Check if command is pressed
		if (_command_keycodes[1] 
		    && check_modifier (&xEvent, _command_keycodes[1]))
		  _command_pressed = YES;
		else if (_command_keycodes[2] 
			 && check_modifier (&xEvent, _command_keycodes[2]))
		  _command_pressed = YES;
		else 
		  _command_pressed = NO;
		// Check if alt is pressed
		if (_alt_keycodes[1] 
		    && check_modifier (&xEvent, _alt_keycodes[1]))
		  _alt_pressed = YES;
		else if (_alt_keycodes[2] 
			 && check_modifier (&xEvent, _alt_keycodes[2]))
		  _alt_pressed = YES;
		else 
		  _alt_pressed = NO;
 		break;

	      // when a window changes state from ummapped to
	      // mapped or vice versa
	      case MapNotify:
		NSDebugLLog(@"NSEvent", @"%d MapNotify\n",
		  xEvent.xmap.window);
		window = [XGContext _windowForXWindow: xEvent.xunmap.window];
		if (window != 0)
		  {
		    window->map_state = IsViewable;
		    nswin = GSWindowWithNumber(window->number);
		    [nswin _setVisible: YES];
		    /*
		     * if the window that was just mapped wants the input
		     * focus, re-do the request.
		     */
		    if (generic.desiredFocusWindow == window->number
		      && generic.focusRequestNumber == 0)
		      {
			[self DPSsetinputfocus: window->number];
		      }
		    /*
		     * Make sure that the newly mapped window displays.
		     */
		    [nswin update];
		  }
		break;

	      // Window is no longer visible.
	      case UnmapNotify:
		NSDebugLLog(@"NSEvent", @"%d UnmapNotify\n",
		  xEvent.xunmap.window);
		window = [XGContext _windowForXWindow: xEvent.xunmap.window];
		if (window != 0)
		  {
		    window->map_state = IsUnmapped;
		    nswin = GSWindowWithNumber(window->number);
		    [nswin _setVisible: NO];
		  }
		break;

	      // like MapNotify but occurs before the request is carried out
	      case MapRequest:
		NSDebugLLog(@"NSEvent", @"%d MapRequest\n",
		  xEvent.xmaprequest.window);
		break;

	      // keyboard or mouse mapping has been changed by another client
	      case MappingNotify:
		NSDebugLLog(@"NSEvent", @"%d MappingNotify\n",
		  xEvent.xmapping.window);
		if ((xEvent.xmapping.request == MappingModifier) 
		    || (xEvent.xmapping.request == MappingKeyboard))
		  {
		    XRefreshKeyboardMapping (&xEvent.xmapping);
		    set_up_num_lock ();
		  }
		break;

	      case MotionNotify:
		NSDebugLLog(@"NSEvent", @"%d MotionNotify - %d %d\n",
		  xEvent.xmotion.window, xEvent.xmotion.x, xEvent.xmotion.y);
		{
		  unsigned int	state;

		  /*
		   * Compress motion events to avoid flooding.
		   */
		  while (XPending(xEvent.xmotion.display))
		    {
		      XEvent	peek;

		      XPeekEvent(xEvent.xmotion.display, &peek);
		      if (peek.type == MotionNotify
			&& xEvent.xmotion.window == peek.xmotion.window
			&& xEvent.xmotion.subwindow == peek.xmotion.subwindow)
			{
			  XNextEvent(xEvent.xmotion.display, &xEvent);
			}
		      else
			{
			  break;
			}
		    }

		  generic.lastMotion = xEvent.xmotion.time;
		  generic.lastTime = generic.lastMotion;
		  state = xEvent.xmotion.state;
		  if (state & Button1Mask)
		    {
		      eventType = NSLeftMouseDragged;
		    }
		  // Count any button other than 1 as right
		  else if (state & (Button2Mask | Button3Mask | Button4Mask | Button5Mask))
		    {
		      eventType = NSRightMouseDragged;
		    }
		  else
		    {
		      eventType = NSMouseMoved;
		    }

		  if (state != last_XEvent_state)
		    {
		      eventFlags = process_modifier_flags(state);
		      last_XEvent_state = state;
		    }

		  // if pointer is grabbed use grab window instead
		  xWin = (grabWindow == 0)
		    ? xEvent.xmotion.window : grabWindow;
		  if (window == 0 || xWin != window->ident)
		    window = [XGContext _windowForXWindow: xWin];
		  if (window == 0)
		    break;

		  eventLocation = NSMakePoint(xEvent.xmotion.x,
					      NSHeight(window->xframe) 
					      - xEvent.xmotion.y);

		  e = [NSEvent mouseEventWithType: eventType
					 location: eventLocation
				    modifierFlags: eventFlags
					timestamp: (NSTimeInterval)generic.lastTime
				     windowNumber: window->number
					  context: self
				      eventNumber: xEvent.xmotion.serial
				       clickCount: 1
					 pressure: 1.0];
		  break;
		}

	      // a window property has changed or been deleted
	      case PropertyNotify:
		NSDebugLLog(@"NSEvent", @"%d PropertyNotify - '%s'\n",
		  xEvent.xproperty.window,
		  XGetAtomName(XDPY, xEvent.xproperty.atom));
		break;

	      // a client successfully reparents a window
	      case ReparentNotify:
		NSDebugLLog(@"NSEvent", @"%d ReparentNotify - offset %d %d\n",
		  xEvent.xreparent.window, xEvent.xreparent.x,
		  xEvent.xreparent.y);
		window = [XGContext _windowForXWindow: xEvent.xreparent.window];
		if (window != 0)
		  {
		    window->parent = xEvent.xreparent.parent;
		  }
		break;

	      // another client attempts to change the size of a window
	      case ResizeRequest:
		NSDebugLLog(@"NSEvent", @"%d ResizeRequest\n",
		  xEvent.xresizerequest.window);
		break;

	      // events dealing with the selection
	      case SelectionClear:
		NSDebugLLog(@"NSEvent", @"%d SelectionClear\n",
		  xEvent.xselectionclear.window);
		break;

	      case SelectionNotify:
		NSDebugLLog(@"NSEvent", @"%d SelectionNotify\n",
		  xEvent.xselection.requestor);
		break;

	      case SelectionRequest:
		NSDebugLLog(@"NSEvent", @"%d SelectionRequest\n",
		  xEvent.xselectionrequest.requestor);
		break;

	      // We shouldn't get here unless we forgot to trap an event above
	      default:
		NSLog(@"Received an untrapped event\n");
		break;
	    }
	  if (e)
	    [event_queue addObject: e];
	  e = nil;
	}
    }
}

// Return the key_code corresponding to the user defaults string
// Return 1 (which is an invalid keycode) if the user default 
// is not set
static KeyCode
default_key_code (Display *display, NSUserDefaults *defaults, 
		  NSString *aString)
{
  NSString *keySymString;
  KeySym a_key_sym;
  
  keySymString = [defaults stringForKey: aString];
  if (keySymString == nil)
    return 1; 
  
  a_key_sym = XStringToKeysym ([keySymString cString]);
  if (a_key_sym == NoSymbol)
    {
      // This is not necessarily an error.
      // If you want on purpose to disable a key, 
      // set its default to 'NoSymbol'.
      NSLog (@"KeySym %@ not found; disabling %@", keySymString, aString);
      return 0;
    }
  
  return XKeysymToKeycode (display, a_key_sym);
}

// This function should be called before any keyboard event is dealed with.
static void
initialize_keyboard (void)
{
  NSUserDefaults *defaults = [NSUserDefaults standardUserDefaults];
  Display *display = [XGContext currentXDisplay];

  // Initialize Control
  _control_keycodes[1] = default_key_code (display, defaults, 
					   @"GSFirstControlKey");
  if (_control_keycodes[1] == 1) // No User Default Set
    _control_keycodes[1] = XKeysymToKeycode (display, XK_Control_L);

  _control_keycodes[2] = default_key_code (display, defaults, 
					   @"GSSecondControlKey");
  if (_control_keycodes[2] == 1) 
    _control_keycodes[2] = XKeysymToKeycode (display, XK_Control_R);

  // Initialize Command
  _command_keycodes[1] = default_key_code (display, defaults, 
					      @"GSFirstCommandKey");
  if (_command_keycodes[1] == 1) 
    _command_keycodes[1] = XKeysymToKeycode (display, XK_Alt_L);

  _command_keycodes[2] = default_key_code (display, defaults, 
					   @"GSSecondCommandKey");
  if (_command_keycodes[2] == 1) 
    _command_keycodes[2] = 0;  

  // Initialize Alt
  _alt_keycodes[1] = default_key_code (display, defaults, 
				       @"GSFirstAlternateKey");
  if (_alt_keycodes[1] == 1) 
    _alt_keycodes[1] = XKeysymToKeycode (display, XK_Alt_R);
  _alt_keycodes[2] = default_key_code (display, defaults, 
				       @"GSSecondAlternateKey");
  if (_alt_keycodes[2] == 1) 
    _alt_keycodes[2] = 0;  
  
  set_up_num_lock ();
  
  _is_keyboard_initialized = YES;
}


static void
set_up_num_lock (void)
{
  XModifierKeymap *modifier_map;
  int i, j;
  unsigned int modifier_masks[8] = 
  {
    ShiftMask, LockMask, ControlMask, Mod1Mask, 
    Mod2Mask, Mod3Mask, Mod4Mask, Mod5Mask
  };
  Display *display = [XGContext currentXDisplay];
  KeyCode _num_lock_keycode;
  
  // Get NumLock keycode
  _num_lock_keycode = XKeysymToKeycode (display, XK_Num_Lock);
  if (_num_lock_keycode == 0)
    {
      // Weird.  There is no NumLock in this keyboard.
      _num_lock_mask = 0; 
      return;
    }

  // Get the current modifier mapping
  modifier_map = XGetModifierMapping (display);
  
  // Scan the modifiers for NumLock
  for (j = 0; j < 8; j++)
    for (i = 0; i < (modifier_map->max_keypermod); i++)
      {
	if ((modifier_map->modifiermap)[i + j*modifier_map->max_keypermod] 
	    == _num_lock_keycode)
	  {
	    _num_lock_mask = modifier_masks[j];
	    XFreeModifiermap (modifier_map);
	    return;
	  }
      }
  // Weird.  NumLock is not among the modifiers
  _num_lock_mask = 0;
  XFreeModifiermap (modifier_map);
  return;
}

	// TODO: correctly map key events to
	// NSEvents. This can be made when 16-bit
	// character strings will be available.
static NSEvent*
process_key_event (XEvent* xEvent, XGContext* context, NSEventType eventType)
{
  char	buf[256];
  int	count;
  XComposeStatus compose;
  NSString	*keys, *ukeys;
  KeySym	keysym;
  NSPoint	eventLocation;
  unsigned short keyCode;
  unsigned int	eventFlags;
  NSEvent	*event = nil;
  NSApplication *theApp = [NSApplication sharedApplication];
  gswindow_device_t *window;
  
  NSDebugLog(@"Process key event");

  if (_is_keyboard_initialized == NO)
    initialize_keyboard ();

  // If this is a Control Key, update _control_pressed variable
  if ((xEvent->xkey.keycode == _control_keycodes[1]) 
      || (xEvent->xkey.keycode == _control_keycodes[2]))
    {
      if (xEvent->xkey.type == KeyPress)
	_control_pressed = YES;
      else if (xEvent->xkey.type == KeyRelease)
	_control_pressed = NO;
      
      eventType = NSFlagsChanged;
      // TODO: I think we can now jump most of the remaining X stuff. 
      // (do the same below, twice)
    }
  // If this is a Command Key, update _command_pressed variable
  else if ((xEvent->xkey.keycode == _command_keycodes[1]) 
	   || (xEvent->xkey.keycode == _command_keycodes[2]))
    {
      if (xEvent->xkey.type == KeyPress)
	_command_pressed = YES;
      else if (xEvent->xkey.type == KeyRelease)
	_command_pressed = NO;
      
      eventType = NSFlagsChanged;
    }
  // If this is a Alt Key, update _alt_pressed variable
  else if ((xEvent->xkey.keycode == _alt_keycodes[1]) 
	   || (xEvent->xkey.keycode == _alt_keycodes[2]))
    {
      if (xEvent->xkey.type == KeyPress)
	_alt_pressed = YES;
      else if (xEvent->xkey.type == KeyRelease)
	_alt_pressed = NO;
      
      eventType = NSFlagsChanged;
    }

  eventFlags = process_modifier_flags(xEvent->xkey.state);

  count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);
  // Add NSNumericPadKeyMask if the key is in the KeyPad
  if (IsKeypadKey (keysym))
    eventFlags = eventFlags | NSNumericPadKeyMask;

  // Make sure that the string is properly terminated
  if (count > 255)
    buf[255] = '\0';
  else
    {
      if (count < 1)
	buf[0] = '\0';
      else
	buf[count] = '\0';
    }

  window = [XGContext _windowWithTag: [[theApp keyWindow] windowNumber]];
  eventLocation.x = xEvent->xbutton.x;
  eventLocation.y = window ? window->siz_hints.height - xEvent->xbutton.y
    : xEvent->xbutton.y;
  NSDebugLog (@"keysym=%d, xLocation = (%d, %d), userLocation = (%f, %f)",
	      keysym, xEvent->xbutton.x, xEvent->xbutton.y,
	      eventLocation.x, eventLocation.y);

  keyCode = process_key_code(xEvent, keysym, eventFlags);
  keys = [NSString stringWithCString: buf];

  // Now the same ignoring modifiers, except Shift, ShiftLock, NumLock.
  xEvent->xkey.state = (xEvent->xkey.state & (ShiftMask | LockMask 
					      | _num_lock_mask));
  count = XLookupString((XKeyEvent *)xEvent, buf, 256, &keysym, &compose);
  
  // Make sure that the string is properly terminated
  if (count > 255)
    buf[255] = '\0';
  else
    {
      if (count < 1)
	buf[0] = '\0';
      else
	buf[count] = '\0';
    }
  ukeys = [NSString stringWithCString: buf];
  
  if (keyCode == 0 && *buf == '\0'
    && context->generic.lastKeyFlags != eventFlags)
    eventType = NSFlagsChanged;
  context->generic.lastKeyFlags = eventFlags;

  event = [NSEvent keyEventWithType: eventType
			   location: eventLocation
		      modifierFlags: eventFlags
			  timestamp: (NSTimeInterval)xEvent->xkey.time
		       windowNumber: window->number
			    context: context
			 characters: keys
	charactersIgnoringModifiers: ukeys
			  isARepeat: NO
			    keyCode: keyCode];

  return event;
}

static unsigned short
process_key_code(XEvent *xEvent, KeySym keysym, 
		 unsigned int eventModifierFlags)
{
  unsigned short keyCode = 0;

  if ((keysym == XK_Return) || (keysym == XK_KP_Enter)
      || (keysym == XK_Linefeed))
    {				// do nothing for now
      keyCode = 0x0d;
    }
  else if ((keysym >= XK_Shift_L) && (keysym <= XK_Hyper_R))
    {
      // The shift button is depressed.  This should have
      // already been detected in the process_modifier_flags
      // function.  Therefore, nothing is done here...
    }
  if ((keysym >= XK_F1) && (keysym <= XK_F35))
    {
      // if a function key was pressed
      eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;

      switch (keysym)
	{
	  case XK_F1: keyCode = NSF1FunctionKey; break;
	  case XK_F2: keyCode = NSF2FunctionKey; break;
	  case XK_F3: keyCode = NSF3FunctionKey; break;
	  case XK_F4: keyCode = NSF4FunctionKey; break;
	  case XK_F5: keyCode = NSF5FunctionKey; break;
	  case XK_F6: keyCode = NSF6FunctionKey; break;
	  case XK_F7: keyCode = NSF7FunctionKey; break;
	  case XK_F8: keyCode = NSF8FunctionKey; break;
	  case XK_F9: keyCode = NSF9FunctionKey; break;
	  case XK_F10: keyCode = NSF10FunctionKey; break;
	  case XK_F11: keyCode = NSF11FunctionKey; break;
	  case XK_F12: keyCode = NSF12FunctionKey; break;
	  case XK_F13: keyCode = NSF13FunctionKey; break;
	  case XK_F14: keyCode = NSF14FunctionKey; break;
	  case XK_F15: keyCode = NSF15FunctionKey; break;
	  case XK_F16: keyCode = NSF16FunctionKey; break;
	  case XK_F17: keyCode = NSF17FunctionKey; break;
	  case XK_F18: keyCode = NSF18FunctionKey; break;
	  case XK_F19: keyCode = NSF19FunctionKey; break;
	  case XK_F20: keyCode = NSF20FunctionKey; break;
	  case XK_F21: keyCode = NSF21FunctionKey; break;
	  case XK_F22: keyCode = NSF22FunctionKey; break;
	  case XK_F23: keyCode = NSF23FunctionKey; break;
	  case XK_F24: keyCode = NSF24FunctionKey; break;
	  case XK_F25: keyCode = NSF25FunctionKey; break;
	  case XK_F26: keyCode = NSF26FunctionKey; break;
	  case XK_F27: keyCode = NSF27FunctionKey; break;
	  case XK_F28: keyCode = NSF28FunctionKey; break;
	  case XK_F29: keyCode = NSF29FunctionKey; break;
	  case XK_F30: keyCode = NSF30FunctionKey; break;
	  case XK_F31: keyCode = NSF31FunctionKey; break;
	  case XK_F32: keyCode = NSF32FunctionKey; break;
	  case XK_F33: keyCode = NSF33FunctionKey; break;
	  case XK_F34: keyCode = NSF34FunctionKey; break;
	  case XK_F35: keyCode = NSF35FunctionKey; break;

	  default: 			// do nothing
	}
    }
  else
    {
      if (keysym == XK_BackSpace)
	{
	  keyCode = NSBackspaceKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Delete)
	{
	  keyCode = NSDeleteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Home)
	{
	  keyCode = NSHomeFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Left)
	{
	  keyCode = NSLeftArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Up)
	{
	  keyCode = NSUpArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Right)
	{
	  keyCode = NSRightArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Down)
	{
	  keyCode = NSDownArrowFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Prior)
	{
	  keyCode = NSPrevFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Up)
	{
	  keyCode = NSPageUpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Next)
	{
	  keyCode = NSNextFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Page_Down)
	{
	  keyCode = NSPageDownFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_End)
	{
	  keyCode = NSEndFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Begin)
	{
	  keyCode = NSBeginFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Select)
	{
	  keyCode = NSSelectFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Print)
	{
	  keyCode = NSPrintFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Execute)
	{
	  keyCode = NSExecuteFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Insert)
	{
	  keyCode = NSInsertFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Undo)
	{
	  keyCode = NSUndoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Redo)
	{
	  keyCode = NSRedoFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Menu)
	{
	  keyCode = NSMenuFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Find)
	{
	  keyCode = NSFindFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Help)
	{
	  keyCode = NSHelpFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Break)
	{
	  keyCode = NSBreakFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Mode_switch)
	{
	  keyCode = NSModeSwitchFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#ifndef NeXT
	}
      else if (keysym == XK_Sys_Req)
	{
	  keyCode = NSSysReqFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
#endif
	}
      else if (keysym == XK_Scroll_Lock)
	{
	  keyCode = NSScrollLockFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Pause)
	{
	  keyCode = NSPauseFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if (keysym == XK_Clear)
	{
	  keyCode = NSClearDisplayFunctionKey;
	  eventModifierFlags = eventModifierFlags | NSFunctionKeyMask;
	}
      else if ((keysym == XK_Shift_L) || (keysym == XK_Shift_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSShiftKeyMask;
	}
      else if ((keysym == XK_Control_L) || (keysym == XK_Control_R))
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSControlKeyMask;
	}
      else if (keysym == XK_Alt_R)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSAlternateKeyMask;
	}
      else if (keysym == XK_Alt_L)
	{
	  eventModifierFlags = eventModifierFlags
	    | NSFunctionKeyMask | NSCommandKeyMask;
	}
#ifdef XK_XKB_KEYS
      else if (keysym == XK_Tab || keysym == XK_ISO_Left_Tab)
#else /* !XK_XKB_KEYS */
      else if (keysym == XK_Tab)
#endif
	{
	  keyCode = 0x09;
	}
      else if (keysym == XK_Escape)
	{
	  keyCode = 0x1b;
	}
    }
  // If the key press originated from the key pad
  if ((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
    {
      eventModifierFlags = eventModifierFlags | NSNumericPadKeyMask;

      switch (keysym)
	{
	  case XK_KP_F1:        keyCode = NSF1FunctionKey;         break;
	  case XK_KP_F2:        keyCode = NSF2FunctionKey;         break;
	  case XK_KP_F3:        keyCode = NSF3FunctionKey;         break;
	  case XK_KP_F4:        keyCode = NSF4FunctionKey;         break;
#ifndef NeXT
	  case XK_KP_Home:      keyCode = NSHomeFunctionKey;       break;
	  case XK_KP_Left:      keyCode = NSLeftArrowFunctionKey;  break;
	  case XK_KP_Up:        keyCode = NSUpArrowFunctionKey;    break;
	  case XK_KP_Right:     keyCode = NSRightArrowFunctionKey; break;
	  case XK_KP_Down:      keyCode = NSDownArrowFunctionKey;  break;
	  case XK_KP_Prior:     keyCode = NSPrevFunctionKey;       break;
//		case XK_KP_Page_Up:   keyCode = NSPageUpFunctionKey;     break;
	  case XK_KP_Next:      keyCode = NSNextFunctionKey;       break;
//		case XK_KP_Page_Down: keyCode = NSPageDownFunctionKey;   break;
	  case XK_KP_End:       keyCode = NSEndFunctionKey;        break;
	  case XK_KP_Begin:     keyCode = NSBeginFunctionKey;      break;
	  case XK_KP_Insert:    keyCode = NSInsertFunctionKey;     break;
	  case XK_KP_Delete:    keyCode = NSDeleteFunctionKey;     break;
#endif
	  default:  break;  /* Nothing to do */
	}
    }

  if (((keysym >= XK_KP_Space) && (keysym <= XK_KP_9))
      || ((keysym >= XK_space) && (keysym <= XK_asciitilde)))
    {
    }					// Not processed

  return keyCode;
}

// process_modifier_flags() determines which modifier keys (Command, Control,
// Shift,  and so forth) were held down while the event occured.
static unsigned int
process_modifier_flags(unsigned int state)
{
  unsigned int eventModifierFlags = 0;

  if (state & ShiftMask)
    eventModifierFlags = eventModifierFlags | NSShiftKeyMask;

  if (state & LockMask)
    eventModifierFlags = eventModifierFlags | NSShiftKeyMask;

  if (_control_pressed)
    eventModifierFlags = eventModifierFlags | NSControlKeyMask;

  if (_command_pressed)
    eventModifierFlags = eventModifierFlags | NSCommandKeyMask;

  if (_alt_pressed)
    eventModifierFlags = eventModifierFlags | NSAlternateKeyMask;
  
  // Other modifiers ignored for now. 

  return eventModifierFlags;
}

- (NSDate*) timedOutEvent: (void*)data
                     type: (RunLoopEventType)type
                  forMode: (NSString*)mode
{
  return nil;
}

/* Drag and Drop */
- (void) _resetDragTypes: (NSArray*)types toWindow: (int)winNum
{
  int count, i;
  Atom *typelist;
  gswindow_device_t *window;
  NSString *type;
  NSCountedSet *drag_set = [self _dragTypesForWindow: winNum];
  NSEnumerator *drag_enum = [drag_set objectEnumerator];

  window = [XGContext _windowWithTag: winNum];
  count = [drag_set count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  i = 0;
  while ((type = [drag_enum nextObject]))
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: type];
      NSDebugLog(@"  registering %@ on window\n", mime);
      typelist[i] = XInternAtom(XDPY, [mime cString], False);
      i++;
    }
  typelist[i] = 0;
  xdnd_set_dnd_aware(&dnd, window->ident, typelist);
  NSZoneFree([self zone], typelist);
}

- (BOOL) _addDragTypes: (NSArray*)types toWindow: (int)winNum
{
  BOOL did_add;
  gswindow_device_t *window;

  window = [XGContext _windowWithTag: winNum];
  if (!xDndInitialized && window)
    {
      xDndInitialized = YES;
      xdnd_init(&dnd, XDPY);
      xdnd_set_dnd_aware(&dnd, window->ident, NULL);
    }
  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (BOOL) _removeDragTypes: (NSArray*)types fromWindow: (int)winNum
{
  BOOL did_add;

  did_add = [super _addDragTypes: types toWindow: winNum];
  if (did_add)
    [self _resetDragTypes: types toWindow: winNum];
  return did_add;
}

- (id <NSDraggingInfo>)_dragInfo
{
  return [XGDragView sharedDragView];
}

- (void) _postExternalEvent: (NSEvent *)theEvent
{
  gswindow_device_t *window;

  window = [XGContext _windowWithTag: [[theEvent window] windowNumber]];
  if ([theEvent subtype] == GSAppKitDraggingStatus)
    {
      Atom xaction;
      NSDragOperation action = [theEvent data2];
      xaction = GSActionForDragOperation(action);
      xdnd_send_status(&dnd, 
		       [theEvent data1],
		       window->ident,
		       (action != NSDragOperationNone),
		       0,
		       0, 0, 0, 0,
		       xaction);
    }
  else if ([theEvent subtype] == GSAppKitDraggingFinished)
    {
      xdnd_send_finished(&dnd, 
			 [theEvent data1],
			 window->ident,
			 0);
    }
  else
    {
      NSDebugLLog(@"NSDragging", @"Internal: unhandled post external event");
    }

}

@end

@implementation XGContext (XSync)
- (BOOL) xSyncMap: (void*)windowHandle
{
  gswindow_device_t *window = (gswindow_device_t*)windowHandle;

  /*
   * if the window is not mapped, make sure we have sent all requests to the
   * X-server, it may be that our mapping request was buffered.
   */
  if (window->map_state != IsViewable)
    {
      XSync(XDPY, False);
      [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
    }
  /*
   * If the window is still not mapped, it may be that the window-manager
   * intercepted our mapping request, and hasn't dealt with it yet.
   * Listen for input for up to a second, in the hope of getting the mapping.
   */
  if (window->map_state != IsViewable)
    {
      NSDate	*d = [NSDate dateWithTimeIntervalSinceNow: 1.0];
      NSRunLoop	*l = [NSRunLoop currentRunLoop];
      NSString	*m = [l currentMode];

      while (window->map_state != IsViewable && [d timeIntervalSinceNow] > 0)
        {
	  [l runMode: m beforeDate: d];
	}
    }
  if (window->map_state != IsViewable)
    {
      NSLog(@"Window still not mapped a second after mapping request made");
      return NO;
    }
  return YES;
}
@end

@implementation XGContext (X11Ops)

/*
 * Return mouse location in base coords ignoring the event loop
 */
- (void) DPSmouselocation: (float*)x :(float*)y
{
  Window	rootWin;
  Window	childWin;
  int		currentX;
  int		currentY;
  int		winX;
  int		winY;
  unsigned	mask;
  BOOL		ok;

  ok = XQueryPointer (XDPY, [self xDisplayRootWindow],
    &rootWin, &childWin, &currentX, &currentY, &winX, &winY, &mask);
  if (ok)
    {
      *x = currentX;
      *y = DisplayHeight(XDPY, XSCR) - currentY;
    }
  else
    {
      *x = 0;
      *y = 0;
    }
}

- (NSEvent*) DPSGetEventMatchingMask: (unsigned)mask
			  beforeDate: (NSDate*)limit
			      inMode: (NSString*)mode
			     dequeue: (BOOL)flag
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  return [super DPSGetEventMatchingMask: mask
			     beforeDate: limit
				 inMode: mode
				dequeue: flag];
}

- (void) DPSDiscardEventsMatchingMask: (unsigned)mask
			  beforeEvent: (NSEvent*)limit
{
  [self receivedEvent: 0 type: 0 extra: 0 forMode: nil];
  [super DPSDiscardEventsMatchingMask: mask
			  beforeEvent: limit];
}

@end

@interface XGRawWindow : NSWindow
@end

@implementation XGRawWindow

- (BOOL) canBecomeMainWindow
{
  return NO;
}

- (BOOL) canBecomeKeyWindow
{
  return NO;
}

- (void) _initDefaults
{
  [super _initDefaults];
  [self setReleasedWhenClosed: NO];
  [self setExcludedFromWindowsMenu: YES];
}

- (id) initWithContentRect: (NSRect)contentRect
		 styleMask: (unsigned int)aStyle
		   backing: (NSBackingStoreType)bufferingType
		     defer: (BOOL)flag
		    screen: (NSScreen*)aScreen
{
  XSetWindowAttributes winattrs;
  unsigned long valuemask;
  gswindow_device_t *window;

  self = [super initWithContentRect: contentRect
			  styleMask: aStyle
			    backing: bufferingType
			      defer: flag
			     screen: aScreen];

  window = [XGContext _windowWithTag: window_num];
  valuemask = (CWSaveUnder|CWOverrideRedirect);
  winattrs.save_under = True;
  winattrs.override_redirect = True;
  XChangeWindowAttributes ([XGContext currentXDisplay], window->ident,
                            valuemask, &winattrs);
  [[NSApplication sharedApplication] removeWindowsItem: self];

  return self;
}
@end



@implementation XGDragView

#define	DWZ	48

static	XGDragView	*sharedDragView = nil;

+ (XGDragView*) sharedDragView
{
  if (!sharedDragView)
    {
      NSRect			winRect = {{0, 0}, {DWZ, DWZ}};
      XGRawWindow		*sharedDragWindow = [XGRawWindow alloc];

      [sharedDragWindow initWithContentRect: winRect
				  styleMask: NSBorderlessWindowMask
				    backing: NSBackingStoreNonretained
				      defer: YES];

      sharedDragView = [XGDragView new];
      [sharedDragWindow setContentView: sharedDragView];
      RELEASE(sharedDragView);
    }

  return sharedDragView;
}

- (void) _sendLocalEvent: (GSAppKitSubtype)subtype
		  action: (NSDragOperation)action
	        position: (NSPoint)eventLocation
	       timestamp: (NSTimeInterval)time
	        toWindow: (NSWindow*)dWindow
{
  NSEvent *e;
  NSGraphicsContext *context = GSCurrentContext();
  gswindow_device_t *windev;

  windev = [XGContext _windowWithTag: [dWindow windowNumber]];
  eventLocation = NSMakePoint(eventLocation.x - NSMinX(windev->xframe),
			      eventLocation.y - NSMinY(windev->xframe));
  eventLocation.y = NSHeight(windev->xframe) - eventLocation.y;

  e = [NSEvent otherEventWithType: NSAppKitDefined
	  location: eventLocation
	  modifierFlags: 0
	  timestamp: time
	  windowNumber: windev->number
	  context: context
	  subtype: subtype
	  data1: windev->ident
	  data2: action];
  [dragWindow sendEvent: e];
}

- (void) setDragInfoFromEvent: (NSEvent*)event
{
  dragSource = nil;
  dragPasteboard = [[NSPasteboard pasteboardWithName: NSDragPboard] retain];
  dragWindow = [event window];
  dragPoint = [event locationInWindow];
  dragSequence = [event timestamp];
  dragMask = [event data2];
  dragLocal = NO;
}

- (void) drawRect: (NSRect)rect
{
  [dragCell drawWithFrame: rect inView: self];
}

- (void) dragImage: (NSImage*)anImage
		at: (NSPoint)viewLocation
	    offset: (NSSize)initialOffset
	     event: (NSEvent*)event
	pasteboard: (NSPasteboard*)pboard
	    source: (id)sourceObject
	 slideBack: (BOOL)slideFlag
{
  int i, count;
  NSArray *types;
  Display *xDisplay = [XGContext currentXDisplay];

  if (!anImage)
    anImage = [NSImage imageNamed: @"common_Close"];

  if (!dragCell)
    {
      dragCell = [[NSCell alloc] initImageCell: anImage];
      [dragCell setBordered: NO];
    }
  [dragCell setImage: anImage];
  DESTROY(dragPasteboard);
  dragPasteboard = [pboard retain];
  dragSource = sourceObject;

  types = [pboard types];
  count = [types count];
  typelist = NSZoneMalloc([self zone], (count+1) * sizeof(Atom));
  for (i = 0; i < count; i++)
    {
      NSString *mime = [NSPasteboard mimeTypeForPasteboardType: 
		       [types objectAtIndex: i]];
      typelist[i] = XInternAtom(xDisplay, [mime cString], False);
    }
  typelist[count] = 0;

  if (!xDndInitialized)
    {
      xDndInitialized = YES;
      xdnd_init(&dnd, xDisplay);
    }

  [self mouseDown: event];
  NSZoneFree([self zone], typelist);
  typelist = NULL;
}

- (void) mouseDown: (NSEvent*)theEvent
{
  XGContext     *context = (XGContext *)GSCurrentContext();
  Display	*xDisplay = [XGContext currentXDisplay];
  int		xScreen;
  unsigned int	eventMask = NSLeftMouseDownMask | NSLeftMouseUpMask
			    | NSLeftMouseDraggedMask | NSMouseMovedMask
			    | NSPeriodicMask | NSAppKitDefinedMask;
  NSPoint	point, lastPoint;
  NSDate	*theDistantFuture = [NSDate distantFuture];
  NSEventType	eventType;
  gswindow_device_t *windev;
  Window	rootWindow;
  float		wx;
  float		wy;
  int		init = 0;
  NSWindow	*eWindow = [theEvent window];
  BOOL		dnd_aware = NO;
  Window        mouseWindow;
  Window	lastMouseWindow = None;
  NSSize	imageSize;

  windev  = [XGContext _windowWithTag: [_window windowNumber]];
  xScreen = windev->screen;
  rootWindow = windev->root;

  [self unregisterDraggedTypes];

  {
    NSImageRep	*rep;

    /* Need to lockFocus to do this. FIXME when image caching works */
    [self lockFocus];
    rep = [[dragCell image] bestRepresentationForDevice: nil];
    if ([rep isKindOfClass: [NSBitmapImageRep class]])
      {
	XShapeCombineMask(xDisplay, windev->ident, ShapeBounding, 0, 0,
	  [(NSBitmapImageRep *)rep xPixmapMask], ShapeSet);
      }
    [self unlockFocus];
  }

  /* convert point and lastPoint into base coordinates (NOT X Coordinates) */
  lastPoint = [[eWindow contentView] convertPoint: [theEvent locationInWindow]
					 fromView: nil];
  lastPoint = [eWindow convertBaseToScreen: lastPoint];
  imageSize = [[dragCell image] size];
  [_window setFrame: NSMakeRect(lastPoint.x-imageSize.width/2, 
		    lastPoint.y-imageSize.height/2, 
		    imageSize.width, imageSize.height)
            display: YES];
  [_window orderFront: nil];

  /*
   * get the X coordinates of the window from the hints, since the initial
   * frame movement will not have been processed yet.
   */
  wx = windev->siz_hints.x;
  wy = windev->siz_hints.y;
  NSDebugLLog(@"NSDragging", @"Drag window X origin %f %f\n", wx, wy);
  dragSequence = [theEvent timestamp];
  dragLocal = NO;

  [NSEvent startPeriodicEventsAfterDelay: 0.02 withPeriod: 0.03];

  while ((eventType = [theEvent type]) != NSLeftMouseUp)
    {
      if (eventType == NSAppKitDefined)
	{
	  GSAppKitSubtype	sub = [theEvent subtype];

	  switch (sub)
	    {
	      case GSAppKitWindowMoved:
		/*
		 * Keep window up-to-date with its current position.
		 */
		[NSApp sendEvent: theEvent];
		break;
		
	      case GSAppKitDraggingStatus:
		NSDebugLLog(@"NSDragging", @"got GSAppKitDraggingStatus\n");
		if ([theEvent data1] == lastMouseWindow)
		  {
		    /* Check the type of operation the target will perform
		       and change the cursor accordingly */
		    /* FIXME: Implement */
		  }
		break;

	      case GSAppKitDraggingFinished:
		NSLog(@"Internal: got GSAppKitDraggingFinished out of seq");
		break;

	      default:
		NSLog(@"Internal: dropped NSAppKitDefined (%d) event", sub);
		break;
	    }
	}
      else if (eventType != NSPeriodic)
	{
	  point = [theEvent locationInWindow];
	  if (init < 1)				// dump the first point
	    {					// because it is in coords
	      init++;				// of the dragging source
	      point = lastPoint;
	    }
	  else
	    {
  	      point = [[eWindow contentView] 
				convertPoint: [theEvent locationInWindow]
				    fromView: nil];
  	      point = [eWindow convertBaseToScreen: point];
	    }
	}
      else
	{
	  if (point.x != lastPoint.x || point.y != lastPoint.y)
	    {
	      BOOL oldDragLocal;
	      NSWindow *oldDragWindow;
	      NSDragOperation action;
	      Atom xAction;
	      gswindow_device_t *dwindev;

	      /* Move drag window (point/lastPoint are reversed since
		 X Coordinates are flipped) */
NSDebugLLog(@"NSDragging", @"move from %f %f to %f %f\n",
  lastPoint.x, lastPoint.y, point.x, point.y);
	      wx += (point.x - lastPoint.x);
	      wy += (lastPoint.y - point.y);
NSDebugLLog(@"NSDragging", @"Drag window X position %f %f\n", wx, wy);
	      XMoveWindow(xDisplay, windev->ident, (int)wx, (int)wy);
	      lastPoint = point;

	      mouseWindow = _findXWindow(xDisplay, rootWindow, rootWindow,
		wx+imageSize.width+1, wy+imageSize.height/2);
	      oldDragWindow = dragWindow;
	      oldDragLocal = dragLocal;
	      dwindev = [XGContext _windowForXWindow: mouseWindow];
	      if (dwindev)
		{
		  dragWindow = GSWindowWithNumber(dwindev->number);
		}
	      else
		{
		  dragWindow = nil;
		}
	      dragLocal = (dragWindow) ? YES : NO;
	      dragPoint = lastPoint;
	      if (dragLocal)
		{
		  dragPoint = [dragWindow convertScreenToBase: dragPoint];
		}
	      action = [dragSource 
		draggingSourceOperationMaskForLocal: dragLocal];
	      xAction = GSActionForDragOperation(action);

	      if (mouseWindow != lastMouseWindow && dnd_aware)
		{
		  if (oldDragLocal)
		    {
		      [self _sendLocalEvent: GSAppKitDraggingExit
				     action: action
				   position: NSZeroPoint
				  timestamp: dragSequence
				   toWindow: oldDragWindow];
		    }
		  else
		    {
		      xdnd_send_leave(&dnd, lastMouseWindow, windev->ident);
		    }
		  lastMouseWindow = None;
		  dnd_aware = NO;

		  DPSsetcursorcolor(context, 0, 0, 0, 1, 1, 1, 
		    [[NSCursor arrowCursor] _cid]);
		}

	      if (mouseWindow != rootWindow && mouseWindow != windev->ident)
		{
		  if (mouseWindow == lastMouseWindow)
		    {
		      NSDebugLLog(@"NSDragging", @"sending dnd pos\n");
		      if (dragLocal)
			{
			  [self _sendLocalEvent: GSAppKitDraggingUpdate
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
			}
		      else
			{
			  xdnd_send_position(&dnd,
			    mouseWindow, windev->ident,
			    xAction, wx+imageSize.width+1, 
			    wy+imageSize.height/2, CurrentTime);
			}
		    }
		  else
		    {
		      dnd_aware = xdnd_is_dnd_aware(&dnd,
			mouseWindow, &dnd.dragging_version, typelist);
		      if (dnd_aware)
			{
			  DPSsetcursorcolor(context, 0, 1, 0, 0, 0, 0, 
			    [[NSCursor arrowCursor] _cid]);
			  
			  NSDebugLLog(@"NSDragging",
			    @"sending dnd enter/pos\n");
			  if (dragLocal)
			    {
			      [self _sendLocalEvent: GSAppKitDraggingEnter
		                   action: action
		                 position: NSMakePoint(wx+imageSize.width+1,
						       wy+imageSize.height/2)
		                timestamp: CurrentTime
		                 toWindow: dragWindow];
			    }
			  else
			    {
			      xdnd_send_enter(&dnd,
				mouseWindow, windev->ident, typelist);
			      xdnd_send_position(&dnd,
				mouseWindow, windev->ident,
				xAction, wx+imageSize.width+1,
				wy+imageSize.height/2, 
				CurrentTime);
			    }
			  lastMouseWindow = mouseWindow;
			}
		    }
		}
	    }
	}
      theEvent = [NSApp nextEventMatchingMask: eventMask
				    untilDate: theDistantFuture
				       inMode: NSEventTrackingRunLoopMode
				      dequeue: YES];
    }

  [NSEvent stopPeriodicEvents];
  [_window orderOut: nil];
  DPSsetcursorcolor(context, 0, 0, 0, 1, 1, 1, [[NSCursor arrowCursor] _cid]);

  if (dnd_aware && mouseWindow != (Window) None)
    {
      NSDebugLLog(@"NSDragging", @"sending dnd drop\n");
      if (dragLocal)
	{
	  [self _sendLocalEvent: GSAppKitDraggingDrop
			 action: 0
		       position: NSZeroPoint
		      timestamp: CurrentTime
		       toWindow: dragWindow];
	}
      else
	{
	  xdnd_send_drop(&dnd, lastMouseWindow, windev->ident, CurrentTime);
	}
      XSetSelectionOwner(xDisplay,
	dnd.XdndSelection, windev->ident, CurrentTime);
      XSync(xDisplay, False);
      if ([dragSource respondsToSelector:
	@selector(draggedImage:endedAt:deposited:)])
	{
	  point = [theEvent locationInWindow];
	  point = [[theEvent window] convertBaseToScreen: point];
	  [dragSource draggedImage: [dragCell image]
			   endedAt: point
			 deposited: YES];
	}
    }
  else
    {
      if ([dragSource respondsToSelector:
	@selector(draggedImage:endedAt:deposited:)])
	{
	  point = [theEvent locationInWindow];
	  point = [[theEvent window] convertBaseToScreen: point];
	  [dragSource draggedImage: [dragCell image]
			   endedAt: point
			 deposited: NO];
	}
    }
}

- (void) setImage: (NSImage *)anImage
{
  [dragCell setImage: anImage];

  [self lockFocus];
  [dragCell drawWithFrame: NSMakeRect(0,0,48,48) inView: self];
  [self unlockFocus];
}

/* NSDraggingInfo protocol */
- (NSWindow*) draggingDestinationWindow
{
  return dragWindow;
}

- (NSPoint) draggingLocation
{
  return dragPoint;
}

- (NSPasteboard*) draggingPasteboard
{
  return dragPasteboard;
}

- (int) draggingSequenceNumber
{
  return dragSequence;
}

- (id) draggingSource
{
  return dragSource;
}

- (NSDragOperation) draggingSourceOperationMask
{
  if (dragSource)
    return [dragSource draggingSourceOperationMaskForLocal: dragLocal];
  else
    return dragMask;
}

- (NSImage*) draggedImage
{
  if (dragSource)
    return [dragCell image];
  else
    return nil;
}

- (NSPoint) draggedImageLocation
{
  NSPoint loc;

  if (dragSource)
    {
      NSSize size;
      size = [[dragCell image] size];
      loc = NSMakePoint(dragPoint.x-size.width/2, dragPoint.y - size.height/2);
    }
  else
    {
      loc = dragPoint;
    }
  return loc;
}

- (void) slideDraggedImageTo: (NSPoint)screenPoint
{
}

@end
