/* 
   PXKView.m

   NSView for GNUstep GUI X/DPS Backend.

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author:  Pascal Forget <pascal@wsc.com>
   Date: January 1996
   Author:  Ovidiu Predescu <ovidiu@net-community.com>
   Date: May 1997
   
   This file is part of the GNUstep GUI X/DPS Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <AppKit/NSColor.h>
#include <AppKit/PSMatrix.h>

#include <Foundation/NSDictionary.h>
#include <Foundation/NSThread.h>
#include <Foundation/NSValue.h>

#include <gnustep/xdps/PXKView.h>
#include <gnustep/xdps/PXKDPSContext.h>
#include <gnustep/xdps/PXKWindow.h>

#include "general.h"

@implementation PSMatrix (BackendMethods)

- (void)set
{
  PSWConcatMatrix (matrix);
}

@end

@implementation PXKView

/* Class variables */
static NSString	*nsview_thread_key = @"NSViewThreadKey";

+ (void) pushFocusView:(NSView *)focusView
{
  if (focusView)
    {
      NSMutableDictionary *dict = [[NSThread currentThread] threadDictionary];
      NSMutableArray *stack = [dict objectForKey: nsview_thread_key];

      if (stack == nil)
	{
	  stack = [[NSMutableArray alloc] initWithCapacity: 4];
	  [dict setObject: stack forKey: nsview_thread_key];
	  [stack release];
	}
      [stack addObject: focusView];
    }
  else
    {
      [NSException raise: NSInternalInconsistencyException
		  format: @"Attempt to push a 'nil' focus view on to stack."];
    }
}

/*
 *	Remove the top focusView for the current thread from the stack
 *	and return the new focusView (or nil if the stack is now empty).
 */
+ (NSView *)popFocusView
{
  NSMutableDictionary *dict = [[NSThread currentThread] threadDictionary];
  NSMutableArray *stack = [dict objectForKey: nsview_thread_key];
  NSView *v = nil;

  if (stack)
    {
      unsigned count = [stack count];

      if (count > 0)
	{
	  [stack removeObjectAtIndex: --count];
	}
      if (count > 0)
	{
	  v = [stack objectAtIndex: --count];
	}
    }
  return v;
}

- (void)removeFromSuperview
{
  NSMutableArray *views;

  // No superview then just return
  if (!super_view) return;

  [self viewWillMoveToWindow:nil];

  /* Remove the view from the linked list of views maintained by the super view
     so that the view will not receive an unneeded display message. */
  [super_view _removeSubviewFromViewsThatNeedDisplay:self];

  views = [super_view subviews];
  [views removeObjectIdenticalTo:self];
  super_view = nil;
}

- (void)replaceSubview:(NSView *)oldView
		  with:(NSView *)newView
{
  if (!newView)
    return;

  if (!oldView)
    [self addSubview:newView];
  else {
    int index = [sub_views indexOfObjectIdenticalTo:oldView];

    if (index != NSNotFound) {
      [oldView viewWillMoveToWindow:nil];
      [oldView viewWillMoveToSuperview:nil];
      [newView setNextResponder:nil];

      /* Remove the view from the linked list of views so that the old view
         will not receive an unneeded display message. */
      [self _removeSubviewFromViewsThatNeedDisplay:oldView];

      [sub_views replaceObjectAtIndex:index withObject:newView];

      [newView viewWillMoveToWindow:window];
      [newView viewWillMoveToSuperview:self];
      [newView setNextResponder:self];
    }
  }
}

- (void)_setNeedsFlush
{
  [window _setNeedsFlush];
}

- (void)displayRect:(NSRect)rect
{
	[super displayRect:rect];
	[window _setNeedsFlush];
}

- (void)setNeedsDisplayO:(BOOL)flag
{
  needs_display = flag;
  if (needs_display) {
//    NSLog (@"NSView setNeedsDisplay:");
    invalidRect = bounds;
    [window _setNeedsDisplay];

    if (super_view)
      [super_view _addSubviewForNeedingDisplay:self];
  }
}

- (void)setNeedsDisplayInRectO:(NSRect)rect
{
//  NSLog (@"NSView setNeedsDisplayInRect:");
  needs_display = YES;
  invalidRect = NSUnionRect (invalidRect, rect);
  [window _setNeedsDisplay];

    if (super_view)
      [super_view _addSubviewForNeedingDisplay:self];
}

- (void)_displayNeededViews
{
  PXKView* subview;

  if (needs_display)
    [self displayIfNeeded];

  subview = (PXKView*)_subviewsThatNeedDisplay;
  while (subview) {
    [subview _displayNeededViews];
    subview = (PXKView*)subview->_nextSiblingSubviewThatNeedsDisplay;
  }
}

- (void)lockFocus
{
  PXKDPSContext *context = (PXKDPSContext*)[NSDPSContext currentContext];
  PXKWindow *w = (PXKWindow*)[self window];
  NSView *s = [self superview];

  // Save the graphics state
  PSgsave();

  // Set the Xgcdrawable
  [context setXDrawable:w];

  // lock our superview
  if (s)
    [s lockFocus];

  // push ourselves
  [[self class] pushFocusView: self];

  [frameMatrix set];
  PSrectclip (0, 0, frame.size.width, frame.size.height);
  [boundsMatrix set];
}

- (void)unlockFocus
{
  NSView *s = [self superview];

  // unlock our superview
  if (s)
    [s unlockFocus];

  // pop ourselves
  [[self class] popFocusView];

  // Restore the graphics state
  PSgrestore();
}

- (void)_removeSubviewFromViewsThatNeedDisplay:(PXKView*)view
{
  // If no subviews need to be displayed then
  // then view must not be among them
  if (!_subviewsThatNeedDisplay)
    return;

  /* Remove view from the list of subviews that need display */
  if (_subviewsThatNeedDisplay == view)
    {
      _subviewsThatNeedDisplay = view->_nextSiblingSubviewThatNeedsDisplay;
      [view _recursivelyResetNeedsDisplayInAllViews];
    }
  else {
    PXKView* currentView;

    for (currentView = _subviewsThatNeedDisplay;
	 currentView && currentView->_nextSiblingSubviewThatNeedsDisplay;
	 currentView = currentView->_nextSiblingSubviewThatNeedsDisplay)
      if (currentView->_nextSiblingSubviewThatNeedsDisplay == view)
	{
	  currentView->_nextSiblingSubviewThatNeedsDisplay
	    = view->_nextSiblingSubviewThatNeedsDisplay;
	  [view _recursivelyResetNeedsDisplayInAllViews];
	  break;
	}
  }
}

- (void)_recursivelyResetNeedsDisplayInAllViews
{
PXKView* currentView = _subviewsThatNeedDisplay;
PXKView* nextView;

	while (currentView) 
		{
		nextView = currentView->_nextSiblingSubviewThatNeedsDisplay;
		[currentView _recursivelyResetNeedsDisplayInAllViews];
		currentView->_nextSiblingSubviewThatNeedsDisplay = NULL;
		currentView = nextView;
		}

	_subviewsThatNeedDisplay = NULL;
	_nextSiblingSubviewThatNeedsDisplay = NULL;
}

- (void)_collectInvalidatedRectanglesInArray:(NSMutableArray*)array
								originMatrix:(PSMatrix*)originMatrix
								sizeMatrix:(PSMatrix*)sizeMatrix
{
PSMatrix* copyOfOriginMatrix;
PSMatrix* copyOfSizeMatrix;
PXKView* subview = _subviewsThatNeedDisplay;

	copyOfOriginMatrix = [originMatrix copy];
	copyOfSizeMatrix = [sizeMatrix copy];
	[copyOfOriginMatrix concatenateWith:frameMatrix];
	[copyOfOriginMatrix concatenateWith:boundsMatrix];
	[copyOfSizeMatrix concatenateWith:boundsMatrix];

//  NSLog (@"_collectInvalidatedRectanglesInArray");

	while (subview) 
		{
		NSRect subviewFrame;
		NSRect intersection;

		if (subview->needs_display) 
			{
			subviewFrame = subview->invalidRect;

									// Compute the origin of the invalidated 
									// rectangle in the receiver's coordinates. 
			subviewFrame = [self convertRect:subviewFrame fromView:subview];

									// If the subview is rotated compute its 
									// bounding rectangle and use this instead 
									// of the invalidated rectangle. 
			if ([subview->frameMatrix isRotated])
				[subview->frameMatrix boundingRectFor:subviewFrame
									  result:&subviewFrame];

									// Determine if the subview's invalidated 
									// frame rectangle intersects our bounds to 
									// find out if the subview gets displayed.
			intersection = NSIntersectionRect (bounds, subviewFrame);
			if (intersection.origin.x || intersection.origin.y
					|| intersection.size.width || intersection.size.height) 
				{
				NSDebugLog(@"intersection(%@) = ((%6.2f,%6.2f),(%6.2f,%6.2f))",
							NSStringFromClass(isa), intersection.origin.x, 
							intersection.origin.y, intersection.size.width, 
							intersection.size.height);
									// Convert the intersection rectangle to 
									// the window coordinates 
				intersection.origin = [copyOfOriginMatrix 
									  pointInMatrixSpace:intersection.origin];
				intersection.size = [copyOfSizeMatrix 
									sizeInMatrixSpace:intersection.size];

				[array addObject:[NSValue valueWithRect:intersection]];
				NSDebugLog (@"intersection in window coords = ((%6.2f, %6.2f),\ 
							(%6.2f, %6.2f))",
				intersection.origin.x, intersection.origin.y,
				intersection.size.width, intersection.size.height);
				}
			}
		else 
			{
			[subview _collectInvalidatedRectanglesInArray:array
					 originMatrix:copyOfOriginMatrix
					 sizeMatrix:copyOfSizeMatrix];
			}

		subview = subview->_nextSiblingSubviewThatNeedsDisplay;
		}

	[copyOfOriginMatrix release];
	[copyOfSizeMatrix release];
}

#if 0
- (void)drawRect:(NSRect)rect
{
  NSColor *c = [[self window] backgroundColor];
  [c set];
  NSRectFill(rect);
}
#endif

- (Class) classForCoder: aCoder
{
  if ([self class] == [PXKView class])
    return [super class];
  return [self class];
}

@end
