// This is -*- C++ -*-
// $Id: RCMatrix.h,v 1.2 1999/03/16 18:29:27 alstrup Exp $

/* 
 * RCMatrix.h
 *
 * Copyright (C) 1998 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#ifndef _INC_RCMATRIX_H
#define _INC_RCMATRIX_H

/*
  A very simple implementation of a matrix w/ reference counting.
*/

#include "RCArray.h"

template<class X>
class RCMatrixBody {
public:
  RCMatrixBody() : refs_(0), rows_(0), cols_(0), size_(0), data_(0) { }
  ~RCMatrixBody() { delete [] data_; }

  void add_ref() { ++refs_; }
  void remove_ref() { --refs_; }
  int refs() { return refs_; }

  size_t rows() const { return rows_; }
  size_t cols() const { return cols_; }
  size_t size() const { return size_; }
  X* data() { return data_; }

  void set_size(size_t r, size_t c) {
    delete [] data_;
    rows_ = r;
    cols_ = c;
    size_ = r*c;
    data_ = new X[size_];
  }

  RCMatrixBody* clone() const {
    RCMatrixBody* rc = new RCMatrixBody;
    rc->set_size(rows(), cols());
    memcpy(rc->data_, data_, rc->size()*sizeof(X));
    return rc;
  }

private:
  int refs_;
  size_t rows_, cols_, size_;
  X* data_;
};

template<class X>
class RCMatrix {
public:
  RCMatrix() : body_(new RCMatrixBody<X>) {
    body_->add_ref();
  }

  RCMatrix(size_t r, size_t c) : body_(new RCMatrixBody<X>) {
    body_->add_ref();
    body_->set_size(r,c);
  }

  RCMatrix(const RCMatrix<X>& r) : body_(r.body_) {
    body_->add_ref();
  }

  ~RCMatrix() {
    body_->remove_ref();
    if (body_->refs() == 0)
      delete body_;
  }

  RCMatrix& operator=(const RCMatrix<X>& r) {
    body_->remove_ref();
    if (body_->refs() == 0)
      delete body_;
    body_ = r.body_;
    body_->add_ref();
    return *this;
  }

  void clear() {
    if (body_->rows() || body_->cols())
      body_->remove_ref();
    if (body_->refs() == 0)
      delete body_;
    body_ = new RCMatrixBody<X>;
    body_->add_ref();
  }

  void prep_for_write() {
    if (body_->refs() > 1) {
      body_->remove_ref();
      body_ = body_->clone();
      body_->add_ref();
    }
  }

  void set_size(size_t r, size_t c) {
    prep_for_write();
    body_->set_size(r,c);
  }

  size_t rows() const { return body_->rows(); }
  size_t cols() const { return body_->cols(); }
  size_t size() const { return body_->size(); }

  const X* data() const { return body_->data(); }
  const X* data_const() const { return body_->data(); }
  X* data() {
    prep_for_write();
    return body_->data();
  }
  X* data_unsafe() { return body_->data(); }

  size_t index(size_t r, size_t c) const { return r*cols()+c; }
  void incr_row(size_t& i) const { i += cols(); }
  void incr_col(size_t& i) const { ++i; }

  X entry(size_t r, size_t c) const { return data()[index(r,c)]; }
  X entry_const(size_t r, size_t c) const { return data()[index(r,c)]; }
  X& entry(size_t r, size_t c) { return data()[index(r,c)]; }
  X& entry_unsafe(size_t r, size_t c) { return data_unsafe()[index(r,c)]; }

private:
  RCMatrixBody<X>* body_;
};

#endif // _INC_RCMATRIX_H

// $Id: RCMatrix.h,v 1.2 1999/03/16 18:29:27 alstrup Exp $
