// This is -*- C++ -*-
// $Id: parametric_estimation.cpp,v 1.4 1999/05/26 20:38:04 trow Exp $

/* 
 * parametric_estimation.cpp
 *
 * Copyright (C) 1999 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@emccta.com>.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <specfns.h>
#include "parametric_estimation.h"

ConfInt
pest_normal_mean(double mean, double sdev, size_t samples,
		       double conf)
{
  double t_val = inv_students_cdf(samples-2, (1+conf)/2);
  double serr = sdev / sqrt(samples);
  ConfInt ci;
  ci.set_by_error(mean, t_val * serr, conf);
  return ci;
}

ConfInt
pest_normal_difference_of_means_equal_sdev(double mean1, double mean2,
					   double sdev,
					   size_t N1, size_t N2,
					   double conf)
{
  double t_val = inv_students_cdf(N1+N2-2, (1+conf)/2);
  double err = sdev * sqrt((1.0/N1+1.0/N2));
  ConfInt ci;
  ci.set_by_error(mean1-mean2, t_val * err, conf);
  return ci;
}

ConfInt
pest_normal_variance(double var, size_t samples, double conf)
{
  if (samples < 2)
    throw Exception("Number of samples must be >= 2");
  if (var < 0)
    throw Exception("Estimated variance must be >= 0");
  double x_lo = inv_chi_square_cdf(samples-1, 1-(1+conf)/2);
  double x_hi = inv_chi_square_cdf(samples-1, (1+conf)/2);
  double est_lo = var*(samples-1)/x_hi;
  double est_hi = var*(samples-1)/x_lo;
  ConfInt ci;
  ci.set(est_lo, var, est_hi, conf);
  return ci;
}

ConfInt
pest_normal_correlation(double r, size_t samples, double conf)
{
  double z_val = inv_normal_cdf(0, 1, (1+conf)/2);

  double fisher_z = 0.5*(log(1+r) - log(1-r));
  double serr = 1/sqrt(samples-3);
  
  double z_lo = fisher_z - z_val * serr;
  double z_hi = fisher_z + z_val * serr;

  double r_lo = (exp(2*z_lo) - 1) / (exp(2*z_lo) + 1);
  double r_hi = (exp(2*z_hi) - 1) / (exp(2*z_hi) + 1);

  ConfInt ci;
  ci.set(r_lo, r, r_hi, conf);
  return ci;
}

ConfInt
pest_binomial_p_normal_approx(size_t sucesses, size_t trials, double conf)
{
  double z_val = inv_normal_cdf(0, 1, (1+conf)/2);
  double p = sucesses/(double)trials;
  double serr = sqrt(p*(1-p)/trials);
  ConfInt ci;
  ci.set_by_error(p, z_val * serr, conf);
  return ci;
}

// This approximation is given on page 272 of Hogg & Craig
ConfInt
pest_binomial_p_normal_approx_alt(size_t sucesses, size_t trials,
				  double conf)
{
  double z_val = inv_normal_cdf(0, 1, (1+conf)/2);
  double p = (sucesses+2.0)/(trials+4.0);
  double err = 
    sqrt((double)sucesses*(trials-sucesses)/trials + 1) / (trials + 4);
  
  ConfInt ci;
  ci.set_by_error(p, z_val * err, conf);
  return ci;
}

// Another interval from Hogg & Craig, p. 273
ConfInt
pest_binomial_p_arcsin_method(size_t sucesses, size_t trials,
			      double conf)
{
  double z_val = inv_normal_cdf(0, 1, (1+conf)/2);
  double sd = sqrt(0.25/trials);
  double aest = asin(sucesses/(double)trials);
  double lo = sin(aest - z_val * sd);
  double hi = sin(aest + z_val * sd);

  ConfInt ci;
  ci.set(lo, sucesses/(double)trials, hi, conf);
  return ci;
}

double*
binomial_success_exact_cdf(size_t trials, double p)
{
  double* cdf = new double[trials+1];

  if (p > 1 || p < 0) throw Exception("p out of range");
  if (p == 0 || p == 1) {
    double init = 0;
    if (p == 0) init = 1;
    for(size_t i=0; i<= trials; ++i)
      cdf[i] = init;
    if (p == 1) cdf[trials] = 1;
    return cdf;
  }

  double logp = log(p);
  double log1mp = log(1-p);

  double run=0;
  for (size_t i=0; i<=trials; ++i) {
    run += exp(log_choose(trials, i) + i*logp + (trials-i)*log1mp);
    cdf[i] = run;
  }

  return cdf;
}

ConfInt
pest_binomial_difference_of_p_normal_approx(size_t suc1, size_t suc2,
					    size_t tri1, size_t tri2,
					    double conf)
{
  double z_val = inv_normal_cdf(0, 1, (1+conf)/2);
  double p1 = suc1/(double)tri1;
  double p2 = suc2/(double)tri2;
  double serr = sqrt(p1*(1-p1)/tri1 + p2*(1-p2)/tri2);

  ConfInt ci;
  ci.set_by_error(p1-p2, z_val * serr);
  return ci;
}



// $Id: parametric_estimation.cpp,v 1.4 1999/05/26 20:38:04 trow Exp $
