/*
Copyright (C) 1995 Free Software Foundation
    written by R.D. Pierce (pierce@math.psu.edu)

This software is free; you can redistribute it and/or modify it under the 
terms of the GNU Library General Public License as published by the Free
Software Foundation; either version 2 of the License, or (at your
option) any later version.  This software is distributed in the hope
that it will be useful, but WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU Library General Public License for more details.
You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free Software
Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// Topological functions 

template<class T,class S,class C>
static C&
swap(const  _OBJECT_ <T,S>& a,C& result,size_t n,size_t m)
{
  if((void*)&a!=(void*)&result) {
    for(size_t j=1;j<=a.size();j++) result(j)=a(j);   // copy array
    result(n)=a(m);
    result(m)=a(n);
  } else {
    C::value_type t=result(n);     // swap values
    result(n)=result(m);
    result(m)=t;
  }
  return result;
}

template<class T,class S,class C> 
static C&
shift(const  _OBJECT_ <T,S>& a,C& result,int n)
{
// Shifts the array by n indeces, filling vacated spots with default values.  
// Whether this is "right" or "left" depends on how you order the indeces.
// Note that C::size_type must be convertable to and from int.
  int l=a.size();
  n%=l;
  if(n==0) {
    if((void*)&a!=(void*)&result) 
      for(int j= _START_ ;j _COMP_ l;j++) result(j)=a(j); // copy
    return result;   // return the identity
  }
  C::value_type t,tp=a( _START_ );    // next and previous elements
  int i,ip=1;                // next and previous indeces
  for(int j=1;j<=l;j++) {    // l swaps
    i=(ip+n)%l;              // next element to swap
    if(i<=0) i+=l;           // reflect back onto the index range
    ip=i+ _START_  -1;
    t=a(ip);               // store current element
    if(ip _COMP_ n) result(ip)=T();   // check if it is to be zeroed.
    else if(ip >= a.size()+n+ _START_ ) result(ip)=T();
    else result(ip)=tp;              // replace with previous element
    tp=t;                    // set up for next swap
    ip=i;
  }
  return result;
}

template<class T,class S,class C> 
static C&
cshift(const  _OBJECT_ <T,S>& a,C& result,int n)
{
// Cyclically shifts the array by n indeces.
  int l=a.size();
  n%=l;
  if(n==0) {
    if((void*)&a!=(void*)&result) 
      for(int j= _START_ ;j _COMP_ l;j++) result(j)=a(j); // copy
    return result;   // return the identity
  }
  C::value_type t,tp=a( _START_ );  // next and previous elements
  int i,ip=1;                // next and previous indeces
  for(int j=1;j<=l;j++) {    // l swaps
    i=(ip+n)%l;              // next element to swap
    if(i<=0) i+=l;           // reflect back onto the index range
    ip=i+ _START_ -1;
    t=a(ip);                 // store current element
    result(ip)=tp;           // replace with previous element
    tp=t;                    // set up for next swap
    ip=i;
  }
  return result;
}

// IO operators

template<class T,class S> 
static ostream& 
operator<<(ostream& s,const  _OBJECT_ <T,S>& a)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) s << a(j) << _edelim_;
  s << _ldelim_;
  return s;
}

template<class T,class S> 
static istream& 
operator>>(istream& s, _OBJECT_ <T,S>& a)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) s >> a(j);
  return s;
}

template<class T,class S,class T2,class S2,class C,class F> 
static C& 
transform(const _OBJECT_ <T,S>& a,const _OBJECT_ <T2,S2>& b,C& result,F f)
{
  for(C::size_type j= _START_ ;j _COMP_ a.size();j++) result(j)=f(a(j),b(j));
  return result;
}

template<class T,class S,class C,class F> 
static C& 
transform(const T& a,const _OBJECT_ <T,S>& b,C& result,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ b.size();j++) result(j)=f(a,b(j));
  return result;
}

template<class T,class S,class C,class F> 
static C& 
transform(const _OBJECT_ <T,S>& a,const T& b,C& result,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) result(j)=f(a(j),b);
  return result;
}

template<class T,class S,class C,class F> 
static C& 
transform(const _OBJECT_ <T,S>& a,C& result,F f)
{
  for(C::size_type j= _START_ ;j _COMP_ a.size();j++) result(j)=f(a(j));
  return result;
}

#ifndef _MEMBER_TEMPLATES_
template<class T,class S,class T2,class S2> 
static _OBJECT_ <T2,S2>& 
assign(const _OBJECT_ <T,S>& a,_OBJECT_ <T2,S2>& result)
// temporary until member templates are implemented
{
  for(_OBJECT_<T,S>::size_type j= _START_ ;j _COMP_ a.size();j++) result(j)=a(j);
  return result;
}
#endif          _MEMBER_TEMPLATES_

template<class T,class S,class T2,class F> 
static T2 
accumulate(const _OBJECT_ <T,S>& a,T2 init,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) init+=f(a(j));
  return init;
}

template<class T,class S,class T2,class S2,class T3,class F> 
static T3 
accumulate(const _OBJECT_ <T,S>& a,const _OBJECT_ <T2,S2>& b,T3 init,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) init+=f(a(j),b(j));
  return init;
}

template<class T,class S,class T2,class F> 
static T2 
accumulate(const T& a,const _OBJECT_ <T,S>& b,T2 init,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ b.size();j++) init+=f(a,b(j));
  return init;
}

template<class T,class S,class T2,class F> 
static T2 
accumulate(const _OBJECT_ <T,S>& a,const T& b,T2 init,F f)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) init+=f(a(j),b);
  return init;
}

template<class T,class S> 
static T 
max(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  T m=a(j);
  for(j++;j _COMP_ a.size();j++) {
    if(m<a(j)) m=a(j);
  }
  return m;
}

template<class T,class S> 
static long double 
sup(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  long double m=abs(a(j)),t;
  for(j++;j _COMP_ a.size();j++) {
    t=abs(a(j));
    if(m<t) m=t;
  }
  return m;
}

template<class T,class S>
static long double
norm(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  long double s=norm(a(j)),t;
  for(j++;j _COMP_ a.size();j++) s+=norm(a(j));
  return s;
}

template<class T,class S> 
static T 
min(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  T m=a(j);
  for(j++;j _COMP_ a.size();j++) {
    if(a(j)<m) m=a(j);
  }
  return m;
}

template<class T,class S> 
static long double 
inf(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  long double m=abs(a(j)),t;
  for(j++;j _COMP_ a.size();j++) {
    t=abs(a(j));
    if(t<m) m=t;
  }
  return m;
}

template<class T,class S> 
static T 
sum(const  _OBJECT_ <T,S>& a)
{
  S::size_type j= _START_ ;
  T t=a(j);
  for(j++;j _COMP_ a.size();j++) t+=a(j);
  return t;
}

template<class T,class S> 
static T 
product(const  _OBJECT_ <T,S>& a) 
{
  S::size_type j= _START_ ;
  T t=a(j);
  for(j++;j _COMP_ a.size();j++) t*=a(j);
  return t;
}


template<class T,class S,class T2,class S2> 
static bool 
operator==(const  _OBJECT_ <T,S>& a,const  _OBJECT_ <T2,S2>& b)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) if(a(j)!=b(j)) return 0;
  return 1;
}

/* template<class T,class S,class T2,class S2> 
static bool 
operator!=(const  _OBJECT_ <T,S>& a,const  _OBJECT_ <T2,S2>& b)
{
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) if(a(j)!=b(j)) return 1;
  return 0;
} */


// Define the delayed execution classes
#ifndef _Delay_classes_
#define _Delay_classes_

template<class C,class F>
class unary_allocator
{
public:
  typedef C::value_type         value_type;
  typedef Delay_unary<C,F> 	pointer;
  typedef C::size_type		size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class C,class F>
class Delay_unary
{
public:
  typedef C::value_type		value_type;
  typedef value_type&		reference;
  typedef const value_type&	const_reference;
  typedef C::size_type 		size_type;
  typedef C::pointer 		c_type;

  Delay_unary() {}
  Delay_unary(size_type n,const_reference t=value_type()) {}
  Delay_unary(const Delay_unary &d) : a(d.a), f(d.f) {}
  Delay_unary(const c_type& g,F fnc) : a(g), f(fnc) {}
  
  Delay_unary& operator=(const Delay_unary& d) { a=d.a; f=d.f; return (*this); }
  Delay_unary& operator++() { return (*this); }
  Delay_unary& operator--() { return (*this); }
  Delay_unary& operator+(size_type) { return (*this); }
  Delay_unary& operator-(size_type) { return (*this); }
  reference operator[](size_type n) { return z=f(a[n]); }
  value_type operator[](size_type n) const { return f(a[n]); }
private:
  c_type a;
  F f;
  value_type z;
};

template<class C1,class C2,class F>
class binary_allocator
{
public:
  typedef C1::value_type	value_type;
  typedef Delay_binary<C1,C2,F> pointer;
  typedef C1::size_type		size_type;
  pointer allocate(size_type n) { pointer p(n); return p; }
  void deallocate(pointer p) { p.~pointer(); }
};

template<class C1,class C2,class F>
class Delay_binary
{
public:
  typedef C1::value_type	value_type;
  typedef value_type&		reference;
  typedef const value_type&	const_reference;
  typedef C1::size_type		size_type;
  typedef C1::pointer 		c1_type;
  typedef C2::pointer 		c2_type;

  Delay_binary() {}
  Delay_binary(size_type n,const_reference t=value_type()) {}
  Delay_binary(const Delay_binary &d) : a1(d.a1), a2(d.a2), f(d.f) {}
  Delay_binary(const c1_type& g1,const c2_type& g2,F fnc) : a1(g1), a2(g2), f(fnc) {}

  Delay_binary& operator=(const Delay_binary& d) { a1=d.a1; a2=d.a2; f=d.f; return (*this); }
  Delay_binary& operator++() { return (*this); }
  Delay_binary& operator--() { return (*this); }
  Delay_binary& operator+(size_type) { return (*this); }
  Delay_binary& operator-(size_type) { return (*this); }
  reference operator[](size_type n) { return z=f(a1[n],a2[n]); }
  value_type operator[](size_type n) const { return f(a1[n],a2[n]); }
private:
  c1_type a1;
  c2_type a2;
  F f;
  value_type z;
};

#endif          _Delay_classes_

// Note: copying a Delay_unary or Delay_binary is cheap, so named return
// values are not really necessary.

template<class T,class S,class F> 
static  _OBJECT_ <T, unary_allocator<S,F>  > 
apply(const  _OBJECT_ <T,S>& a,F f)
{
  _OBJECT_ <T,unary_allocator<S,F> > r;
  r.resize(a.size());
  r.container()= Delay_unary<S,F>(a.container(),f);
  return r;
}

template<class T,class S,class F> 
static  _OBJECT_ <T, binary_allocator<scalar_allocator<T>,S,F> > 
apply(const T& b,const  _OBJECT_ <T,S>& a,F f)
{
  _OBJECT_ <T,binary_allocator<scalar_allocator<T>,S,F> > r;
  r.resize(a.size());
  r.container()= Delay_binary<scalar_allocator<T>,S,F>(
	scalar_pointer<T>(a.size(),b),a.container(),f);
  return r;
}

template<class T,class S,class F> 
static  _OBJECT_ <T, binary_allocator<S,scalar_allocator<T>,F> > 
apply(const  _OBJECT_ <T,S>& a,const T& b,F f)
{
  _OBJECT_ <T,binary_allocator<S,scalar_allocator<T>,F> > r;
  r.resize(a.size());
  r.container()= Delay_binary<S,scalar_allocator<T>,F>(
	a.container(),scalar_pointer<T>(a.size(),b),f);
  return r;
}

template<class T,class S,class T2,class S2,class F> 
static  _OBJECT_ <T, binary_allocator<S,S2,F> > 
apply(const  _OBJECT_ <T,S>& a,const  _OBJECT_ <T2,S2>& b,F f)
{
  _OBJECT_ <T,binary_allocator<S,S2,F> > r;
  r.resize(a.size());
  r.container()= Delay_binary<S,S2,F>(a.container(), b.container(),f);
  return r;
}


// MACRO DEFINITIONS

#ifndef _symbol_classes_
#define _symbol_classes_

#define binary_class(_SYMBOL_,_SMBL_)					\
									\
template<class C1,class C2>						\
class _SMBL_ ## allocator						\
{									\
public:									\
  typedef C1::value_type        value_type;				\
  typedef Delay_ ## _SMBL_<C1,C2> pointer;				\
  typedef C1::size_type		size_type;				\
  pointer allocate(size_type n) { pointer p(n); return p; }		\
  void deallocate(pointer p) { p.~pointer(); }				\
};									\
									\
template<class C1,class C2>						\
class Delay_ ## _SMBL_							\
{									\
public:									\
  typedef C1::value_type	value_type;				\
  typedef value_type&		reference;				\
  typedef const value_type&	const_reference;			\
  typedef C1::size_type		size_type;				\
  typedef C1::pointer           c1_type;				\
  typedef C2::pointer           c2_type;				\
									\
  Delay_ ## _SMBL_() {}							\
  Delay_ ## _SMBL_(size_type n,const_reference t=value_type()) {}	\
  Delay_ ## _SMBL_(const Delay_ ## _SMBL_ & d) : a1(d.a1), a2(d.a2) {}	\
  Delay_ ## _SMBL_(const c1_type& g1,const c2_type& g2) : a1(g1), a2(g2) {}\
									\
  Delay_ ## _SMBL_& operator=(const Delay_ ## _SMBL_& d) { 		\
    a1=d.a1; a2=d.a2; return (*this); }					\
  Delay_ ## _SMBL_& operator++() { return (*this); }			\
  Delay_ ## _SMBL_& operator--() { return (*this); }			\
  Delay_ ## _SMBL_& operator+(size_type) { return (*this); }		\
  Delay_ ## _SMBL_& operator-(size_type) { return (*this); }		\
  reference operator[](size_type n) { return z = a1[n] _SYMBOL_ a2[n]; }\
  value_type operator[](size_type n) const { 				\
    return a1[n] _SYMBOL_ a2[n]; }					\
private:								\
  c1_type a1;								\
  c2_type a2;								\
  value_type z;								\
};

#define unary_class(_SYMBOL_,_SMBL_)					\
									\
template<class C>							\
class _SMBL_ ## allocator						\
{									\
public:									\
  typedef C::value_type         value_type;				\
  typedef Delay_ ## _SMBL_<C> 	pointer;				\
  typedef C::size_type          size_type;				\
  pointer allocate(size_type n) { pointer p(n); return p; }		\
  void deallocate(pointer p) { p.~pointer(); }				\
};									\
									\
template<class C>							\
class Delay_ ## _SMBL_							\
{									\
public:									\
  typedef C::value_type		value_type;				\
  typedef value_type&		reference;				\
  typedef const value_type&	const_reference;			\
  typedef C::size_type 		size_type;				\
  typedef C::pointer		c_type;					\
									\
  Delay_ ## _SMBL_() {}							\
  Delay_ ## _SMBL_(size_type n,const_reference t=value_type()) {}	\
  Delay_ ## _SMBL_(const Delay_ ## _SMBL_ & d) : a(d.a) {}		\
  Delay_ ## _SMBL_(const c_type& g) : a(g) {}				\
  									\
  Delay_ ## _SMBL_& operator=(const Delay_ ## _SMBL_& d) { 		\
    a=d.a; return (*this); }						\
  Delay_ ## _SMBL_& operator+(size_type) { return (*this); }		\
  Delay_ ## _SMBL_& operator-(size_type) { return (*this); }		\
  Delay_ ## _SMBL_& operator++() { return (*this); }			\
  Delay_ ## _SMBL_& operator--() { return (*this); }			\
  reference operator[](size_type n) { return z = _SYMBOL_ a[n]; }	\
  value_type operator[](size_type n) const { return _SYMBOL_ a[n]; }	\
private:								\
  c_type a;								\
  value_type z;								\
};

binary_class(+,pl_)
binary_class(-,mi_)
binary_class(*,mu_)
binary_class(/,di_)
binary_class(%,pe_)
binary_class(&,an_)
binary_class(|,or_)
binary_class(<<,ll_)
binary_class(>>,gg_)
#undef binary_class

unary_class(+,u_pl_)
unary_class(-,u_mi_)
unary_class(~,u_ti_)
unary_class(!,u_ba_)
#undef unary_class
#endif                 // end of symbol classes

#define binary_function(_SYMBOL_,_SMBL_)				\
									\
template<class T,class S> 						\
static  _OBJECT_ <T,S>& 						\
operator _SYMBOL_ ## =( _OBJECT_ <T,S>& a,T b)				\
{									\
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) 			\
    a(j) _SYMBOL_ ## = b;						\
  return a;								\
}									\
									\
template<class T,class S,class T2,class S2> 				\
static  _OBJECT_ <T,S>& 						\
operator _SYMBOL_ ## =( _OBJECT_ <T,S>& a,const  _OBJECT_ <T2,S2>& b)	\
{									\
  for(S::size_type j= _START_ ;j _COMP_ a.size();j++) 			\
    a(j) _SYMBOL_ ## = b(j);						\
  return a;								\
}									\
									\
template<class T,class S> 						\
static  _OBJECT_ < T, _SMBL_ ## allocator<S,scalar_allocator<T> > >	\
operator _SYMBOL_(const  _OBJECT_ <T,S>& a,const T& b)			\
{									\
  _OBJECT_ <T,_SMBL_ ## allocator<S,scalar_allocator<T> > > r;		\
  r.resize(a.size());							\
  r.container()= Delay_ ## _SMBL_<S,scalar_allocator<T> >(		\
	a.container(),scalar_pointer<T>(a.size(),b));			\
  return r;								\
}									\
									\
template<class T,class S> 						\
static  _OBJECT_ <T, _SMBL_ ## allocator<scalar_allocator<T>,S> > 	\
operator _SYMBOL_(const T& b,const  _OBJECT_ <T,S>& a)			\
{									\
  _OBJECT_ <T,_SMBL_ ## allocator<scalar_allocator<T>,S> > r;		\
  r.resize(a.size());							\
  r.container()= Delay_ ## _SMBL_<scalar_allocator<T>,S>(		\
	scalar_pointer<T>(a.size(),b),a.container());			\
  return r;								\
}									\
									\
template<class T,class S,class T2,class S2>				\
static  _OBJECT_ <T, _SMBL_ ## allocator<S,S2> >			\
operator _SYMBOL_(const  _OBJECT_ <T,S>& a,const  _OBJECT_ <T2,S2>& b)	\
{									\
  _OBJECT_ <T,_SMBL_ ## allocator<S,S2> > r;				\
  r.resize(a.size());							\
  r.container()= Delay_ ## _SMBL_<S,S2>(a.container(),b.container());	\
  return r;								\
}

#define unary_function(_SYMBOL_,_SMBL_)					\
									\
template<class T,class S> 						\
static  _OBJECT_ < T, _SMBL_ ## allocator<S> >				\
operator _SYMBOL_(const  _OBJECT_ <T,S>& a) 				\
{									\
  _OBJECT_ <T,_SMBL_ ## allocator<S> > r;				\
  r.resize(a.size());							\
  r.container()= Delay_ ## _SMBL_<S> (a.container());			\
  return r;								\
}

binary_function(+,pl_)
binary_function(-,mi_)
binary_function(*,mu_)
binary_function(/,di_)
binary_function(%,pe_)
binary_function(&,an_)
binary_function(|,or_)
binary_function(<<,ll_)
binary_function(>>,gg_)
#undef binary_function

unary_function(+,u_pl_)
unary_function(-,u_mi_)
unary_function(~,u_ti_)
unary_function(!,u_ba_)
#undef unary_function

