/* glplpx7.c (simplex table routines) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <float.h>
#include "glpspx.h"

/*----------------------------------------------------------------------
-- lpx_eval_tab_row - compute row of the simplex table.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_eval_tab_row(LPX *lp, int k, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_eval_tab_row computes a row of the current simplex
-- table for the basic variable, which is specified by the number k:
-- if 1 <= k <= m, x[k] is k-th auxiliary variable; if m+1 <= k <= m+n,
-- x[k] is (k-m)-th structural variable, where m is number of rows, and
-- n is number of columns. The current basis must be valid.
--
-- The routine stores column indices and numerical values of non-zero
-- elements of the computed row using sparse format to the locations
-- ndx[1], ..., ndx[len] and val[1], ..., val[len] respectively, where
-- 0 <= len <= n is number of non-zeros returned on exit.
--
-- Element indices stored in the array ndx have the same sense as the
-- index k, i.e. indices 1 to m denote auxiliary variables and indices
-- m+1 to m+n denote structural ones (all these variables are obviously
-- non-basic by the definition).
--
-- The computed row shows how the specified basic variable x[k] = xB[i]
-- depends on non-basic variables:
--
--    xB[i] = alfa[i,1]*xN[1] + alfa[i,2]*xN[2] + ... + alfa[i,n]*xN[n],
--
-- where alfa[i,j] are elements of the simplex table row, xN[j] are
-- non-basic (auxiliary and structural) variables.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- specified simplex table row as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns number of non-zero elements in the simplex table
-- row stored in the arrays ndx and val.
--
-- *Unscaling*
--
-- Let A~ = (I | -A) is the augmented constraint matrix of the original
-- (unscaled) problem. In the scaled LP problem instead the matrix A the
-- scaled matrix A" = R*A*S is actually used, so
--
--    A~" = (I | A") = (I | R*A*S) = (R*I*inv(R) | R*A*S) =
--                                                                   (1)
--        = R*(I | A)*S~ = R*A~*S~,
--
-- is the scaled augmented constraint matrix, where R and S are diagonal
-- scaling matrices used to scale rows and columns of the matrix A, and
--
--    S~ = diag(inv(R) | S)                                          (2)
--
-- is an augmented diagonal scaling matrix.
--
-- By the definition the simplex table is the matrix
--
--    T = - inv(B) * N,                                              (3)
--
-- where B is the basis matrix that consists of basic columns of the
-- augmented constraint matrix A~, and N is a matrix that consists of
-- non-basic columns of A~. From (1) it follows that
--
--    A~" = (B" | N") = (R*B*SB | R*N*SN),                           (4)
--
-- where SB and SN are parts of the augmented scaling matrix S~ that
-- correspond to basic and non-basic variables respectively. Thus, in
-- the scaled case
--
--    T" = - inv(B") * N" = - inv(R*B*SB) * (R*N*SN) =
--
--       = - inv(SB)*inv(B)*inv(R) * (R*N*SN) =                      (5)
--
--       = inv(SB) * (-inv(B) * N) * SN = inv(SB) * T * SN,
--
-- that allows us expressing the original simplex table T through the
-- scaled simplex table T":
--
--    T = SB * T" * inv(SN).                                         (6)
--
-- The formula (6) is used by the routine for unscaling elements of the
-- computed simplex table row. */

int lpx_eval_tab_row(LPX *lp, int k, int ndx[], double val[])
{     int m = lp->m, n = lp->n;
      int i, j, len;
      double *rho, *row, sb_i, sn_j;
      if (!(1 <= k && k <= m+n))
         fault("lpx_eval_tab_row: k = %d; variable number out of range",
            k);
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_eval_tab_row: current basis is undefined");
      if (lp->tagx[k] != LPX_BS)
         fault("lpx_eval_tab_row: k = %d; variable should be basic", k);
      i = lp->posx[k]; /* x[k] = xB[i] */
      insist(1 <= i && i <= m);
      /* allocate working arrays */
      rho = ucalloc(1+m, sizeof(double));
      row = ucalloc(1+n, sizeof(double));
      /* compute i-th row of the inverse inv(B) */
      spx_eval_rho(lp, i, rho);
      /* compute i-th row of the simplex table */
      spx_eval_row(lp, rho, row);
      /* unscale and store the required row */
      sb_i = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
      len = 0;
      for (j = 1; j <= n; j++)
      {  if (row[j] != 0.0)
         {  k = lp->indx[m+j]; /* x[k] = xN[j] */
            sn_j = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
            len++;
            ndx[len] = k;
            val[len] = (sb_i / sn_j) * row[j];
         }
      }
      /* free working arrays */
      ufree(rho);
      ufree(row);
      /* return to the calling program */
      return len;
}

/*----------------------------------------------------------------------
-- lpx_eval_tab_col - compute column of the simplex table.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_eval_tab_col(LPX *lp, int k, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_eval_tab_col computes a column of the current simplex
-- table for the non-basic variable, which is specified by the number k:
-- if 1 <= k <= m, x[k] is k-th auxiliary variable; if m+1 <= k <= m+n,
-- x[k] is (k-m)-th structural variable, where m is number of rows, and
-- n is number of columns. The current basis must be valid.
--
-- The routine stores row indices and numerical values of non-zero
-- elements of the computed column using sparse format to the locations
-- ndx[1], ..., ndx[len] and val[1], ..., val[len] respectively, where
-- 0 <= len <= m is number of non-zeros returned on exit.
--
-- Element indices stored in the array ndx have the same sense as the
-- index k, i.e. indices 1 to m denote auxiliary variables and indices
-- m+1 to m+n denote structural ones (all these variables are obviously
-- basic by the definition).
--
-- The computed column shows how basic variables depend on the specified
-- non-basic variable x[k] = xN[j]:
--
--    xB[1] = ... + alfa[1,j]*xN[j] + ...
--    xB[2] = ... + alfa[2,j]*xN[j] + ...
--             . . . . . .
--    xB[m] = ... + alfa[m,j]*xN[j] + ...
--
-- where alfa[i,j] are elements of the simplex table column, xB[i] are
-- basic (auxiliary and structural) variables.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- specified simplex table column as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns number of non-zero elements in the simplex table
-- column stored in the arrays ndx and val.
--
-- *Unscaling*
--
-- The routine unscales elements of the computed simplex table column
-- using the formula (6), which is derived and explained in description
-- of the routine lpx_eval_tab_row (see above). */

int lpx_eval_tab_col(LPX *lp, int k, int ndx[], double val[])
{     int m = lp->m, n = lp->n;
      int i, j, len;
      double *col, sb_i, sn_j;
      if (!(1 <= k && k <= m+n))
         fault("lpx_eval_tab_col: k = %d; variable number out of range",
            k);
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_eval_tab_col: current basis is undefined");
      if (lp->tagx[k] == LPX_BS)
         fault("lpx_eval_tab_col; k = %d; variable should be non-basic",
            k);
      j = lp->posx[k] - m; /* x[k] = xN[j] */
      insist(1 <= j && j <= n);
      /* allocate working array */
      col = ucalloc(1+m, sizeof(double));
      /* compute j-th column of the simplex table */
      spx_eval_col(lp, j, col, 0);
      /* unscale and store the required column */
      sn_j = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
      len = 0;
      for (i = 1; i <= m; i++)
      {  if (col[i] != 0.0)
         {  k = lp->indx[i]; /* x[k] = xB[i] */
            sb_i = (k <= m ? 1.0 / lp->rs[k] : lp->rs[k]);
            len++;
            ndx[len] = k;
            val[len] = (sb_i / sn_j) * col[i];
         }
      }
      /* free working array */
      ufree(col);
      /* return to the calling program */
      return len;
}

/*----------------------------------------------------------------------
-- lpx_transform_row - transform explicitly specified row.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_transform_row(LPX *lp, int len, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_transform_row performs the same operation as the
-- routine lpx_eval_tab_row, except that the transformed row should be
-- specified explicitly.
--
-- The explicitly specified row may be thought as a linear form
--
--    x = a[1]*x[m+1] + a[2]*x[m+2] + ... + a[n]*x[m+n],             (1)
--
-- where x is an auxiliary variable for this row, a[j] are coefficients
-- of the linear form, x[m+j] are structural variables.
--
-- On entry column indices and numerical values of non-zero elements of
-- the transformed row should be in locations ndx[1], ..., ndx[len] and
-- val[1], ..., val[len], where len is number of non-zero elements.
--
-- This routine uses the system of equality constraints and the current
-- basis in order to express the auxiliary variable x in (1) through the
-- current non-basic variables (as if the transformed row were added to
-- the problem object and the auxiliary variable were basic), i.e. the
-- resultant row has the form:
--
--    x = alfa[1]*xN[1] + alfa[2]*xN[2] + ... + alfa[n]*xN[n],       (2)
--
-- where xN[j] are non-basic (auxiliary or structural) variables, n is
-- number of columns in the specified problem object.
--
-- On exit the routine stores indices and numerical values of non-zero
-- elements of the resultant row (2) in locations ndx[1], ..., ndx[len']
-- and val[1], ..., val[len'], where 0 <= len' <= n is count of non-zero
-- elements in the resultant row returned by the routine. Note that
-- indices of non-basic variables stored in the array ndx correspond to
-- original ordinal numbers of variables: indices 1 to m mean auxiliary
-- variables and indices m+1 to m+n mean structural ones.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- resultant row as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns len' that is number of non-zero elements in the
-- resultant row stored in the arrays ndx and val.
--
-- *Algorithm*
--
-- The explicitly specified row (1) is transformed in the same way as
-- it were the objective function row. Rewrite (1) as
--
--    x = aB' * xB + aN' * xN,                                       (3)
--
-- where aB is subvector of a at basic variables xB and aN is subvector
-- of a at non-basic variables xN. The simplex table that correspond to
-- the current basis has the form
--
--    xB = [-inv(B) * N] * xN.                                       (4)
--
-- Therefore substituting xB from (4) to (3) we have
--
--    x = aB' * [-inv(B) * N] * xN + aN' * xN =
--                                                                   (5)
--      = (aN' - aB' * inv(B) * N) * xN = alfa' * xN,
--
-- where
--
--    alfa = aN - N' * inv(B') * aB                                  (6)
--
-- is the resultant row computed by the routine. */

int lpx_transform_row(LPX *lp, int len, int ndx[], double val[])
{     int m = lp->m;
      int n = lp->n;
      double *rs = lp->rs;
      int *A_ptr = lp->A->ptr;
      int *A_len = lp->A->len;
      int *A_ndx = lp->A->ndx;
      double *A_val = lp->A->val;
      int *tagx = lp->tagx;
      int *posx = lp->posx;
      int *indx = lp->indx;
      int i, j, k, t, beg, end, ptr;
      double *v, *alfa;
      /* check the explicitly specified row */
      if (!(0 <= len && len <= n))
         fault("lpx_transform_row: len = %d; invalid row length", len);
      for (t = 1; t <= len; t++)
      {  j = ndx[t];
         if (!(1 <= j && j <= n))
            fault("lpx_transform_row: ndx[%d] = %d; column number out o"
               "f range", t, j);
      }
      /* the current basis should be valid */
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_transform_row: current basis is undefined");
      /* v := aB (scaled) */
      v = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) v[i] = 0.0;
      for (t = 1; t <= len; t++)
      {  j = ndx[t];
         if (tagx[m+j] == LPX_BS)
         {  i = posx[m+j]; /* xB[i] = xS[j] */
            v[i] += val[t] * rs[m+j];
         }
      }
      /* v := inv(B') * aB (scaled) */
      spx_btran(lp, v);
      /* alfa := aN (scaled) */
      alfa = ucalloc(1+n, sizeof(double));
      for (j = 1; j <= n; j++) alfa[j] = 0.0;
      for (t = 1; t <= len; t++)
      {  j = ndx[t];
         if (tagx[m+j] != LPX_BS)
         {  k = posx[m+j]; /* xN[k] = xS[j] */
            alfa[k-m] = val[t] * rs[m+j];
         }
      }
      /* alfa := alfa - N' * inv(B') * aB (scaled) */
      for (j = 1; j <= n; j++)
      {  k = indx[m+j]; /* x[k] = xN[j] */
         if (k <= m)
         {  /* x[k] is auxiliary variable */
            alfa[j] -= v[k];
         }
         else
         {  /* x[k] is structural variable */
            beg = A_ptr[k];
            end = beg + A_len[k] - 1;
            for (ptr = beg; ptr <= end; ptr++)
               alfa[j] += v[A_ndx[ptr]] * A_val[ptr];
         }
      }
      /* unscale alfa and store it in sparse format */
      len = 0;
      for (j = 1; j <= n; j++)
      {  if (alfa[j] != 0.0)
         {  k = indx[m+j]; /* x[k] = xN[j] */
            len++;
            ndx[len] = k;
            if (k <= m)
            {  /* x[k] is auxiliary variable */
               val[len] = alfa[j] * rs[k];
            }
            else
            {  /* x[k] is structural variable */
               val[len] = alfa[j] / rs[k];
            }
         }
      }
      /* free working arrays and return */
      ufree(v);
      ufree(alfa);
      return len;
}

/*----------------------------------------------------------------------
-- lpx_transform_col - transform explicitly specified column.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_transform_col(LPX *lp, int len, int ndx[], double val[]);
--
-- *Description*
--
-- The routine lpx_transform_col performs the same operation as the
-- routine lpx_eval_tab_col, except that the transformed column should
-- be specified explicitly.
--
-- The explicitly specified column may be thought as it were added to
-- the original system of equality constraints:
--
--    x[1] = a[1,1]*x[m+1] + ... + a[1,n]*x[m+n] + a[1]*x
--    x[2] = a[2,1]*x[m+1] + ... + a[2,n]*x[m+n] + a[2]*x            (1)
--       .  .  .  .  .  .  .  .  .  .  .  .  .  .  .
--    x[m] = a[m,1]*x[m+1] + ... + a[m,n]*x[m+n] + a[m]*x
--
-- where x[i] are auxiliary variables, x[m+j] are structural variables,
-- x is a structural variable for the explicitly specified column, a[i]
-- are constraint coefficients for x.
--
-- On entry row indices and numerical values of non-zero elements of
-- the transformed column should be in locations ndx[1], ..., ndx[len]
-- and val[1], ..., val[len], where len is number of non-zero elements.
--
-- This routine uses the system of equality constraints and the current
-- basis in order to express the current basic variables through the
-- structural variable x in (1) (as if the transformed column were added
-- to the problem object and the variable x were non-basic):
--
--    xB[1] = ... + alfa[1]*x
--    xB[2] = ... + alfa[2]*x                                        (2)
--       .  .  .  .  .  .
--    xB[m] = ... + alfa[m]*x
--
-- where xB are basic (auxiliary and structural) variables, m is number
-- of rows in the specified problem object.
--
-- On exit the routine stores indices and numerical values of non-zero
-- elements of the resultant column alfa (2) in locations ndx[1], ...,
-- ndx[len'] and val[1], ..., val[len'], where 0 <= len' <= m is count
-- of non-zero element in the resultant column returned by the routine.
-- Note that indices of basic variables stored in the array ndx
-- correspond to original ordinal numbers of variables; indices 1 to m
-- mean auxiliary variables, indices m+1 to m+n mean structural ones.
--
-- Even if the problem is (internally) scaled, the routine returns the
-- resultant column as if the problem were unscaled.
--
-- *Returns*
--
-- The routine returns len' that is number of non-zero elements in the
-- resultant column stored in the arrays ndx and val.
--
-- *Algorithm*
--
-- The explicitly specified column (1) is transformed in the same way
-- as any other column of the constraint matrix using the formula:
--
--    alfa = inv(B) * a,
--
-- where alfa is the resultant column. */

int lpx_transform_col(LPX *lp, int len, int ndx[], double val[])
{     int m = lp->m;
      double *rs = lp->rs;
      int *indx = lp->indx;
      int i, t, k;
      double *alfa;
      /* check the explicitly specified column */
      if (!(0 <= len && len <= m))
         fault("lpx_transform_col: len = %d; invalid column length",
            len);
      for (t = 1; t <= len; t++)
      {  i = ndx[t];
         if (!(1 <= i && i <= m))
            fault("lpx_transform_col: ndx[%d] = %d; row number out of r"
               "ange", t, i);
      }
      /* the current basis should be valid */
      if (lp->b_stat != LPX_B_VALID)
         fault("lpx_transform_col: current basis is undefined");
      /* alfa := a (scaled) */
      alfa = ucalloc(1+m, sizeof(double));
      for (i = 1; i <= m; i++) alfa[i] = 0.0;
      for (t = 1; t <= len; t++)
      {  i = ndx[t];
         alfa[i] += val[t] * rs[i];
      }
      /* alfa := inv(B) * a */
      spx_ftran(lp, alfa, 0);
      /* unscale alfa and store it in sparse format */
      len = 0;
      for (i = 1; i <= m; i++)
      {  if (alfa[i] != 0.0)
         {  k = indx[i]; /* x[k] = xB[i] */
            len++;
            ndx[len] = k;
            if (k <= m)
            {  /* x[k] is auxiliary variable */
               val[len] = alfa[i] / rs[k];
            }
            else
            {  /* x[k] is structural variable */
               val[len] = alfa[i] * rs[k];
            }
         }
      }
      /* free working array and return */
      ufree(alfa);
      return len;
}

/*----------------------------------------------------------------------
-- lpx_prim_ratio_test - perform primal ratio test.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_prim_ratio_test(LPX *lp, int len, int ndx[], double val[],
--    int how, double tol);
--
-- *Description*
--
-- The routine lpx_prim_ratio_test performs the primal ratio test for
-- an explicitly specified column of the simplex table.
--
-- The primal basic solution associated with an LP problem object,
-- which the parameter lp points to, should be feasible. No components
-- of the LP problem object are changed by the routine.
--
-- The explicitly specified column of the simplex table shows how the
-- basic variables xB depend on some non-basic variable y (which is not
-- necessarily presented in the problem object):
--
--    xB[1] = ... + alfa[1]*y + ...
--    xB[2] = ... + alfa[2]*y + ...                                  (*)
--       .  .  .  .  .  .  .  .
--    xB[m] = ... + alfa[m]*y + ...
--
-- The column (*) is specifed on entry to the routine using the sparse
-- format. Ordinal numbers of basic variables xB[i] should be placed in
-- locations ndx[1], ..., ndx[len], where ordinal number 1 to m denote
-- auxiliary variables, and ordinal numbers m+1 to m+n denote structural
-- variables. The corresponding non-zero coefficients alfa[i] should be
-- placed in locations val[1], ..., val[len]. The arrays ndx and val are
-- not changed on exit.
--
-- The parameter how specifies in which direction the variable y changes
-- on entering the basis: +1 means increasing, -1 means decreasing.
--
-- The parameter tol is a relative tolerance (small positive number)
-- used by the routine to skip small alfa[i] of the column (*).
--
-- The routine determines the ordinal number of some basic variable
-- (specified in ndx[1], ..., ndx[len]), which should leave the basis
-- instead the variable y in order to keep primal feasibility, and
-- returns it on exit. If the choice cannot be made (i.e. if the
-- adjacent basic solution is primal unbounded), the routine returns
-- zero.
--
-- *Note*
--
-- If the non-basic variable y is presented in the LP problem object,
-- the column (*) can be computed using the routine lpx_eval_tab_col.
-- Otherwise it can be computed using the routine lpx_transform_col.
--
-- *Returns*
--
-- The routine lpx_prim_ratio_test returns the ordinal number of some
-- basic variable xB[i], which should leave the basis instead the
-- variable y in order to keep primal feasibility. If the adjacent basic
-- solution is primal unbounded and therefore the choice cannot be made,
-- the routine returns zero. */

int lpx_prim_ratio_test(LPX *lp, int len, int ndx[], double val[],
      int how, double tol)
{     int m = lp->m;
      int n = lp->n;
      int *typx = lp->typx;
      double *lb = lp->lb;
      double *ub = lp->ub;
      double *rs = lp->rs;
      int *tagx = lp->tagx;
      int *posx = lp->posx;
      double *bbar = lp->bbar;
      int i, k, p, t;
      double alfa_i, abs_alfa_i, big, eps, bbar_i, lb_i, ub_i, temp,
         teta;
      /* the current basic solution should be primal feasible */
      if (lp->p_stat != LPX_P_FEAS)
         fault("lpx_prim_ratio_test: current basic solution is not prim"
            "al feasible");
      /* check if the parameter how is correct */
      if (!(how == +1 || how == -1))
         fault("lpx_prim_ratio_test: how = %d; invalid parameter", how);
      /* compute the largest absolute value of the specified influence
         coefficients */
      big = 0.0;
      for (t = 1; t <= len; t++)
      {  temp = val[t];
         if (temp < 0.0) temp = - temp;
         if (big < temp) big = temp;
      }
      /* compute the absolute tolerance eps used to skip small entries
         of the column */
      if (!(0.0 < tol && tol < 1.0))
         fault("lpx_prim_ratio_test: tol = %g; invalid tolerance", tol);
      eps = tol * (1.0 + big);
      /* initial settings */
      p = 0, teta = DBL_MAX, big = 0.0;
      /* walk through the entries of the specified column */
      for (t = 1; t <= len; t++)
      {  /* get the ordinal number of basic variable */
         k = ndx[t];
         if (!(1 <= k && k <= m+n))
            fault("lpx_prim_ratio_test: ndx[%d] = %d; ordinal number ou"
               "t of range", t, k);
         if (tagx[k] != LPX_BS)
            fault("lpx_prim_ratio_test: ndx[%d] = %d; non-basic variabl"
               "e not allowed", t, k);
         /* determine index of the variable x[k] in the vector xB */
         i = posx[k]; /* x[k] = xB[i] */
         insist(1 <= i && i <= m);
         /* determine unscaled bounds and value of the basic variable
            xB[i] in the current basic solution */
         if (k <= m)
         {  lb_i = lb[k] / rs[k];
            ub_i = ub[k] / rs[k];
            bbar_i = bbar[i] / rs[k];
         }
         else
         {  lb_i = lb[k] * rs[k];
            ub_i = ub[k] * rs[k];
            bbar_i = bbar[i] * rs[k];
         }
         /* determine influence coefficient for the basic variable
            x[k] = xB[i] in the explicitly specified column and turn to
            the case of increasing the variable y in order to simplify
            program logic */
         alfa_i = (how > 0 ? +val[t] : -val[t]);
         abs_alfa_i = (alfa_i > 0.0 ? +alfa_i : -alfa_i);
         /* analyze main cases */
         switch (typx[k])
         {  case LPX_FR:
               /* xB[i] is free variable */
               continue;
            case LPX_LO:
lo:            /* xB[i] has an lower bound */
               if (alfa_i > -eps) continue;
               temp = (lb_i - bbar_i) / alfa_i;
               break;
            case LPX_UP:
up:            /* xB[i] has an upper bound */
               if (alfa_i < +eps) continue;
               temp = (ub_i - bbar_i) / alfa_i;
               break;
            case LPX_DB:
               /* xB[i] has both lower and upper bounds */
               if (alfa_i < 0.0) goto lo; else goto up;
            case LPX_FX:
               /* xB[i] is fixed variable */
               if (abs_alfa_i < eps) continue;
               temp = 0.0;
               break;
            default:
               insist(typx != typx);
         }
         /* if the value of the variable xB[i] violates its lower or
            upper bound (slightly, because the current basis is assumed
            to be primal feasible), temp is negative; we can think this
            happens due to round-off errors and the value is exactly on
            the bound; this allows replacing temp by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the minimal ratio test */
         if (teta > temp || teta == temp && big < abs_alfa_i)
            p = k, teta = temp, big = abs_alfa_i;
      }
      /* return the ordinal number of the chosen basic variable */
      return p;
}

/*----------------------------------------------------------------------
-- lpx_dual_ratio_test - perform dual ratio test.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_dual_ratio_test(LPX *lp, int len, int ndx[], double val[],
--    int how, double tol);
--
-- *Description*
--
-- The routine lpx_dual_ratio_test performs the dual ratio test for an
-- explicitly specified row of the simplex table.
--
-- The dual basic solution associated with an LP problem object, which
-- the parameter lp points to, should be feasible. No components of the
-- LP problem object are changed by the routine.
--
-- The explicitly specified row of the simplex table is a linear form,
-- which shows how some basic variable y (not necessarily presented in
-- the problem object) depends on non-basic variables xN:
--
--    y = alfa[1]*xN[1] + alfa[2]*xN[2] + ... + alfa[n]*xN[n].       (*)
--
-- The linear form (*) is specified on entry to the routine using the
-- sparse format. Ordinal numbers of non-basic variables xN[j] should be
-- placed in locations ndx[1], ..., ndx[len], where ordinal numbers 1 to
-- m denote auxiliary variables, and ordinal numbers m+1 to m+n denote
-- structural variables. The corresponding non-zero coefficients alfa[j]
-- should be placed in locations val[1], ..., val[len]. The arrays ndx
-- and val are not changed on exit.
--
-- The parameter how specifies in which direction the variable y changes
-- on leaving the basis: +1 means increasing, -1 means decreasing.
--
-- The parameter tol is a relative tolerance (small positive number)
-- used by the routine to skip small alfa[j] of the form (*).
--
-- The routine determines the ordinal number of some non-basic variable
-- (specified in ndx[1], ..., ndx[len]), which should enter the basis
-- instead the variable y in order to keep dual feasibility, and returns
-- it on exit. If the choice cannot be made (i.e. if the adjacent basic
-- solution is dual unbounded), the routine returns zero.
--
-- *Note*
--
-- If the basic variable y is presented in the LP problem object, the
-- row (*) can be computed using the routine lpx_eval_tab_row. Otherwise
-- it can be computed using the routine lpx_transform_row.
--
-- *Returns*
--
-- The routine lpx_dual_ratio_test returns the ordinal number of some
-- non-basic variable xN[j], which should enter the basis instead the
-- variable y in order to keep dual feasibility. If the adjacent basic
-- solution is dual unbounded and therefore the choice cannot be made,
-- the routine returns zero. */

int lpx_dual_ratio_test(LPX *lp, int len, int ndx[], double val[],
      int how, double tol)
{     int m = lp->m;
      int n = lp->n;
      double *rs = lp->rs;
      double dir = (lp->dir == LPX_MIN ? +1.0 : -1.0);
      int *tagx = lp->tagx;
      int *posx = lp->posx;
      double *cbar = lp->cbar;
      int j, k, t, q;
      double alfa_j, abs_alfa_j, big, eps, cbar_j, temp, teta;
      /* the current basic solution should be dual feasible */
      if (lp->d_stat != LPX_D_FEAS)
         fault("lpx_dual_ratio_test: current basic solution is not dual"
            " feasible");
      /* check if the parameter how is correct */
      if (!(how == +1 || how == -1))
         fault("lpx_dual_ratio_test: how = %d; invalid parameter", how);
      /* compute the largest absolute value of the specified influence
         coefficients */
      big = 0.0;
      for (t = 1; t <= len; t++)
      {  temp = val[t];
         if (temp < 0.0) temp = - temp;
         if (big < temp) big = temp;
      }
      /* compute the absolute tolerance eps used to skip small entries
         of the row */
      if (!(0.0 < tol && tol < 1.0))
         fault("lpx_dual_ratio_test: tol = %g; invalid tolerance", tol);
      eps = tol * (1.0 + big);
      /* initial settings */
      q = 0, teta = DBL_MAX, big = 0.0;
      /* walk through the entries of the specified row */
      for (t = 1; t <= len; t++)
      {  /* get ordinal number of non-basic variable */
         k = ndx[t];
         if (!(1 <= k && k <= m+n))
            fault("lpx_dual_ratio_test: ndx[%d] = %d; ordinal number ou"
               "t of range", t, k);
         if (tagx[k] == LPX_BS)
            fault("lpx_dual_ratio_test: ndx[%d] = %d; basic variable no"
               "t allowed", t, k);
         /* determine index of the variable x[k] in the vector xN */
         j = posx[k] - m; /* x[k] = xN[j] */
         insist(1 <= j && j <= n);
         /* determine unscaled reduced cost of the non-basic variable
            x[k] = xN[j] in the current basic solution */
         cbar_j = (k <= m ? cbar[j] * rs[k] : cbar[j] / rs[k]);
         /* determine influence coefficient at the non-basic variable
            x[k] = xN[j] in the explicitly specified row and turn to
            the case of increasing the variable y in order to simplify
            program logic */
         alfa_j = (how > 0 ? +val[t] : -val[t]);
         abs_alfa_j = (alfa_j > 0.0 ? +alfa_j : -alfa_j);
         /* analyze main cases */
         switch (tagx[k])
         {  case LPX_NL:
               /* xN[j] is on its lower bound */
               if (alfa_j < +eps) continue;
               temp = (dir * cbar_j) / alfa_j;
               break;
            case LPX_NU:
               /* xN[j] is on its upper bound */
               if (alfa_j > -eps) continue;
               temp = (dir * cbar_j) / alfa_j;
               break;
            case LPX_NF:
               /* xN[j] is non-basic free variable */
               if (abs_alfa_j < eps) continue;
               temp = 0.0;
               break;
            case LPX_NS:
               /* xN[j] is non-basic fixed variable */
               continue;
            default:
               insist(tagx != tagx);
         }
         /* if the reduced cost of the variable xN[j] violates its zero
            bound (slightly, because the current basis is assumed to be
            dual feasible), temp is negative; we can think this happens
            due to round-off errors and the reduced cost is exact zero;
            this allows replacing temp by zero */
         if (temp < 0.0) temp = 0.0;
         /* apply the minimal ratio test */
         if (teta > temp || teta == temp && big < abs_alfa_j)
            q = k, teta = temp, big = abs_alfa_j;
      }
      /* return the ordinal number of the chosen non-basic variable */
      return q;
}

/* eof */
