/* glplpx8.c (additional utility routines) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <float.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include "glplpx.h"
#include "glpmps.h"

/*----------------------------------------------------------------------
-- lpx_read_mps - read in problem data using MPS format.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- LPX *lpx_read_mps(char *fname);
--
-- *Description*
--
-- The routine lpx_read_mps reads LP/MIP problem data using MPS format
-- from a text file whose name is the character string fname.
--
-- *Returns*
--
-- If no error occurred, the routine returns a pointer to the created
-- problem object. Otherwise the routine returns NULL. */

struct mps_info
{     /* transit information passed to the routine mps_mat */
      MPS *mps;
      /* MPS data block */
      int j;
      /* current column number */
      MPSCQE *cqe;
      /* pointer to the next element in the current column list */
};

static double mps_mat(void *_info, int *i, int *j)
{     /* "reads" a next element of the constraint matrix */
      struct mps_info *info = _info;
      double aij;
read: if (info->j == 0 || info->cqe == NULL)
      {  /* either it is the first call or the current column has been
            completely read; set the first/next column */
         info->j++;
         if (info->j > info->mps->n_col)
         {  /* the entire matrix has been completely read */
            *i = *j = 0; /* end of file */
            info->j = 0; /* rewind */
            return 0.0;
         }
         /* set the pointer to the first element of the column */
         info->cqe = info->mps->col[info->j]->ptr;
         /* the column may be empty, so repeat the check */
         goto read;
      }
      /* obtain the next element from the current column */
      *i = info->cqe->ind;
      *j = info->j;
      aij = info->cqe->val;
      /* advance the pointer */
      info->cqe = info->cqe->next;
      /* skip the element if it has zero value */
      if (aij == 0.0) goto read;
      /* bring the next element to the calling program */
      return aij;
}

LPX *lpx_read_mps(char *fname)
{     LPX *lp = NULL;
      MPS *mps;
      int m, n, obj, i, j;
      /* read MPS data block */
      mps = load_mps(fname);
      if (mps == NULL) goto fail;
      m = mps->n_row;
      n = mps->n_col;
      obj = 0;
      /* create problem object */
      lp = lpx_create_prob();
      lpx_set_prob_name(lp, mps->name);
      lpx_add_rows(lp, m);
      lpx_add_cols(lp, n);
      /* process ROW section */
      for (i = 1; i <= m; i++)
      {  MPSROW *row;
         int typx;
         row = mps->row[i];
         lpx_set_row_name(lp, i, row->name);
         if (strcmp(row->type, "N") == 0)
         {  typx = LPX_FR;
            if (obj == 0) obj = i;
         }
         else if (strcmp(row->type, "L") == 0)
            typx = LPX_UP;
         else if (strcmp(row->type, "G") == 0)
            typx = LPX_LO;
         else if (strcmp(row->type, "E") == 0)
            typx = LPX_FX;
         else
         {  print("lpx_read_mps: row `%s' has unknown type `%s'",
               row->name, row->type);
            goto fail;
         }
         lpx_set_row_bnds(lp, i, typx, 0.0, 0.0);
      }
      if (obj == 0)
      {  print("lpx_read_mps: objective function row not found");
         goto fail;
      }
      /* process COLUMN section */
      for (j = 1; j <= n; j++)
      {  MPSCOL *col;
         MPSCQE *cqe;
         col = mps->col[j];
         lpx_set_col_name(lp, j, col->name);
         lpx_set_col_bnds(lp, j, LPX_LO, 0.0, 0.0);
         if (col->flag)
         {  lpx_set_class(lp, LPX_MIP);
            lpx_set_col_kind(lp, j, LPX_IV);
         }
         for (cqe = col->ptr; cqe != NULL; cqe = cqe->next)
            if (cqe->ind == obj) lpx_set_col_coef(lp, j, cqe->val);
      }
      /* load the constraint matrix */
      {  struct mps_info info;
         info.mps = mps;
         info.j = 0;
         info.cqe = NULL;
         lpx_load_mat(lp, &info, mps_mat);
      }
      /* process RHS section */
      if (mps->n_rhs > 0)
      {  MPSCQE *cqe;
         for (cqe = mps->rhs[1]->ptr; cqe != NULL; cqe = cqe->next)
         {  int typx;
            double lb, ub;
            lpx_get_row_bnds(lp, cqe->ind, &typx, NULL, NULL);
            switch (typx)
            {  case LPX_FR:
                  /* if it is the objective function row, the specified
                     right-hand side is considered as a constant term of
                     the objective function; for other free rows the rhs
                     is ignored */
                  if (cqe->ind == obj) lpx_set_obj_c0(lp, cqe->val);
                  lb = ub = 0.0;
                  break;
               case LPX_LO:
                  lb = cqe->val, ub = 0.0;
                  break;
               case LPX_UP:
                  lb = 0.0, ub = cqe->val;
                  break;
               case LPX_FX:
                  lb = ub = cqe->val;
                  break;
               default:
                  insist(typx != typx);
            }
            lpx_set_row_bnds(lp, cqe->ind, typx, lb, ub);
         }
      }
      /* process RANGES section */
      if (mps->n_rng > 0)
      {  MPSCQE *cqe;
         for (cqe = mps->rng[1]->ptr; cqe != NULL; cqe = cqe->next)
         {  int typx;
            double lb, ub;
            lpx_get_row_bnds(lp, cqe->ind, &typx, &lb, &ub);
            switch (typx)
            {  case LPX_FR:
                  print("lpx_read_mps: range vector entry refers to row"
                     " `%s' of N type", mps->row[cqe->ind]->name);
                  goto fail;
               case LPX_LO:
                  ub = lb + fabs(cqe->val);
                  break;
               case LPX_UP:
                  lb = ub - fabs(cqe->val);
                  break;
               case LPX_FX:
                  if (cqe->val >= 0.0)
                     ub += fabs(cqe->val);
                  else
                     lb -= fabs(cqe->val);
                  break;
               default:
                  insist(typx != typx);
            }
            lpx_set_row_bnds(lp, cqe->ind, lb == ub ? LPX_FX : LPX_DB,
               lb, ub);
         }
      }
      /* process BOUNDS section */
      if (mps->n_bnd > 0)
      {  MPSBQE *bqe;
         for (bqe = mps->bnd[1]->ptr; bqe != NULL; bqe = bqe->next)
         {  int typx;
            double lb, ub;
            lpx_get_col_bnds(lp, bqe->ind, &typx, &lb, &ub);
            if (typx == LPX_FR || typx == LPX_UP) lb = -DBL_MAX;
            if (typx == LPX_FR || typx == LPX_LO) ub = +DBL_MAX;
            if (strcmp(bqe->type, "LO") == 0)
               lb = bqe->val;
            else if (strcmp(bqe->type, "UP") == 0)
               ub = bqe->val;
            else if (strcmp(bqe->type, "FX") == 0)
               lb = ub = bqe->val;
            else if (strcmp(bqe->type, "FR") == 0)
               lb = -DBL_MAX, ub = +DBL_MAX;
            else if (strcmp(bqe->type, "MI") == 0)
               lb = -DBL_MAX;
            else if (strcmp(bqe->type, "PL") == 0)
               ub = +DBL_MAX;
            else if (strcmp(bqe->type, "UI") == 0)
            {  /* integer structural variable with upper bound */
               lpx_set_class(lp, LPX_MIP);
               lpx_set_col_kind(lp, bqe->ind, LPX_IV);
               ub = bqe->val;
            }
            else if (strcmp(bqe->type, "BV") == 0)
            {  /* binary structural variable */
               lpx_set_class(lp, LPX_MIP);
               lpx_set_col_kind(lp, bqe->ind, LPX_IV);
               lb = 0.0, ub = 1.0;
            }
            else
            {  print("lpx_read_mps: bound vector entry for column `%s' "
                  "has unknown type `%s'",
               mps->col[bqe->ind]->name, bqe->type);
               goto fail;
            }
            if (lb == -DBL_MAX && ub == +DBL_MAX)
               typx = LPX_FR;
            else if (ub == +DBL_MAX)
               typx = LPX_LO;
            else if (lb == -DBL_MAX)
               typx = LPX_UP;
            else if (lb != ub)
               typx = LPX_DB;
            else
               typx = LPX_FX;
            lpx_set_col_bnds(lp, bqe->ind, typx, lb, ub);
         }
      }
      /* deallocate MPS data block */
      free_mps(mps);
      /* return to the calling program */
      return lp;
fail: /* the operation failed */
      if (lp != NULL) lpx_delete_prob(lp);
      if (mps != NULL) free_mps(mps);
      return NULL;
}

/*----------------------------------------------------------------------
-- lpx_print_sol - write LP problem solution in printable format.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_print_sol(LPX *lp, char *fname);
--
-- *Description*
--
-- The routine lpx_print_sol writes the current basic solution of an LP
-- problem, which is specified by the pointer lp, to a text file, whose
-- name is the character string fname, in printable format.
--
-- Information reported by the routine lpx_print_sol is intended mainly
-- for visual analysis.
--
-- *Returns*
--
-- If the operation was successful, the routine returns zero. Otherwise
-- the routine prints an error message and returns non-zero. */

int lpx_print_sol(LPX *lp, char *fname)
{     FILE *fp;
      int what, round;
      print("lpx_print_sol: writing LP problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  print("lpx_print_sol: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char *name;
         name = lpx_get_prob_name(lp);
         if (name == NULL) name = "";
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         nr = lpx_get_num_rows(lp);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc;
         nc = lpx_get_num_cols(lp);
         fprintf(fp, "%-12s%d\n", "Columns:", nc);
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         nz = lpx_get_num_nz(lp);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* solution status */
      {  int status;
         status = lpx_get_status(lp);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == LPX_OPT    ? "OPTIMAL" :
            status == LPX_FEAS   ? "FEASIBLE" :
            status == LPX_INFEAS ? "INFEASIBLE (INTERMEDIATE)" :
            status == LPX_NOFEAS ? "INFEASIBLE (FINAL)" :
            status == LPX_UNBND  ? "UNBOUNDED" :
            status == LPX_UNDEF  ? "UNDEFINED" : "???");
      }
      /* objective function */
      {  int dir;
         double obj;
         dir = lpx_get_obj_dir(lp);
         round = lp->round, lp->round = 1;
         obj = lpx_get_obj_val(lp);
         lp->round = round;
         fprintf(fp, "%-12s%.10g %s\n", "Objective:", obj,
            dir == LPX_MIN ? "(MINimum)" :
            dir == LPX_MAX ? "(MAXimum)" : "(???)");
      }
      /* main sheet */
      for (what = 1; what <= 2; what++)
      {  int mn, ij;
         fprintf(fp, "\n");
         fprintf(fp, "   No. %-12s St   Activity     Lower bound   Uppe"
            "r bound    Marginal\n",
            what == 1 ? "  Row name" : "Column name");
         fprintf(fp, "------ ------------ -- ------------- ------------"
            "- ------------- -------------\n");
         mn = (what == 1 ? lpx_get_num_rows(lp) : lpx_get_num_cols(lp));
         for (ij = 1; ij <= mn; ij++)
         {  char *name;
            int typx, tagx;
            double lb, ub, valx, dx;
            if (what == 1)
            {  name = lpx_get_row_name(lp, ij);
               if (name == NULL) name = "";
               lpx_get_row_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               lpx_get_row_info(lp, ij, &tagx, &valx, &dx);
               lp->round = round;
            }
            else
            {  name = lpx_get_col_name(lp, ij);
               if (name == NULL) name = "";
               lpx_get_col_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               lpx_get_col_info(lp, ij, &tagx, &valx, &dx);
               lp->round = round;
            }
            /* row/column ordinal number */
            fprintf(fp, "%6d ", ij);
            /* row column/name */
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%20s", name, "");
            /* row/column status */
            fprintf(fp, "%s ",
               tagx == LPX_BS ? "B " :
               tagx == LPX_NL ? "NL" :
               tagx == LPX_NU ? "NU" :
               tagx == LPX_NF ? "NF" :
               tagx == LPX_NS ? "NS" : "??");
            /* row/column primal activity */
            fprintf(fp, "%13.6g ", valx);
            /* row/column lower bound */
            if (typx == LPX_LO || typx == LPX_DB || typx == LPX_FX)
               fprintf(fp, "%13.6g ", lb);
            else
               fprintf(fp, "%13s ", "");
            /* row/column upper bound */
            if (typx == LPX_UP || typx == LPX_DB)
               fprintf(fp, "%13.6g ", ub);
            else if (typx == LPX_FX)
               fprintf(fp, "%13s ", "=");
            else
               fprintf(fp, "%13s ", "");
            /* row/column dual activity */
            if (tagx != LPX_BS)
            {  if (dx == 0.0)
                  fprintf(fp, "%13s", "< eps");
               else
                  fprintf(fp, "%13.6g", dx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  print("lpx_print_sol: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/*----------------------------------------------------------------------
-- lpx_print_mip - write MIP problem solution in printable format.
--
-- *Synopsis*
--
-- #include "glplpx.h"
-- int lpx_print_mip(LPX *lp, char *fname);
--
-- *Description*
--
-- The routine lpx_print_mip writes a best known integer solution of
-- a MIP problem, which is specified by the pointer lp, to a text file,
-- whose name is the character string fname, in printable format.
--
-- Information reported by the routine lpx_print_mip is intended mainly
-- for visual analysis.
--
-- *Returns*
--
-- If the operation was successful, the routine returns zero. Otherwise
-- the routine prints an error message and returns non-zero. */

int lpx_print_mip(LPX *lp, char *fname)
{     FILE *fp;
      int what, round;
      if (lpx_get_class(lp) != LPX_MIP)
         fault("lpx_print_mip: error -- not a MIP problem");
      print("lpx_print_mip: writing MIP problem solution to `%s'...",
         fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  print("lpx_print_mip: can't create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      /* problem name */
      {  char *name;
         name = lpx_get_prob_name(lp);
         if (name == NULL) name = "";
         fprintf(fp, "%-12s%s\n", "Problem:", name);
      }
      /* number of rows (auxiliary variables) */
      {  int nr;
         nr = lpx_get_num_rows(lp);
         fprintf(fp, "%-12s%d\n", "Rows:", nr);
      }
      /* number of columns (structural variables) */
      {  int nc, nc_int, nc_bin;
         nc = lpx_get_num_cols(lp);
         nc_int = lpx_get_num_int(lp);
         nc_bin = lpx_get_num_bin(lp);
         fprintf(fp, "%-12s%d (%d integer, %d binary)\n", "Columns:",
            nc, nc_int, nc_bin);
      }
      /* number of non-zeros (constraint coefficients) */
      {  int nz;
         nz = lpx_get_num_nz(lp);
         fprintf(fp, "%-12s%d\n", "Non-zeros:", nz);
      }
      /* solution status */
      {  int status;
         status = lpx_get_mip_stat(lp);
         fprintf(fp, "%-12s%s\n", "Status:",
            status == LPX_I_UNDEF  ? "UNDEFINED" :
            status == LPX_I_OPT    ? "INTEGER OPTIMAL" :
            status == LPX_I_FEAS   ? "INTEGER NON-OPTIMAL" :
            status == LPX_I_NOFEAS ? "INFEAS..." : "???");
      }
      /* objective function */
      {  int dir;
         double mip_obj, lp_obj;
         dir = lpx_get_obj_dir(lp);
         round = lp->round, lp->round = 1;
         mip_obj = lpx_get_mip_obj(lp);
         lp_obj = lpx_get_obj_val(lp);
         lp->round = round;
         fprintf(fp, "%-12s%.10g %s\n", "Objective:", mip_obj,
            dir == LPX_MIN ? "(MINimum)" :
            dir == LPX_MAX ? "(MAXimum)" : "(???)");
         fprintf(fp, "%-12s%.10g (LP relaxation)\n", "", lp_obj);
      }
      /* main sheet */
      for (what = 1; what <= 2; what++)
      {  int mn, ij;
         fprintf(fp, "\n");
         fprintf(fp, "   No. %-12s      Activity     Lower bound   Uppe"
            "r bound\n",
            what == 1 ? "  Row name" : "Column name");
         fprintf(fp, "------ ------------    ------------- ------------"
            "- -------------\n");
         mn = (what == 1 ? lpx_get_num_rows(lp) : lpx_get_num_cols(lp));
         for (ij = 1; ij <= mn; ij++)
         {  char *name;
            int kind, typx;
            double lb, ub, valx;
            if (what == 1)
            {  name = lpx_get_row_name(lp, ij);
               if (name == NULL) name = "";
               kind = LPX_CV;
               lpx_get_row_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               valx = lpx_get_mip_row(lp, ij);
               lp->round = round;
            }
            else
            {  name = lpx_get_col_name(lp, ij);
               if (name == NULL) name = "";
               kind = lpx_get_col_kind(lp, ij);
               lpx_get_col_bnds(lp, ij, &typx, &lb, &ub);
               round = lp->round, lp->round = 1;
               valx = lpx_get_mip_col(lp, ij);
               lp->round = round;
            }
            /* row/column ordinal number */
            fprintf(fp, "%6d ", ij);
            /* row column/name */
            if (strlen(name) <= 12)
               fprintf(fp, "%-12s ", name);
            else
               fprintf(fp, "%s\n%20s", name, "");
            /* row/column kind */
            fprintf(fp, "%s  ",
               kind == LPX_CV ? " " : kind == LPX_IV ? "*" : "?");
            /* row/column primal activity */
            fprintf(fp, "%13.6g", valx);
            /* row/column lower and upper bounds */
            switch (typx)
            {  case LPX_FR:
                  break;
               case LPX_LO:
                  fprintf(fp, " %13.6g", lb);
                  break;
               case LPX_UP:
                  fprintf(fp, " %13s %13.6g", "", ub);
                  break;
               case LPX_DB:
                  fprintf(fp, " %13.6g %13.6g", lb, ub);
                  break;
               case LPX_FX:
                  fprintf(fp, " %13.6g %13s", lb, "=");
                  break;
               default:
                  insist(typx != typx);
            }
            fprintf(fp, "\n");
         }
      }
      fprintf(fp, "\n");
      fprintf(fp, "End of output\n");
      fflush(fp);
      if (ferror(fp))
      {  print("lpx_print_mip: can't write to `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
