/* glpapi2.c (glp_simplex) */

/*----------------------------------------------------------------------
-- Copyright (C) 2000, 2001, 2002 Andrew Makhorin <mao@mai2.rcnet.ru>,
--               Department for Applied Informatics, Moscow Aviation
--               Institute, Moscow, Russia. All rights reserved.
--
-- This file is a part of GLPK (GNU Linear Programming Kit).
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
-- 02111-1307, USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_init_spx - initialize parameter block by default values.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_init_spx(struct spx *parm);
--
-- *Description*
--
-- The routine glp_init_spx initializes the parameter block passed to
-- the simplex method driver by default values. */

void glp_init_spx(struct spx *parm)
{     parm->msg_lev = 3;
      parm->scale   = 1;
      parm->initb   = 1;
      parm->price   = 1;
      parm->relax   = 1;
      parm->tol_bnd = 1e-7;
      parm->tol_dj  = 1e-7;
      parm->tol_piv = 1e-9;
      parm->round   = 1;
      parm->it_lim  = -1;
      parm->tm_lim  = -1.0;
      parm->out_frq = 200;
      return;
}

/*----------------------------------------------------------------------
-- glp_simplex - solve LP problem using simplex method.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_simplex(LPI *lp, struct spx *parm);
--
-- *Description*
--
-- The routine glp_simplex is an LP problem solver, which is based on
-- the two-phase primal simplex method.
--
-- This routine obtains problem data from a problem object, which the
-- parameter lp points to, solves the problem, and stores the computed
-- solution back to the problem object.
--
-- The parameter parm is a pointer to the parameter block used by the
-- solver. This block may be initialized by the routine glp_init_spx by
-- standard default values. It is allowed to specify NULL, in which
-- case standard default values are used. If the problem is not very
-- hard, the standard default values fit for most cases.
--
-- Since large-scale problems may take a long time, the solver reports
-- some visual information about current status of the search. This
-- information is sent to stdout once per 100 simplex iterations using
-- the following format:
--
--    nnn:   objval = xxx   infeas = yyy (ddd)
--
-- where nnn is the iteration count, xxx is the current value of the
-- objective function (which is unscaled and has correct sign), yyy is
-- a measure of primal infeasibility (which is the current value of the
-- artificial variable introduced on the phase I), ddd is the number of
-- basic fixed variables.
--
-- Please note that this solver is not perfect. Although it has been
-- successfully tested on a wide set of real LP problems, there are so
-- called hard problems, which can't be resolved by this solver.
--
-- The routine glp_simplex returns one of the following codes:
--
-- 0 - no errors. This case means that the solver has successfully
--     finished solving the problem. (Note, for example, if the problem
--     has no feasible solution, the solver returns zero code);
-- 1 - iterations limit or time limit exceeded. In this case the solver
--     reports the most recently obtained basic solution;
-- 2 - numerical problems with basis matrix. This case means that the
--     solver is not able to solve the problem. */

#define prefix "glp_simplex: "

int glp_simplex(LPI *lp, struct spx *parm)
{     int m = glp_get_num_rows(lp);
      int n = glp_get_num_cols(lp);
      struct spx my_parm;
      LPX *lpx;
      int ret;
      if (m == 0) fault(prefix "problem has no rows");
      if (n == 0) fault(prefix "problem has no columns");
      /* invalidate the solution */
      glp_put_soln_info(lp, 'N', GLP_UNDEF, 0.0);
      /* if parameter block is not specified, use the dummy one */
      if (parm == NULL)
      {  parm = &my_parm;
         glp_init_spx(parm);
      }
      /* check control parameters for correctness */
      if (!(0 <= parm->msg_lev && parm->msg_lev <= 3))
         fault(prefix "msg_lev = %d; invalid parameter", parm->msg_lev);
      if (!(parm->scale == 0 || parm->scale == 1))
         fault(prefix "scale = %d; invalid parameter", parm->scale);
      if (!(parm->initb == 0 || parm->initb == 1 || parm->initb == 2))
         fault(prefix "initb = %d; invalid parameter", parm->initb);
      if (!(parm->price == 0 || parm->price == 1))
         fault(prefix "price = %d; invalid parameter", parm->price);
      if (!(parm->relax == 0 || parm->relax == 1))
         fault(prefix "relax = %d; invalid parameter", parm->relax);
      if (!(0.0 < parm->tol_bnd && parm->tol_bnd < 1.0))
         fault(prefix "tol_bnd = %g; invalid parameter", parm->tol_bnd);
      if (!(0.0 < parm->tol_dj && parm->tol_dj < 1.0))
         fault(prefix "tol_dj = %g; invalid parameter", parm->tol_dj);
      if (!(0.0 < parm->tol_piv && parm->tol_piv < 1.0))
         fault(prefix "tol_piv = %g; invalid parameter", parm->tol_piv);
      if (!(parm->round == 0 || parm->round == 1))
         fault(prefix "round = %d; invalid parameter", parm->round);
      if (!(parm->out_frq > 0))
         fault(prefix "out_frq = %d; invalid parameter", parm->out_frq);
      /* convert LP problem instance to the low-level object */
      lpx = glp_lpi_to_lpx(lp, 0);
      /* scale the problem (if required) */
      if (parm->scale) lpx_scale_prob(lpx);
      /* build an initial basis */
      switch (parm->initb)
      {  case 0:
            lpx_std_basis(lpx); break;
         case 1:
            lpx_adv_basis(lpx); break;
         case 2:
            /* use the current basis */
            break;
         default:
            insist(parm->initb != parm->initb);
      }
      /* set up other control parameters */
      lpx->msg_lev = parm->msg_lev;
      lpx->price = parm->price;
      if (!parm->relax) lpx->relax = 0.0;
      lpx->tol_bnd = parm->tol_bnd;
      lpx->tol_dj = parm->tol_dj;
      lpx->tol_piv = parm->tol_piv;
      lpx->round = parm->round;
      lpx->it_lim = parm->it_lim;
      lpx->tm_lim = parm->tm_lim;
      lpx->out_frq = parm->out_frq;
      /* solve the problem */
      ret = lpx_simplex(lpx);
      /* analyze return code */
      switch (ret)
      {  case LPX_E_OK:
            ret = 0; break;
         case LPX_E_FAULT:
            ret = 2; break;
         case LPX_E_ITLIM:
         case LPX_E_TMLIM:
            ret = 1; break;
         case LPX_E_SING:
            ret = 2; break;
         default:
            insist(ret != ret);
      }
      /* load solution information from low-level LP object */
      glp_load_lpx_soln(lp, lpx, parm->round);
      /* delete the low-level object */
      lpx_delete_prob(lpx);
      /* return to the calling program */
      return ret;
}

/* eof */
