/* glppfi/update_pfi.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include "glppfi.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- update_pfi - update PFI for adjacent basis matrix.
--
-- *Synopsis*
--
-- #include "glppfi.h"
-- int update_pfi(PFI *pfi, int p);
--
-- *Description*
--
-- The update_pfi routine recomputes PFI corresponding to the current
-- basis matrix B, so that the updated PFI will correspond to the new
-- (adjacent) basis matrix Bnew, where Bnew is a result of change p-th
-- column of B by other column.
--
-- Note that new p-th column of the basis matrix is passed implicitly to
-- this routine: the update_pfi routine assumes that the transformed new
-- p-th column was saved before by the pfi_ftran routine.
--
-- *Returns*
--
-- The update_pfi routine returns one of the following codes:
--
-- 0 - PFI has been successfully updated;
-- 1 - PFI became inaccurate;
-- 2 - PFI became too long.
--
-- If the returned code is non-zero, PFI should be rebuilt by the means
-- of the build_pfi routine. */

int update_pfi(PFI *pfi, int p)
{     int m = pfi->m, i;
      double big, *h, rtol = 1e-6, drop = 1e-15;
      if (!(1 <= p && p <= m))
         fault("update_pfi: invalid column number");
      if (!pfi->flag)
         fault("update_pfi: transformed column not ready");
      /* the current basis matrix is B = B0*H; the new basis matrix is
         Bnew = B*Hcol = B0*H*Hcol = B0*Hnew, where Hnew = H*Hcol, and
         Hcol differs from the unity matrix only by one column, which
         is inv(B)*(new p-th column), i.e. that column is a result of
         forward transformation of new p-th column */
      h = pfi->col; /* p-th column of Hcol */
      /* compute maximal absolute value of elements of column h */
      big = 0.0;
      for (i = 1; i <= m; i++)
      {  double t = fabs(h[i]);
         if (big < t) big = t;
      }
      /* if diagonal element h[p] is relatively small, Hcol is near to
         singular matrix; therefore PFI should be rebuilt */
      if (h[p] == 0.0 || fabs(h[p]) < rtol * big) return 1;
      /* diagonal element of p-th column of Hcol becomes the first
         eta-term */
      app_term(pfi->eta, p, p, h[p]);
      /* non-zero non-diagonal elements of p-th column of Hcol become
         other eta-terms */
      for (i = 1; i <= m; i++)
      {  if (i == p) continue;
         if (h[i] == 0.0 || fabs(h[i]) < drop * big) continue;
         app_term(pfi->eta, i, p, h[i]);
      }
      /* if eta-file is too long, PFI should be rebuilt */
      {  int size = pfi->eta->pool->count;
         int nzL  = pfi->lu->L->pool->count;
         int nzU  = pfi->lu->U->pool->count;
         if (size > 3 * (nzL + nzU)) return 2;
      }
      /* the transformed column is no longer valid */
      pfi->flag = 0;
      /* PFI has been successfully updated */
      return 0;
}

/* eof */
