/* glplp/prepro_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <math.h>
#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- prepro_lp - perform preprocessing LP/MIP problem.
--
-- *Synopsis*
--
-- #include "glplp.h"
-- int prepro_lp(LP *lp);
--
-- *Description*
--
-- The prepro_lp routine performs preprocessing LP/MIP problem specified
-- by the data block, which the parameter lp points to.
--
-- The result of preprocessing is a problem, which has the same feasible
-- region as the original problem.
--
-- In the resultant problem lower and/or upper bounds of some auxiliary
-- variables may be removed, and bounds of some structural variables may
-- be tightened (in particular, additional fixed variables may appear).
-- The constraint matrix is not changed.
--
-- *Returns*
--
-- The prepro_lp routine returns one of the following codes:
--
-- 0 - ok (no infeasibility detected);
-- i - i-th row is inconsistent. */

static LP *lp;
/* LP/MIP problem data block */

static int m;
/* number of rows = number of auxiliary variables */

static int n;
/* number of columns = number of structural variables */

struct limit { double val; int cnt; };
/* this structure represents infimum/supremum of a row; val is a finite
   part, and cnt is number of infinite terms */

static struct limit *f_min; /* struct limit f_min[1+m]; */
/* f_min[i] is an infimum of the i-th row (1 <= i <= m) computed using
   the current bounds of structural variables */

static struct limit *f_max; /* struct limit f_max[1+m]; */
/* f_max[i] is a supremum of the i-th row (1 <= i <= m) computed using
   the current bounds of structural variables */

static int nr; /* 0 <= nr <= m */
/* current number of rows in the active row list */

static int *rlist; /* int rlist[1+m]; */
/* row list: the elements rlist[1], ..., rlist[nr] are numbers of rows,
   which should be processed on the next pass */

static char *rflag; /* char rflag[1+m]; */
/* rflag[i] != 0 means that the i-th row is in the row list */

static int nc; /* 0 <= nc <= n */
/* current number of columns in the active column list */

static int *clist; /* int clist[1+n]; */
/* column list: the elements clist[1], ..., clist[nc] are numbers of
   columns, which should be processed on the next pass */

static char *cflag; /* char cflag[1+n]; */
/* cflag[j] != 0 means that the j-th column is in the column list */

/*----------------------------------------------------------------------
-- prepro_row - perform preprocessing i-th row.
--
-- This routine performs the following:
--
-- estimating range of i-th row using the current bounds of columns;
--
-- checking necessary feasibility conditions;
--
-- removing lower or/and upper bound of i-th row in case of redundancy.
--
-- The parameter tol is a relative tolerance used for detecting bound
-- infeasibility/redundancy.
--
-- If necessary feasibility conditions for the given row are satisfied,
-- the routine returns zero. Otherwise, the routine returns non-zero. */

static int prepro_row(int i, double tol)
{     ELEM *e;
      int k;
      insist(1 <= i && i <= m);
      /* determine infimum (f_min) and supremum (f_max) of the i-th row
         using the current bounds of structural variables */
      f_min[i].val = 0.0, f_min[i].cnt = 0;
      f_max[i].val = 0.0, f_max[i].cnt = 0;
      for (e = lp->A->row[i]; e != NULL; e = e->row)
      {  k = m + e->j; /* x[k] = j-th structural variable */
         if (e->val > 0.0)
         {  if (lp->lb[k] == -DBL_MAX)
               f_min[i].cnt++;
            else
               f_min[i].val += e->val * lp->lb[k];
            if (lp->ub[k] == +DBL_MAX)
               f_max[i].cnt++;
            else
               f_max[i].val += e->val * lp->ub[k];
         }
         else if (e->val < 0.0)
         {  if (lp->ub[k] == +DBL_MAX)
               f_min[i].cnt++;
            else
               f_min[i].val += e->val * lp->ub[k];
            if (lp->lb[k] == -DBL_MAX)
               f_max[i].cnt++;
            else
               f_max[i].val += e->val * lp->lb[k];
         }
      }
      /* analyze f_min[i] */
      if (lp->ub[i] != +DBL_MAX && f_min[i].cnt == 0 &&
          check_rr(f_min[i].val, lp->ub[i], tol) > +1)
      {  /* if f_min[i] > ub[i] + eps, the i-th row is infeasible */
         return 1;
      }
      if (lp->lb[i] != -DBL_MAX && (f_min[i].cnt > 0 ||
          check_rr(f_min[i].val, lp->lb[i], tol) <= +1))
      {  /* if f_min[i] <= lb[i] + eps, the lower bound of the i-th row
            can be active and therefore may involve tightening bounds of
            structural variables in this row */
         for (e = lp->A->row[i]; e != NULL; e = e->row)
         {  if (e->val != 0.0 && cflag[e->j] == 0)
               clist[++nc] = e->j, cflag[e->j] = 1;
         }
      }
      else
      {  /* if f_min[i] > lb[i] + eps, the lower bound of the i-th row is
            redundant and therefore may be removed */
         lp->lb[i] = -DBL_MAX;
      }
      /* analyze f_max[i] */
      if (lp->lb[i] != -DBL_MAX && f_max[i].cnt == 0 &&
          check_rr(f_max[i].val, lp->lb[i], tol) < -1)
      {  /* f_max[i] < lb[i] - eps, the i-th row is infeasible */
         return 1;
      }
      if (lp->ub[i] != +DBL_MAX && (f_max[i].cnt > 0 ||
          check_rr(f_max[i].val, lp->ub[i], tol) >= -1))
      {  /* if f_max[i] >= ub[i] + eps, the upper bound of the i-th row
            can be active and therefore may involve tightening bounds of
            structural variables in this row */
         for (e = lp->A->row[i]; e != NULL; e = e->row)
         {  if (e->val != 0.0 && cflag[e->j] == 0)
               clist[++nc] = e->j, cflag[e->j] = 1;
         }
      }
      else
      {  /* if f_max[i] < ub[i] + eps, the upper bound of the i-th row is
            redundant and therefore may be removed */
         lp->ub[i] = +DBL_MAX;
      }
      return 0;
}

/*----------------------------------------------------------------------
-- prepro_col - perform preprocessing j-th column.
--
-- This routine performs the following:
--
-- estimating range of j-th column using the current bounds of rows and
-- other columns;
--
-- checking necessary feasibility conditions;
--
-- improving (tightening) bounds of j-th column in case of redundancy.
--
-- The parameter tol is a relative tolerance used for detecting bound
-- infeasibility/redundancy.
--
-- The parameter tol1 is an absolute tolerance used for rounding bounds
-- of integer structural variables. */

static void prepro_col(int j, double tol, double tol1)
{     ELEM *e;
      int i, k;
      double hmin, hmax, tmin, tmax, xmin, xmax;
      insist(1 <= j && j <= n);
      k = m + j; /* x[k] = j-th structural variable */
      /* determine infimum (xmin) and supremum (xmax) of the j-th column
         using the current bounds of rows and other columns */
      xmin = -DBL_MAX, xmax = +DBL_MAX;
      for (e = lp->A->col[j]; e != NULL; e = e->col)
      {  i = e->i;
         /* lb[i] - hmax <= a[i,j]*x[m+j] <= ub[i] - hmin, where hmin
            and hmax are, respectively, infimum and supremum of the sum
            a[i,1]*x[m+1] + ... + a[i,n]*x[m+n] - a[i,j]*x[m+j]; thus,
            knowing hmin and hmax it is possible to compute the range
            tmin <= x[m+j] <= tmax */
         if (e->val > 0.0)
         {  if (f_min[i].cnt == 0)
               hmin = f_min[i].val - e->val * lp->lb[k];
            else if (f_min[i].cnt == 1 && lp->lb[k] == -DBL_MAX)
               hmin = f_min[i].val;
            else
               hmin = -DBL_MAX;
            if (f_max[i].cnt == 0)
               hmax = f_max[i].val - e->val * lp->ub[k];
            else if (f_max[i].cnt == 1 && lp->ub[k] == +DBL_MAX)
               hmax = f_max[i].val;
            else
               hmax = +DBL_MAX;
            if (lp->lb[i] == -DBL_MAX || hmax == +DBL_MAX)
               tmin = -DBL_MAX;
            else
               tmin = (lp->lb[i] - hmax) / e->val;
            if (lp->ub[i] == +DBL_MAX || hmin == -DBL_MAX)
               tmax = +DBL_MAX;
            else
               tmax = (lp->ub[i] - hmin) / e->val;
         }
         else if (e->val < 0.0)
         {  if (f_min[i].cnt == 0)
               hmin = f_min[i].val - e->val * lp->ub[k];
            else if (f_min[i].cnt == 1 && lp->ub[k] == +DBL_MAX)
               hmin = f_min[i].val;
            else
               hmin = -DBL_MAX;
            if (f_max[i].cnt == 0)
               hmax = f_max[i].val - e->val * lp->lb[k];
            else if (f_max[i].cnt == 1 && lp->lb[k] == -DBL_MAX)
               hmax = f_max[i].val;
            else
               hmax = +DBL_MAX;
            if (lp->ub[i] == +DBL_MAX || hmin == -DBL_MAX)
               tmin = -DBL_MAX;
            else
               tmin = (lp->ub[i] - hmin) / e->val;
            if (lp->lb[i] == -DBL_MAX || hmax == +DBL_MAX)
               tmax = +DBL_MAX;
            else
               tmax = (lp->lb[i] - hmax) / e->val;
         }
         else
            tmin = -DBL_MAX, tmax = +DBL_MAX;
         /* the final range (xmin, xmax) is the intersection of all the
            ranges (tmin, tmax) */
         if (xmin < tmin) xmin = tmin;
         if (xmax > tmax) xmax = tmax;
      }
      /* if the j-th structural variable is of integer kind, its implied
         bounds should be rounded off */
      if (lp->kind != NULL && lp->kind[j])
      {  if (xmin != -DBL_MAX)
         {  if (fabs(xmin - floor(xmin + 0.5)) <= tol1)
               xmin = floor(xmin + 0.5);
            else
               xmin = ceil(xmin);
         }
         if (xmax != +DBL_MAX)
         {  if (fabs(xmax - floor(xmax + 0.5)) <= tol1)
               xmax = floor(xmax + 0.5);
            else
               xmax = floor(xmax);
         }
      }
      /* if xmin > lb[m+j] + eps, the current lower bound of x[m+j] may
         be replaced by the implied (tighter) bound */
      if (xmin != -DBL_MAX && (lp->lb[m+j] == -DBL_MAX ||
          check_rr(xmin, lp->lb[m+j], tol) > +1))
      {  lp->lb[m+j] = xmin;
         /* tightening bounds may involve changing bounds of rows with
            this structural variable */
         for (e = lp->A->col[j]; e != NULL; e = e->col)
            if (e->val != 0.0 && rflag[e->i] == 0)
               rlist[++nr] = e->i, rflag[e->i] = 1;
      }
      /* if xmax < ub[m+j] - eps, the current upper bound of x[m+j] may
         be replaced by the implied (tighter) bound */
      if (xmax != +DBL_MAX && (lp->ub[m+j] == +DBL_MAX ||
          check_rr(xmax, lp->ub[m+j], tol) < -1))
      {  lp->ub[m+j] = xmax;
         /* tightening bounds may involve changing bounds of rows with
            this structural variable */
         for (e = lp->A->col[j]; e != NULL; e = e->col)
            if (e->val != 0.0 && rflag[e->i] == 0)
               rlist[++nr] = e->i, rflag[e->i] = 1;
      }
      return;
}

/*--------------------------------------------------------------------*/

int prepro_lp(LP *_lp)
{     int i, j, k, ret = 0;
      lp = _lp;
      m = lp->m, n = lp->n;
      /* introduce explicit "infinite" bounds (for simplicity) */
      for (k = 1; k <= m+n; k++)
      {  switch (lp->type[k])
         {  case 'F':
               lp->lb[k] = -DBL_MAX, lp->ub[k] = +DBL_MAX; break;
            case 'L':
               lp->ub[k] = +DBL_MAX; break;
            case 'U':
               lp->lb[k] = -DBL_MAX; break;
            case 'D':
            case 'S':
               break;
            default:
               insist(lp->type[k] != lp->type[k]);
         }
      }
      /* allocate working arrays */
      f_min = ucalloc(1+m, sizeof(struct limit));
      f_max = ucalloc(1+m, sizeof(struct limit));
      rlist = ucalloc(1+m, sizeof(int));
      rflag = ucalloc(1+m, sizeof(char));
      clist = ucalloc(1+n, sizeof(int));
      cflag = ucalloc(1+n, sizeof(char));
      /* initially the row list contains all rows, and the column list
         is empty */
      nr = m;
      for (i = 1; i <= m; i++) rlist[i] = i, rflag[i] = 1;
      nc = 0;
      for (j = 1; j <= n; j++) cflag[j] = 0;
      /* main preprocessing loop */
      while (nr > 0)
      {  /* perform preprocessing affected rows */
         for (k = 1; k <= nr; k++)
         {  i = rlist[k], rflag[i] = 0;
            if (prepro_row(i, 1e-5))
            {  ret = i;
               break;
            }
         }
         nr = 0;
         /* perform preprocessing affected columns */
         for (k = 1; k <= nc; k++)
         {  j = clist[k], cflag[j] = 0;
            prepro_col(j, 1e-5, 1e-5);
         }
         nc = 0;
      }
      /* free working arrays */
      ufree(f_min);
      ufree(f_max);
      ufree(rlist);
      ufree(rflag);
      ufree(clist);
      ufree(cflag);
      /* remove explicit "infinite" bounds */
      for (k = 1; k <= m+n; k++)
      {  if (lp->lb[k] == -DBL_MAX && lp->ub[k] == +DBL_MAX)
            lp->type[k] = 'F', lp->lb[k] = lp->ub[k] = 0.0;
         else if (lp->ub[k] == +DBL_MAX)
            lp->type[k] = 'L', lp->ub[k] = 0.0;
         else if (lp->lb[k] == -DBL_MAX)
            lp->type[k] = 'U', lp->lb[k] = 0.0;
         else
         {  /* if lower and upper bounds are close to each other, the
               corresponding variable can be fixed */
            if (lp->lb[k] != lp->ub[k])
            {  int t;
               t = check_rr(lp->lb[k], lp->ub[k], 1e-6);
               if (-1 <= t && t <= +1)
                  lp->lb[k] = lp->ub[k] = 0.5 * (lp->lb[k] + lp->ub[k]);
            }
            if (lp->lb[k] != lp->ub[k])
               lp->type[k] = 'D';
            else
               lp->type[k] = 'S';
         }
      }
      /* return to the calling program */
      return ret;
}

/* eof */
