/* glpapi/glp_set_row_name.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_row_name - assign (change) row name.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_set_row_name(LPI *lp, int i, char *name);
--
-- *Description*
--
-- The routine glp_set_row_name assigns the given symbolic name to the
-- i-th row (auxiliary variable).
--
-- If the parameter name is NULL, the routine just erases an existing
-- name of the i-th row.
--
-- *Complexity*
--
-- Time complexity is O(log m'), where m' is number of named rows in the
-- problem object. */

void glp_set_row_name(LPI *lp, int i, char *name)
{     if (!(1 <= i && i <= lp->m))
         fault("glp_set_row_name: i = %d; invalid row number", i);
      /* erase existing name */
      if (lp->row[i]->name != NULL)
      {  AVLNODE *node;
         node = find_by_key(lp->row_tree, lp->row[i]->name);
         insist(node != NULL);
         delete_node(lp->row_tree, node);
         delete_str(lp->row[i]->name), lp->row[i]->name = NULL;
      }
      /* assign new name */
      if (name != NULL)
      {  AVLNODE *node;
         if (glp_check_name(name))
            fault("glp_set_row_name: invalid row name");
         lp->row[i]->name = create_str(lp->str_pool);
         set_str(lp->row[i]->name, name);
         if (find_by_key(lp->row_tree, lp->row[i]->name) != NULL)
            fault("glp_set_row_name: duplicate row name `%s'", name);
         node = insert_by_key(lp->row_tree, lp->row[i]->name);
         node->link = lp->row[i];
      }
      return;
}

/* eof */
