/* glpapi/glp_set_row_bnds.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_row_bnds - set row bounds.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- void glp_set_row_bnds(LPI *lp, int i, int type, double lb,
--    double ub);
--
-- *Description*
--
-- The routine glp_set_row_bnds sets (changes) type and bounds of the
-- i-th row.
--
-- Parameters type, lb, and ub specify respectively the type, the lower
-- bound, and the upper bound, which should be set for the i-th row:
--
--    Type        Bounds            Note
--    ----------------------------------------
--    'F'   -inf <  x <  +inf   free variable
--    'L'     lb <= x <  +inf   lower bound
--    'U'   -inf <  x <=  ub    upper bound
--    'D'     lb <= x <=  ub    double bound
--    'S'           x  =  lb    fixed variable
--
-- where x is the corresponding auxiliary variable.
--
-- If the row has no lower bound, the parameter lb is ignored. If the
-- row has no upper bound, the parameter ub is ignored. If the row is
-- of fixed type, the parameter lb is used, and the parameter ub is
-- ignored. */

void glp_set_row_bnds(LPI *lp, int i, int type, double lb, double ub)
{     if (!(1 <= i && i <= lp->m))
         fault("glp_set_row_bnds: i = %d; invalid row number", i);
      lp->row[i]->type = type;
      switch (type)
      {  case 'F':
            lp->row[i]->lb = lp->row[i]->ub = 0.0;
            break;
         case 'L':
            lp->row[i]->lb = lb, lp->row[i]->ub = 0.0;
            break;
         case 'U':
            lp->row[i]->lb = 0.0, lp->row[i]->ub = ub;
            break;
         case 'D':
            lp->row[i]->lb = lb, lp->row[i]->ub = ub;
            break;
         case 'S':
            lp->row[i]->lb = lp->row[i]->ub = lb;
            break;
         default:
            fault("glp_set_row_bnds: type = %d; invalid row type",
               type);
      }
      return;
}

/* eof */
