/* glprsm/scale_rsm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- scale_rsm - scale problem components in RSM block.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void scale_rsm(RSM *rsm, double R[], double S[]);
--
-- *Description*
--
-- The routine scale_rsm() performs scaling problem components in the
-- RSM block, which the parameter rsm points to. These components are
-- bounds of variables and constraint coefficients.
--
-- The array R specifies a row scaling matrix. Diagonal elements of this
-- matrix should be placed in locations R[1], R[2], ..., R[m], where m
-- is number of rows (auxiliary variables). R may be NULL, in which case
-- rows are not scaled as if R would be unity matrix.
--
-- The array S specifies a column scaling matrix. Diagonal elements of
-- this matrix should be placed in locations S[1], S[2], ..., S[n],
-- where n is number of columns (structural variables). S may be NULL,
-- in which case columns are not scaled as if S would be unity matrix.
--
-- The purpose of scaling is to replace the original constraint matrix
-- A by the scaled matrix A' = R * A * S. */

void scale_rsm(RSM *rsm, double R[], double S[])
{     int m = rsm->m, n = rsm->n, i, j, k;
      /* scale bounds of auxiliary variables */
      if (R != NULL)
      {  for (i = 1; i <= m; i++)
         {  k = i;
            rsm->lb[k] *= R[i];
            rsm->ub[k] *= R[i];
         }
      }
      /* scale bounds of structural variables */
      if (S != NULL)
      {  for (j = 1; j <= n; j++)
         {  k = m + j;
            rsm->lb[k] /= S[j];
            rsm->ub[k] /= S[j];
         }
      }
      /* scale the augmented constraint matrix (unity submatrix is not
         changed) */
      if (!(R == NULL && S == NULL))
      {  for (j = 1; j <= n; j++)
         {  ELEM *e;
            k = m + j;
            for (e = rsm->A->col[k]; e != NULL; e = e->col)
            {  i = e->i;
               if (R != NULL) e->val *= R[i];
               if (S != NULL) e->val *= S[j];
            }
         }
      }
      return;
}

/* eof */
