/* glpmat/delete_lines.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- delete_lines() -- delete rows and columns from sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *delete_lines(MAT *A, int rs[], int cs[]);
--
-- *Description*
--
-- The routine delete_lines() deletes the specified subsets of rows and
-- columns from the sparse matrix A.
--
-- The array rs specifies a subset of rows and should have at least 1+m
-- locations, where m is number of rows in the matrix A before the
-- operation. The location rs[0] is not used. If rs[i] is non-zero, the
-- deletes the i-th row from the matrix. Otherwise, if rs[i] is zero,
-- the i-th row is kept.
--
-- The array cs specifies a subset of columns and should have at least
-- 1+n locations, where n is number of columns in the matrix A before
-- the operation. The location cs[0] is not used. If cs[j] is non-zero,
-- the routine deletes the j-th column from the matrix. Otherwise, if
-- cs[j] is zero, the j-th column is kept.
--
-- It is allowed to specify rs = NULL or cs = NULL, in which case the
-- subset of rows or columns is assumed to be empty.
--
-- *Complexity*
--
-- Independently on how many rows and columns should be deleted the
-- time complexity is O(nz), where nz is the number of elements in the
-- matrix A before the operation.
--
-- *Returns*
--
-- The routine delete_lines() returns a pointer to the matrix A. */

MAT *delete_lines(MAT *A, int rs[], int cs[])
{     ELEM *e;
      int m, n, i, j;
      clear_lines(A, rs, cs);
      /* adjust the row list */
      if (rs != NULL)
      {  m = 0;
         for (i = 1; i <= A->m; i++)
         {  if (!rs[i])
            {  m++;
               A->row[m] = A->row[i];
               for (e = A->row[m]; e != NULL; e = e->row) e->i = m;
            }
         }
         A->m = m;
      }
      /* adjust the column list */
      if (cs != NULL)
      {  n = 0;
         for (j = 1; j <= A->n; j++)
         {  if (!cs[j])
            {  n++;
               A->col[n] = A->col[j];
               for (e = A->col[n]; e != NULL; e = e->col) e->j = n;
            }
         }
         A->n = n;
      }
      return A;
}

/* eof */
