/* glpapi/glp_read_lpm1.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpk.h"
#include "glplang.h"

/*----------------------------------------------------------------------
-- glp_read_lpm1 - read linear programming model written in GLPK/L.
--
-- *Synopsis*
--
-- #include "glpk.h:
-- LPI *glp_read_lpm1(char *infile, char *outfile);
--
-- *Description*
--
-- The glp_read_lpm1 routine reads an LP model written in the modeling
-- language GLPK/L from a text file whose name is infile.
--
-- If the parameter outfile is not NULL, the routine writes generated
-- LP/MIP problem to a text file whose name is outfile.
--
-- For detailed description of GLPK/L modeling language see the program
-- documentation.
--
-- *Returns*
--
-- If the operation was successful, the routine returns a pointer to the
-- created problem object. Otherwise the routine returns NULL. */

LPI *glp_read_lpm1(char *infile, char *outfile)
{     LPI *lp = NULL;
      struct prob *prob = NULL;
      int m, n, i, j;
      /* initialize the language processor environment */
      if (initialize(infile) != 0) goto fail;
      /* set error handling */
      pdb->flag = 1;
      if (setjmp(pdb->jump)) goto fail;
      /* parse model description */
      load_model();
      /* output generated LP/MIP problem to the specified file in plain
         text format (if required) */
      if (outfile != NULL)
      {  int ret;
         ret = gener_lp(outfile);
         if (ret != 0) goto fail;
      }
      /* create data structure for generating LP/MIP */
      prob = create_prob();
      m = prob->m;
      n = prob->n;
      /* create problem instance */
      lp = glp_create_prob(pdb->model_name);
      /* create columns that correspond model variables */
      for (j = 1; j <= n; j++)
      {  VAR *var = prob->memb[m+j]->link;
         char *name = gener_name(prob, m+j);
         glp_new_col(lp, name);
         if (var->kind) glp_set_col_kind(lp, j, 'I');
         glp_set_col_bnds(lp, j, var->type, var->lb, var->ub);
      }
      /* create rows that correspond model constraints; build the
         constraint matrix */
      for (i = 1; i <= m; i++)
      {  CONS *cons = prob->memb[i]->link;
         char *name = gener_name(prob, i);
         struct elem *form, *e;
         glp_new_row(lp, name);
         form = build_form(prob, i);
         if (form == NULL) goto fail;
         for (e = form; e != NULL; e = e->next)
         {  if (e->j == 0)
            {  if (cons->type == 'F')
               {  error("glp_read_lpm1: free row `%s' has constant term"
                     , name);
                  goto fail;
               }
               cons->lb -= e->val, cons->ub -= e->val;
            }
            else
               glp_new_aij(lp, i, e->j, e->val);
            /* set coefficient of the objective function */
            if (prob->obj_row == i)
               glp_set_obj_coef(lp, e->j, e->val);
         }
         erase_form(prob, form);
         glp_set_row_bnds(lp, i, cons->type, cons->lb, cons->ub);
      }
      /* set the objective function */
      glp_set_obj_sense(lp, prob->obj_dir);
      /* free auxiliary data structure */
      delete_prob(prob);
      /* terminate the language processor environment */
      terminate();
      /* model has been read successfully */
      return lp;
fail: /* the operation failed */
      error("glp_read_lpm1: processing terminated due to errors");
      if (lp != NULL) glp_delete_prob(lp);
      if (prob != NULL) delete_prob(prob);
      if (pdb != NULL) terminate();
      return NULL;
}

/* eof */
