/* glprsm/create_rsm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <math.h>
#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- create_rsm - create revised simplex method common block.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- RSM *create_rsm(LPI *lp, int form);
--
-- *Description*
--
-- The routine create_rsm() obtains information from LP problem object,
-- which the parameter lp points to, and creates a common block used by
-- the revised simplex method routines.
--
-- The parameter form specifies the form of representation of the basis
-- matrix which should be used constructed:
--
-- 0 - EFI;
-- 1 - RFI + Bartels-Golub updating technique;
-- 2 - RFI + Forrest-Tomlin updating technique.
--
-- The initial basis is standard (all auxiliary variables are basic and
-- all structural variables are non-basic).
--
-- Being created the representation of the basis matrix for the initial
-- standard basis is valid, therefore no reinversion is needed.
--
-- *Returns*
--
-- The routine create_rsm() returns a pointer to the created block. */

RSM *create_rsm(LPI *lp, int form)
{     RSM *rsm;
      int m = glp_get_num_rows(lp), n = glp_get_num_cols(lp), i, j, k;
      if (m == 0) fault("create_rsm: problem has no rows");
      if (n == 0) fault("create_rsm: problem has no columns");
      /* allocate RSM structure */
      rsm = umalloc(sizeof(RSM));
      rsm->m = m;
      rsm->n = n;
      rsm->type = ucalloc(1+m+n, sizeof(int));
      rsm->lb = ucalloc(1+m+n, sizeof(double));
      rsm->ub = ucalloc(1+m+n, sizeof(double));
      rsm->A = create_mat(m, m+n);
      rsm->posx = ucalloc(1+m+n, sizeof(int));
      rsm->indb = ucalloc(1+m, sizeof(int));
      rsm->indn = ucalloc(1+n, sizeof(int));
      rsm->tagn = ucalloc(1+n, sizeof(int));
      switch (form)
      {  case 0:
            /* use EFI */
            rsm->efi = create_efi(m);
            rsm->rfi = NULL;
            break;
         case 1:
            /* use RFI + Bartels & Golub updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_BG;
            break;
         case 2:
            /* use RFI + Forrest & Tomlin updating technique */
            rsm->efi = NULL;
            rsm->rfi = create_rfi(m);
            rsm->rfi->tech = RFI_FT;
            break;
         default:
            fault("create_rsm: form = %d; invalid parameter", form);
      }
      rsm->iter = 0;
      /* obtain types and bounds of rows */
      for (i = 1; i <= m; i++)
      {  glp_get_row_bnds(lp, i, &rsm->type[i], &rsm->lb[i],
            &rsm->ub[i]);
      }
      /* obtain types and bounds of columns */
      for (j = 1; j <= n; j++)
      {  glp_get_col_bnds(lp, j, &rsm->type[m+j], &rsm->lb[m+j],
            &rsm->ub[m+j]);
      }
      /* build the expanded matrix A = (I | -A'), where I is the unity
         matrix, A' is the original matrix of constraint coefficients */
      {  int *cn = ucalloc(1+n, sizeof(int));
         double *ai = ucalloc(1+n, sizeof(double));
         for (i = 1; i <= m; i++)
         {  int nz = glp_get_row_coef(lp, i, cn, ai), t;
            new_elem(rsm->A, i, i, +1.0);
            for (t = 1; t <= nz; t++)
            {  if (ai[t] != 0.0)
                  new_elem(rsm->A, i, m + cn[t], - ai[t]);
            }
         }
         ufree(cn);
         ufree(ai);
      }
      /* the constraint matrix should have no multiplets */
      if (check_mplets(rsm->A))
         fault("create_rsm: constraint matrix has multiplets");
      /* construct standard initial basis (all auxiliary variables are
         basic and all structural variables are non-basic; in this case
         B = I and N = -A') */
      for (i = 1; i <= m; i++)
      {  k = i; /* x[k] = xB[i] */
         rsm->posx[k] = +i;
         rsm->indb[i] =  k;
      }
      for (j = 1; j <= n; j++)
      {  k = m+j; /* x[k] = xN[j] */
         rsm->posx[k] = -j;
         rsm->indn[j] =  k;
         switch (rsm->type[k])
         {  case 'F':
               rsm->tagn[j] = 'F'; break;
            case 'L':
               rsm->tagn[j] = 'L'; break;
            case 'U':
               rsm->tagn[j] = 'U'; break;
            case 'D':
               rsm->tagn[j] =
                  (fabs(rsm->lb[k]) <= fabs(rsm->ub[k]) ? 'L' : 'U');
               break;
            case 'S':
               rsm->tagn[j] = 'S'; break;
            default:
               insist(rsm->type[k] != rsm->type[k]);
         }
      }
      /* check RSM for correctness */
      check_rsm(rsm);
      /* return to the calling program */
      return rsm;
}

/* eof */
