/* glpmat/save_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <errno.h>
#include <float.h>
#include <stdio.h>
#include <string.h>
#include "glpmat.h"

/*----------------------------------------------------------------------
-- save_mat - write sparse matrix to text file using plain format.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- int save_mat(MAT *A, char *fname);
--
-- *Description*
--
-- The save_mat routine writes the sparse matrix A to the text file,
-- whose name is the character string fname, using the following plain
-- format:
--
--    MAT      \n
--    m n nz   \n
--    i j aij  \n    |
--    . . .          |-> nz lines
--    i j aij  \n    |
--    END      \n
--
-- where m is number of rows, n is number of columns, nz is number of
-- matrix elements, i is row number, j is column number, aij is value
-- of matrix elements. Total number of lines is nz+3.
--
-- The save_mat routine writes the matrix in row-wise manner. In order
-- that elements follow in the order of increasing column numbers the
-- sort_mat routine may be used before.
--
-- *Returns*
--
-- If the operation was successful, the save_mat routine returns zero.
-- Otherwise the routine returns non-zero. */

int save_mat(MAT *A, char *fname)
{     FILE *fp;
      int nz, i;
      print("save_mat: writing sparse matrix to `%s'...", fname);
      fp = fopen(fname, "w");
      if (fp == NULL)
      {  error("save_mat: unable to create `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fprintf(fp, "MAT\n");
      nz = count_nz(A, 0);
      fprintf(fp, "%d %d %d\n", A->m, A->n, nz);
      for (i = 1; i <= A->m; i++)
      {  ELEM *e;
         for (e = A->row[i]; e != NULL; e = e->row)
            fprintf(fp, "%d %d %.*g\n", e->i, e->j, DBL_DIG, e->val);
      }
      fprintf(fp, "END\n");
      fflush(fp);
      if (ferror(fp))
      {  error("save_mat: writing error on `%s' - %s", fname,
            strerror(errno));
         goto fail;
      }
      fclose(fp);
      return 0;
fail: if (fp != NULL) fclose(fp);
      return 1;
}

/* eof */
