/* glpimg/set_imgval.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpimg.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- set_imgval.c - set pixel to given value.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- void set_imgval(IMG *img, int x, int y, int val);
--
-- *Description*
--
-- The set_imgval routine sets the pixel of the raster image which has
-- absolute coordinates (x,y) to the given value val.
--
-- If the pixel (x,y) is out of the image, the routine does nothing. */

void set_imgval(IMG *img, int x, int y, int val)
{     unsigned char *byte;
      if (!(0 <= x && x < img->xsize && 0 <= y && y < img->ysize))
         goto skip;
      switch (img->type)
      {  case IMG_2: /* 2 colors (1 bit per pixel) */
            byte = img->scan + img->len * y + (x >> 3);
            switch (x & 7)
            {  case 0:
                  *byte &= 0x7F, *byte |= (char)((val & 1) << 7);
                  break;
               case 1:
                  *byte &= 0xBF, *byte |= (char)((val & 1) << 6);
                  break;
               case 2:
                  *byte &= 0xDF, *byte |= (char)((val & 1) << 5);
                  break;
                case 3:
                  *byte &= 0xEF, *byte |= (char)((val & 1) << 4);
                  break;
                case 4:
                  *byte &= 0xF7, *byte |= (char)((val & 1) << 3);
                  break;
                case 5:
                  *byte &= 0xFB, *byte |= (char)((val & 1) << 2);
                  break;
                case 6:
                  *byte &= 0xFD, *byte |= (char)((val & 1) << 1);
                  break;
                case 7:
                  *byte &= 0xFE, *byte |= (char)((val & 1) << 0);
                  break;
            }
            break;
         case IMG_4: /* 4 colors (2 bits per pixel) */
            byte = img->scan + img->len * y + (x >> 2);
            switch (x & 3)
            {  case 0:
                  *byte &= 0x3F, *byte |= (char)((val & 3) << 6);
                  break;
               case 1:
                  *byte &= 0xCF, *byte |= (char)((val & 3) << 4);
                  break;
               case 2:
                  *byte &= 0xF3, *byte |= (char)((val & 3) << 2);
                  break;
               case 3:
                  *byte &= 0xFC, *byte |= (char)((val & 3) << 0);
                  break;
            }
            break;
         case IMG_16: /* 16 colors (4 bits per pixel) */
            byte = img->scan + img->len * y + (x >> 1);
            switch (x & 1)
            {  case 0:
                  *byte &= 0x0F, *byte |= (char)((val & 15) << 4);
                  break;
               case 1:
                  *byte &= 0xF0, *byte |= (char)((val & 15) << 0);
                  break;
            }
            break;
         case IMG_256: /* 256 colors (8 bits per pixel) */
            byte = img->scan + img->len * y + x;
            *byte = (char)(val & 255);
            break;
         default:
            insist(img->type != img->type);
      }
skip: return;
}

/* eof */
