/* glpmat/aat_numb.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- aat_numb - compute matrix product S = A * D * A'; numeric phase
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- MAT *aat_numb(MAT *S, MAT *A, double D[], double work[]);
--
-- *Description*
--
-- The aat_numb computes the matrix product S = A * D * A' numerically,
-- assuming that the pattern of the matrix S (i.e. distribution of its
-- non-zero elements) is known. (Symbolic phase of this operation can
-- be performed by means of the aat_symb routine.)
--
-- Input matrix A remains unchanged on exit.
--
-- The array D specifies diagonal matrix D of order n, where n is number
-- of columns of the matrix A. Diagonal elements of D should be placed
-- in locations D[1], D[2], ..., D[n]. If the parameter D is NULL, it is
-- assumed that D is unity matrix.
--
-- The resultant matrix S should not coincide with the matrix A. Order
-- of S should be equal number of rows of A. On entry the distribution
-- of non-zero elements of S should be correct, however, numeric values
-- of elements are not essential (note that the aat_symb routine stores
-- only upper triangular part of S, because it is a symmetric matrix).
-- On exit the matrix S has the same pattern (some its elements may be
-- equal to zero because of numeric cancellation, i.e. this matrix may
-- contain explicit zeros).
--
-- The auxiliary array work should contain at least 1+n elements, where
-- n is number of columns of the matrix A. If this parameter is NULL,
-- the aat_numb routine automatically allocates and frees the working
-- array.
--
-- *Returns*
--
-- The aat_numb routine returns a pointer to the matrix S. */

MAT *aat_numb(MAT *S, MAT *A, double D[], double _work[])
{     ELEM *e, *ee;
      int i, j;
      double *work = _work;
      if (S == A)
         fault("aat_numb: invalid specification of resultant matrix");
      if (!(S->m == S->n && S->m == A->m))
         fault("aat_numb: inconsistent dimension; product undefined");
      if (_work == NULL) work = ucalloc(1+A->n, sizeof(double));
      for (j = 1; j <= A->n; j++) work[j] = 0.0;
      for (i = 1; i <= S->m; i++)
      {  /* work := i-th row of A */
         for (e = A->row[i]; e != NULL; e = e->row) work[e->j] = e->val;
         /* compute i-th row of S */
         for (e = S->row[i]; e != NULL; e = e->row)
         {  /* s[i,j] = a[i,*] * a[j,*] */
            double sum = 0.0;
            if (D == NULL)
            {  for (ee = A->row[e->j]; ee != NULL; ee = ee->row)
                  sum += work[ee->j] * ee->val;
            }
            else
            {  for (ee = A->row[e->j]; ee != NULL; ee = ee->row)
                  sum += work[ee->j] * D[ee->j] * ee->val;
            }
            e->val = sum;
         }
         /* clear working array */
         for (e = A->row[i]; e != NULL; e = e->row) work[e->j] = 0.0;
      }
      if (_work == NULL) ufree(work);
      return S;
}

/* eof */
