/* glpapi/glp_set_cpar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_set_cpar - set value of text control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_set_cpar(char *name, char *val);
--
-- *Description*
--
-- The glp_set_cpar routine assigns the character string val to the text
-- control parameter with the given name.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid;
-- 2 - the given value (character string) is invalid. */

int glp_set_cpar(char *name, char *val)
{     if (strcmp(name, "fn_gener") == 0)
      {  /* file name to output generated LP/MIP problem */
         if (strlen(val) > GLP_MAX_NAME) return 2;
         if (glp->fn_gener == NULL)
            glp->fn_gener = create_str(glp->str_pool);
         if (val[0] == '\0')
         {  delete_str(glp->fn_gener);
            glp->fn_gener = NULL;
         }
         else
            set_str(glp->fn_gener, val);
      }
      else if (strcmp(name, "mps_bnd_name") == 0)
      {  /* the name of bound vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_bnd_name, val);
      }
      else if (strcmp(name, "mps_obj_name") == 0)
      {  /* the name of the objective function */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_obj_name, val);
      }
      else if (strcmp(name, "mps_rhs_name") == 0)
      {  /* the name of the right-hand side (RHS) vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_rhs_name, val);
      }
      else if (strcmp(name, "mps_rng_name") == 0)
      {  /* the name of the range vector */
         if (strlen(val) > 8) return 2;
         strcpy(glp->mps_rng_name, val);
      }
      else if (strcmp(name, "obj_row") == 0)
      {  /* the name of the objective function */
         if (val[0] == '\0')
            glp->obj_row = NULL;
         else
         {  STR *row_name;
            AVLNODE *node;
            if (glp_check_name(val)) return 2;
            row_name = set_str(create_str(glp->str_pool), val);
            node = find_by_key(glp->row_tab, row_name);
            delete_str(row_name);
            if (node == NULL) return 2; /* row not found */
            glp->obj_row = node->link;
         }
         /* nullify the constant term, because the objective function
            has been changed */
         glp->c0 = 0.0;
      }
      else if (strcmp(name, "problem") == 0)
      {  /* the name of problem */
         if (glp_check_name(val)) return 2;
         set_str(glp->problem, val);
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/* eof */
