/* glpipm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "glpk.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- This program is a stand-alone LP solver based on the primal-dual
-- interior point method.
----------------------------------------------------------------------*/

static char *version =
   "GLPIPM -- GLPK Interior Point LP Solver, Version 2.3";

static char *dat_fname = "";
/* name of the input text file which contains LP problem data in MPS or
   LPM format */

static int dat_fmt = 0;
/* format of the input text file (0 means MPS, 1 means LPM) */

static char *out_fname = "";
/* the name of the output text file to which the final solution should
   be sent using printable format (optional) */

/*----------------------------------------------------------------------
-- display_help - display help.
--
-- This routine displays help information about the program as it is
-- required by the GNU Coding Standards. */

static void display_help(char *my_name)
{     print("Usage: %s [options...] data-file", my_name);
      print("");
      print("Options:");
      print("   --mps             read LP data using MPS format (defaul"
         "t)");
      print("   --lpm             read LP data using LPM format");
      print("   --min             objective function should be minimize"
         "d (default)");
      print("   --max             objective function should be maximize"
         "d");
      print("   --any             any feasible solution is required");
      print("   --fin             final solution is required (default)")
         ;
      print("   -o filename, --output filename");
      print("                     send solution to filename using print"
         "able format");
      print("   --sum             enable multiplets in constraint matri"
         "x (will be");
      print("                     replaced by their sum)");
      print("   --nosum           disable multiplets in constraint matr"
         "ix (default)");
      print("   -h, --help        display this help information and exi"
         "t");
      print("   -v, --version     display program version and exit");
      print("");
      print("Please, report bugs to <bug-glpk@gnu.org>");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- display_version - display version.
--
-- This routine displays version information for the program as it is
-- required by the GNU Coding Standards. */

static void display_version(void)
{     print("%s", version);
      print("Copyright (C) 2000, 2001 Andrew Makhorin <mao@gnu.org>");
      print("This program is free software; you may redistribute it und"
         "er the terms of");
      print("the GNU General Public License. This program has absolutel"
         "y no warranty.");
      exit(EXIT_SUCCESS);
      /* no return */
}

/*----------------------------------------------------------------------
-- process_cmdline - process command line parameters.
--
-- This routine processes parameters specified in command line. */

static void process_cmdline(int argc, char *argv[])
{     int k;
      for (k = 1; k < argc; k++)
      {  if (strcmp(argv[k], "--mps") == 0)
            dat_fmt = 0;
         else if (strcmp(argv[k], "--lpm") == 0)
            dat_fmt = 1;
         else if (strcmp(argv[k], "--min") == 0)
            glp_set_ipar("obj_dir", GLP_MIN);
         else if (strcmp(argv[k], "--max") == 0)
            glp_set_ipar("obj_dir", GLP_MAX);
         else if (strcmp(argv[k], "--any") == 0)
            glp_set_ipar("option", GLP_ANY);
         else if (strcmp(argv[k], "--fin") == 0)
            glp_set_ipar("option", GLP_FIN);
         else if (strcmp(argv[k], "-o") == 0 ||
                  strcmp(argv[k], "--output") == 0)
         {  k++;
            if (k == argc)
            {  error("No output file name specified");
               exit(EXIT_FAILURE);
            }
            if (out_fname[0] != '\0')
            {  error("Only one output file allowed");
               exit(EXIT_FAILURE);
            }
            out_fname = argv[k];
         }
         else if (strcmp(argv[k], "--sum") == 0)
            glp_set_ipar("sum_aij", GLP_YES);
         else if (strcmp(argv[k], "--nosum") == 0)
            glp_set_ipar("sum_aij", GLP_NO);
         else if (strcmp(argv[k], "-h") == 0 ||
                  strcmp(argv[k], "--help") == 0)
            display_help(argv[0]);
         else if (strcmp(argv[k], "-v") == 0 ||
                  strcmp(argv[k], "--version") == 0)
            display_version();
         else if (argv[k][0] == '-' ||
                 (argv[k][0] == '-' && argv[k][1] == '-'))
         {  error("Invalid option `%s'; try %s --help", argv[k],
               argv[0]);
            exit(EXIT_FAILURE);
         }
         else
         {  if (dat_fname[0] != '\0')
            {  error("Only one input file allowed");
               exit(EXIT_FAILURE);
            }
            dat_fname = argv[k];
         }
      }
      return;
}

/*----------------------------------------------------------------------
-- main - main routine.
--
-- This main routine is called by the control program and manages the
-- solving process. */

int main(int argc, char *argv[])
{     int option, ret;
      clock_t t_start;
      /* initialize GLPK API */
      glp_initialize();
      /* process command line parameters */
      process_cmdline(argc, argv);
      /* read LP problem data from the input file */
      if (dat_fname[0] == '\0')
      {  error("No input file name specified");
         exit(EXIT_FAILURE);
      }
      glp_get_ipar("option", &option);
      if (dat_fmt == 0)
      {  if (glp_read_mps(dat_fname) != 0)
         {  error("Can't read MPS file");
            exit(EXIT_FAILURE);
         }
      }
      else
      {  if (glp_read_lpm(dat_fname) != 0)
         {  error("Can't read LPM file");
            exit(EXIT_FAILURE);
         }
      }
      glp_set_ipar("option", option);
      /* solve LP problem */
      t_start = clock();
      ret = glp_interior();
      switch (ret)
      {  case 0:
            break;
         case 1:
            error("Can't solve the problem because of errors");
            exit(EXIT_FAILURE);
         case 2:
            error("Optimization terminated because of solver failure");
            break;
         default:
            insist(ret != ret);
      }
      /* display statistics */
      print("Time used: %.1f secs",
         (double)(clock() - t_start) / (double)CLOCKS_PER_SEC);
      print("Memory used: %.1fM (%d bytes)",
         (double)mem_tpeak / (1024.0 * 1024.0), mem_tpeak);
      /* write final solution found by the solver */
      if (out_fname[0] != '\0' && glp_print_sol(out_fname) != 0)
      {  error("Can't write solution");
         exit(EXIT_FAILURE);
      }
      /* terminate GLPK API */
      glp_terminate();
      /* exit to the control program */
      return 0;
}

/* eof */
