/* glprsm/init_dvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- init_dvec - initialize the vector delta.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void init_dvec(RSM *rsm, double dvec[]);
--
-- *Description*
--
-- The init_dvec computes the vector delta for the initial basis and
-- stores its elements into locations dvec[1], ..., dvec[m].
--
-- Initial basis is a basis where all auxiliary variables are basic and
-- all structural variables are non-basic. In this special case B = I
-- and N = A^, therefore alfa[i,j] = a^[i,j], where a^[i,j] are elements
-- of the original constraint matrix A^, that allows using the formula
--
--    dvec[i] = 1 + alfa[i,1]^2 + ... + alfa[i,n]^2
--
-- directly for computing the initial vector delta. */

void init_dvec(RSM *rsm, double dvec[])
{     ELEM *e;
      int m = rsm->m, n = rsm->n, i, j;
      for (i = 1; i <= m; i++) dvec[i] = 1.0;
      for (j = 1; j <= n; j++)
      {  for (e = rsm->A->col[m+j]; e != NULL; e = e->col)
            dvec[e->i] += e->val * e->val;
      }
      return;
}

/* eof */
