/* glprsm/exact_gvec.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- exact_gvec - compute exact value of gamma[j].
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- double exact_gvec(RSM *rsm, int j);
--
-- *Description*
--
-- The exact_gvec routine computes exact value of gamma[j] using its
-- definition:
--
--    gamma[j] = 1 + alfa[1,j]^2 + ... + alfa[m,j]^2
--
-- where alfa[i,j] is the element of the current simplex table placed
-- in i-th row and j-th column.
--
-- This operation is extremely inefficient and may be used only for
-- debugging purposes.
--
-- *Returns*
--
-- The exact_gvec routine returns the computed value of gamma[j]. */

double exact_gvec(RSM *rsm, int j)
{     int m = rsm->m, n = rsm->n, i;
      double *aj, t;
      insist(1 <= j && j <= n);
      aj = ucalloc(1+m, sizeof(double));
      eval_col(rsm, j, aj, 0);
      t = 1.0;
      for (i = 1; i <= m; i++) t += aj[i] * aj[i];
      ufree(aj);
      return t;
}

/* eof */
