/* glprsm/eval_row.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_row - compute row of simplex table.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_row(RSM *rsm, double zeta[], double ai[]);
--
-- *Description*
--
-- The eval_row routine computes i-th row of the simplex table, i.e.
-- i-th row of the matrix A~ = -inv(B)*N, and stores its elements into
-- locations ai[1], ..., ai[n] respectively.
--
-- The array zeta should contain i-th row of the inverse inv(B), where
-- B is the current basis matrix, computed by means of the eval_zeta
-- routine. This array is not changed on exit.
--
-- The i-th row of the simplex table is computed using the following
-- formula:
--
--    a~[i] = - N' * zeta,
--
-- where N' is a matrix transposed to N, N is the submatrix formed by
-- non-basic columns of the expanded matrix A, zeta is i-th row of the
-- inverse inv(B). */

void eval_row(RSM *rsm, double zeta[], double ai[])
{     ELEM *e;
      int n = rsm->n, j , k;
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         ai[j] = 0.0;
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            ai[j] -= e->val * zeta[e->i];
      }
      return;
}

/* eof */
