/* glprsm/eval_cbar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- eval_cbar - compute reduced costs of non-basic variables.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- void eval_cbar(RSM *rsm, double c[], double pi[], double cbar[]);
--
-- *Description*
--
-- The eval_cbar routine computes reduced costs d = (d_1, ..., d_n) of
-- non-basic variables that correspond to the current basis solution and
-- stores d_1, ..., d_n to locations cbar[1], ..., cbar[n] respectively.
--
-- On entry the array c should contain the vector of coefficients of the
-- objective function in locations c[1], ..., c[m+n]. The array c is not
-- changed on exit.
--
-- On entry the array pi should contain the vector of simplex (Lagrange)
-- multipliers pi computed by means of the eval_pi routine for the same
-- vector c. The array pi is not changed on exit.
--
-- The vector d is computed using the following formula:
--
--    d[j] = cN[j] - pi' * N[j], j = 1, 2, ..., n,
--
-- where cN[j] is coefficient of the objective function at the variable
-- xN[j], pi is the vector of simplex multipliers, N[j] is the column of
-- the expanded constraint matrix A, which corresponds to the variable
-- xN[j]. */

void eval_cbar(RSM *rsm, double c[], double pi[], double cbar[])
{     ELEM *e;
      int n = rsm->n, j, k;
      /* cbar[j] = d[j] = cN[j] - pi * N[j] */
      for (j = 1; j <= n; j++)
      {  k = rsm->indn[j]; /* x[k] = xN[j] */
         cbar[j] = c[k];
         for (e = rsm->A->col[k]; e != NULL; e = e->col)
            cbar[j] -= pi[e->i] * e->val;
      }
      return;
}

/* eof */
