/* glprsm/change_b.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glprsm.h"

/*----------------------------------------------------------------------
-- change_b - change basis.
--
-- *Synopsis*
--
-- #include "glprsm.h"
-- int change_b(RSM *rsm, int p, int tagp, int q);
--
-- *Description*
--
-- The change_b routine changes the current basis replacing it by the
-- adjacent one. The routine takes the basic variable (xB)p
-- (1 <= p <= m) out the set of basic variables xB and brings instead
-- (xB)p the non-basic variables (xN)q (1 <= q <= n) into xB.
-- Correspondingly, the basic variable (xB)p replaces the non-basic
-- variable (xN)q in the set of non-basic variables xN.
--
-- The parameter tagp specifies to what subset xF, xL, xU, or xS the
-- basic variable (xB)p should be attributed after it has left the
-- basis. The value of tagp has the same meaning as the field rsm.tagn
-- (see the structure RSM in glprsm.h) and should be compatible with
-- the type of variable (xB)p.
--
-- The special case p < 0 means that the current basis is not changed,
-- but the non-basic variable (xN)q (which should be double-bounded
-- variable) just goes from its current bound to the opposite one. The
-- parameter tagp is ignored in this special case.
--
-- The change_b routine also replaces p-th column of the basis matrix
-- B by q-th column of the matrix N and updates the representation of B
-- by means of the update_b routine. Note that new p-th column of B is
-- passed implicitly; it is assumed that this column was saved before
-- by the eval_col routine. If the representation becomes too long or
-- inaccurate, the change_b routine automatically rebuilds it by means
-- of the invert_b routine.
--
-- One call to the change_b routine is considered as one iteration of
-- the simplex method.
--
-- *Returns*
--
-- If the representation of the basis matrix was not rebuilt or it was
-- rebuilt successfully, the change_b routine returns zero. Otherwise
-- the routine returns non-zero. On the latter case the calling program
-- should not use the representation until the basis matrix will be
-- corrected and the representation will be rebuilt anew. */

int change_b(RSM *rsm, int p, int tagp, int q)
{     int m = rsm->m, n = rsm->n, ret = 0, k, kp, kq;
      if (p < 0)
      {  /* xN[q] goes from the current bound to the opposite one */
         k = rsm->indn[q]; /* x[k] = xN[q] */
         insist(rsm->type[k] == 'D');
         insist(rsm->tagn[q] == 'L' || rsm->tagn[q] == 'U');
         rsm->tagn[q] = (rsm->tagn[q] == 'L') ? 'U' : 'L';
      }
      else
      {  /* xB[p] leaves the basis, xN[q] enters the basis */
         insist(1 <= p && p <= m);
         insist(1 <= q && q <= n);
         kp = rsm->indb[p]; kq = rsm->indn[q];
         rsm->posx[kq] = +p; rsm->posx[kp] = -q;
         rsm->indb[p] = kq; rsm->indn[q] = kp;
         rsm->tagn[q] = tagp;
         /* update representation of the basis matrix */
         ret = update_b(rsm, p);
         /* if representation became inaccurate or too long, reinvert
            the basis matrix */
         if (ret != 0) ret = invert_b(rsm);
      }
      /* check common block for correctness */
      check_rsm(rsm);
      /* increase iteration count */
      rsm->iter++;
      /* returns to the calling program */
      return ret;
}

/* eof */
