/* glprfi/build_rfi.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpgel.h"
#include "glprfi.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- build_rfi - build RFI for given basis matrix.
--
-- *Synopsis*
--
-- #include "glprfi.h"
-- int build_rfi(RFI *rfi, MAT *A, int indb[]);
--
-- *Description*
--
-- The build_rfi routine builds RFI for the given basis matrix B (see
-- below) in the form B = H*V. This is used when: (a) the current RFI
-- became inaccurate, (b) the current RFI requires too much memory, and
-- (c) the basis matrix B was completely changed.
--
-- The given basis matrix B should be specified implicitly by the matrix
-- A and the array indb. The matrix A should have m rows, where m is the
-- order of the basis matrix B. The array indb should specify a list of
-- column numbers of the matrix A, which form the matrix B. These column
-- numbers should be placed in locations indb[1], indb[2], ..., indb[m].
--
-- *Returns*
--
-- If the RFI has been built, the routine returns zero. Otherwise, the
-- routine returns non-zero. The latter case can happen if the matrix B
-- is numerically singular or ill conditioned; for details see the gel
-- routine. */

static RFI *_rfi;

static void func(int i, int p, double f)
{     /* add the next term to the eta-file H */
      RFI *rfi = _rfi;
      i = rfi->P->row[i], p = rfi->P->row[p];
      app_term(rfi->H, i, p, f);
      return;
}

#define maxtry 4

int build_rfi(RFI *rfi, MAT *A, int indb[])
{     static double tol[1+maxtry] = { 0.00, 0.01, 0.10, 0.40, 0.85 };
      int m = rfi->m, ret, try;
      double Vmax, Vbig, *rmax = rfi->col;
      _rfi = rfi;
      if (A->m != m)
         fault("build_rfi: invalid number of rows");
      for (try = 1; try <= maxtry; try++)
      {  int i, dum = 0;
         /* H := I */
         reset_eta(rfi->H);
         /* V := B */
         clear_mat(rfi->V);
         for (i = 1; i <= m; i++)
         {  ELEM *e;
            int k = indb[i]; /* i-th column of B is k-th column of A */
            if (!(1 <= k && k <= A->n))
               fault("build_rfi: invalid column list");
            for (e = A->col[k]; e != NULL; e = e->col)
            {  if (e->val != 0.0)
                  new_elem(rfi->V, e->i, i, e->val);
            }
         }
         /* P := I, Q := I */
         reset_per(rfi->P);
         reset_per(rfi->Q);
         /* factorize the matrix U = P*V*Q using gaussian elimination
            and accumulate elementary gaussian transformations in the
            eta-file H */
         ret = gel(rfi->V, rfi->P, rfi->Q, func, tol[try], 1e+10, &dum,
            &Vmax, &Vbig, rfi->rs, rfi->cs, rmax, rfi->work);
         if (ret == 0 || try == maxtry) break;
      }
      /* save size of the eta-file H */
      rfi->nzH0 = rfi->H->pool->count;
      /* save number of non-zeros of the matrix V */
      rfi->nzV0 = rfi->V->pool->count;
      /* clear transformed column flag */
      rfi->flag = 0;
      return ret;
}

/* eof */
