/* qmdupd.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- THIS CODE IS THE RESULT OF TRANSLATION OF THE FORTRAN SUBROUTINE
-- QMDUPD FROM THE BOOK:
--
-- ALAN GEORGE, JOSEPH W-H LIU. COMPUTER SOLUTION OF LARGE SPARSE
-- POSITIVE DEFINITE SYSTEMS. PRENTICE-HALL, 1981.
--
-- THE TRANSLATION HAS BEEN DONE WITH THE PERMISSION OF THE AUTHORS
-- OF THE ORIGINAL FORTRAN SUBROUTINE: ALAN GEORGE AND JOSEPH LIU,
-- UNIVERSITY OF WATERLOO, WATERLOO, ONTARIO, CANADA.
--
-- The translation was made by Andrew Makhorin <mao@mai2.rcnet.ru>,
-- <mao@gnu.org>, Department for Applied Informatics, Moscow Aviation
-- Institute, Moscow, Russia.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include "glpqmd.h"

/*----------------------------------------------------------------------
-- qmdupd - Quotient MD UPDate.
--
-- *Synopsis*
--
-- #include "glpqmd.h"
-- void qmdupd(int xadj[], int adjncy[], int *nlist, int list[],
--    int deg[], int qsize[], int qlink[], int marker[], int rchset[],
--    int nbrhd[]);
--
-- *Purpose*
--
-- This routine performs degree update for a set of nodes in the minimum
-- degree algorithm.
--
-- *Input parameters*
--
-- (xadj, adjncy) -
--          the adjancy structure;
-- (nlist, list) -
--          the list of nodes whose degree has to be updated.
--
-- *Updated parameters*
--
-- deg    - the degree vector;
-- qsize  - size of indistinguishable supernodes;
-- qlink  - linked list for indistinguishable nodes;
-- marker - used to mark those nodes in reach/nbrhd sets.
--
-- *Working parameters*
--
-- rchset - the reachable set;
-- nbrhd  - the neighborhood set.
--
-- *Program subroutines*
--
-- qmdmrg.
----------------------------------------------------------------------*/

void qmdupd(int xadj[], int adjncy[], int *_nlist, int list[],
      int deg[], int qsize[], int qlink[], int marker[], int rchset[],
      int nbrhd[])
{     int deg0, deg1, il, inhd, inode, irch, j, jstop, jstrt, mark,
         nabor, nhdsze, node, rchsze;
#     define nlist  (*_nlist)
      /* Find all eliminated supernodes that are adjacent to some nodes
         in the given list. Put them into (nhdsze, nbrhd). deg0 contains
         the number of nodes in the list. */
      if (nlist <= 0) return;
      deg0 = 0;
      nhdsze = 0;
      for (il = 1; il <= nlist; il++)
      {  node = list[il];
         deg0 += qsize[node];
         jstrt = xadj[node];
         jstop = xadj[node+1] - 1;
         for (j = jstrt; j <= jstop; j++)
         {  nabor = adjncy[j];
            if (marker[nabor] == 0 && deg[nabor] < 0)
            {  marker[nabor] = -1;
               nhdsze++;
               nbrhd[nhdsze] = nabor;
            }
         }
      }
      /* Merge indistinguishable nodes in the list by calling the
         subroutine qmdmrg. */
      if (nhdsze > 0)
         qmdmrg(xadj, adjncy, deg, qsize, qlink, marker, &deg0, &nhdsze,
            nbrhd, rchset, &nbrhd[nhdsze+1]);
      /* Find the new degrees of the nodes that have not been merged. */
      for (il = 1; il <= nlist; il++)
      {  node = list[il];
         mark = marker[node];
         if (mark == 0 || mark == 1)
         {  marker[node] = 2;
            qmdrch(&node, xadj, adjncy, deg, marker, &rchsze, rchset,
               &nhdsze, nbrhd);
            deg1 = deg0;
            if (rchsze > 0)
            {  for (irch = 1; irch <= rchsze; irch++)
               {  inode = rchset[irch];
                  deg1 += qsize[inode];
                  marker[inode] = 0;
               }
            }
            deg[node] = deg1 - 1;
            if (nhdsze > 0)
            {  for (inhd = 1; inhd <= nhdsze; inhd++)
               {  inode = nbrhd[inhd];
                  marker[inode] = 0;
               }
            }
         }
      }
      return;
#     undef nlist
}

/* eof */
