/* glpprob/load_lp.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpprob.h"

/*----------------------------------------------------------------------
-- load_lp - read LP data block from text file using plain format.
--
-- *Synopsis*
--
-- #include "glpprob.h"
-- LP *load_lp(char *fname);
--
-- *Description*
--
-- The load_lp routine reads an LP problem data block from the text
-- file, whose name is the character string fname, using plain format
-- (this format is described in comments to the routine save_lp; note
-- that the order of matrix elements may be arbitrary).
--
-- *Returns*
--
-- If the operation was successful, the load_lp routine returns a
-- pointer to the loaded LP data block. Otherwise the routine returns
-- NULL. */

LP *load_lp(char *fname)
{     DATA *ds;
      LP *lp = NULL;
      int m, n, nz, nc, i, j, k;
      double lb, ub, aij, cj;
      char str[255+1];
      print("load_lp: reading LP data block from `%s'...", fname);
      ds = open_data(fname);
      if (ds == NULL) goto fail;
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "LP"))
      {  error("%s:%d: marker `LP' not found", ds->fn, ds->cn);
         goto fail;
      }
      if (scan_data(ds, 'i', &m, 'i', &n, 'i', &nz, 's', str, 'i', &nc,
         0)) goto fail;
      if (m <= 0)
      {  error("%s:%d: invalid number of rows", ds->fn, ds->cn);
         goto fail;
      }
      if (n <= 0)
      {  error("%s:%d: invalid number of columns", ds->fn, ds->cn);
         goto fail;
      }
      if (nz < 0)
      {  error("%s:%d: invalid number of constraint coefficients",
            ds->fn, ds->cn);
         goto fail;
      }
      if (!(strcmp(str, "+") == 0 || strcmp(str, "-") == 0))
      {  error("%s:%d: invalid optimization direction flag", ds->fn,
            ds->cn);
         goto fail;
      }
      if (nc < 0)
      {  error("%s:%d: invalid number of coefficents of the objective f"
            "unction", ds->fn, ds->cn);
         goto fail;
      }
      lp = create_lp(m, n);
      lp->dir = str[0];
      for (k = 1; k <= m+n; k++)
      {  if (scan_data(ds, 's', str, 'e', &lb, 'e', &ub, 0)) goto fail;
         if (strcmp(str, "F") == 0)
            lp->type[k] = 'F', lp->lb[k] = lp->ub[k] = 0.0;
         else if (strcmp(str, "L") == 0)
            lp->type[k] = 'L', lp->lb[k] = lb, lp->ub[k] = 0.0;
         else if (strcmp(str, "U") == 0)
            lp->type[k] = 'U', lp->lb[k] = 0.0, lp->ub[k] = ub;
         else if (strcmp(str, "D") == 0)
            lp->type[k] = 'D', lp->lb[k] = lb, lp->ub[k] = ub;
         else if (strcmp(str, "S") == 0)
            lp->type[k] = 'S', lp->lb[k] = lp->ub[k] = lb;
         else
         {  error("%s:%d: invalid type of variable", ds->fn, ds->cn);
            goto fail;
         }
      }
      for (k = 1; k <= nz; k++)
      {  if (scan_data(ds, 'i', &i, 'i', &j, 'e', &aij, 0)) goto fail;
         if (!(1 <= i && i <= m))
         {  error("%s:%d: row index out of range", ds->fn, ds->cn);
            goto fail;
         }
         if (!(1 <= j && j <= n))
         {  error("%s:%d: column index out of range", ds->fn, ds->cn);
            goto fail;
         }
         new_elem(lp->A, i, j, aij);
      }
      for (k = 1; k <= nc; k++)
      {  if (scan_data(ds, 'i', &j, 'e', &cj, 0)) goto fail;
         if (!(0 <= j && j <= n))
         {  error("%s:%d: column index out of range", ds->fn, ds->cn);
            goto fail;
         }
         lp->c[j] = cj;
      }
      if (scan_data(ds, 's', str, 0)) goto fail;
      if (strcmp(str, "EOD"))
      {  error("%s:%d: marker `EOD' not found", ds->fn, ds->cn);
         goto fail;
      }
      close_data(ds);
      return lp;
fail: if (ds != NULL) close_data(ds);
      if (lp != NULL) delete_lp(lp);
      return NULL;
}

/* eof */
