/* glpmat/check_mat.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- check_mat - check sparse matrix for correctness.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- void check_mat(MAT *A);
--
-- *Description*
--
-- The check_mat routine checks the representation of the sparse matrix
-- A for correctness. In the case of error the routine displays an error
-- message and terminates a program. Should note that the matrix is
-- considered as correct if it has multiplets (i.e. elements that have
-- identical row and column numbers) and/or zeros (i.e. elements that
-- exist and have zero value).
--
-- This operation is extremely ineffecient. */

void check_mat(MAT *A)
{     ELEM *e, *ee;
      int i, j, flag;
      if (!(A->m > 0 && A->n > 0))
         fault("check_mat: invalid dimension");
      for (i = 1; i <= A->m; i++)
      {  flag = 0;
         for (e = A->row[i]; e != NULL; e = e->row)
         {  if (flag && e == A->row[i])
               fault("check_mat: row list has a cycle");
            if (e->i != i)
               fault("check_mat: element has wrong row number");
            flag = 1;
         }
      }
      for (j = 1; j <= A->n; j++)
      {  flag = 0;
         for (e = A->col[j]; e != NULL; e = e->col)
         {  if (flag && e == A->col[j])
               fault("check_mat: column list has a cycle");
            if (e->j != j)
               fault("check_mat: element has wrong column number");
            flag = 1;
         }
      }
      for (i = 1; i <= A->m; i++)
      {  for (e = A->row[i]; e != NULL; e = e->row)
         {  for (ee = A->col[e->j]; ee != NULL; ee = ee->col)
               if (ee == e) break;
            if (ee == NULL)
               fault("check_mat: element not found in column list");
         }
      }
      for (j = 1; j <= A->n; j++)
      {  for (e = A->col[j]; e != NULL; e = e->col)
         {  for (ee = A->row[e->i]; ee != NULL; ee = ee->row)
               if (ee == e) break;
            if (ee == NULL)
               fault("check_mat: element not found in row list");
         }
      }
      return;
}

/* eof */
