/* glpavl/next_node.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpavl.h"

/*----------------------------------------------------------------------
-- next_node - search for the next node in AVL-tree.
--
-- *Synopsis*
--
-- #include "glpavl.h"
-- AVLNODE *next_node(AVLTREE *tree, AVLNODE *node);
--
-- *Description*
--
-- The next_node routine searches for the next node in AVL-tree that
-- follows the given node. If the parameter node is NULL, the routine
-- searches for the first node of AVL-tree. The next node always has a
-- key that is not less than a key of the given node.
--
-- The next_node routine can be used to visit all nodes of the AVL-tree
-- in the given order by means of the following for statement:
--
-- for (n = next_node(t, NULL); n != NULL; n = next_node(t, n)) ...
--
-- *Returns*
--
-- If the parameter node is NULL, the next_node routine returns a
-- pointer to the first node of the AVL-tree that has smallest key, or
-- NULL if the AVL-tree is empty. If the parameter node points to the
-- given node, the next_node routine returns a pointer to the next node,
-- or NULL if the next node does not exist. */

AVLNODE *next_node(AVLTREE *tree, AVLNODE *node)
{     AVLNODE *p = node, *q;
      if (tree->root == NULL) return NULL;
      q = (p == NULL ? tree->root : p->right);
      if (q == NULL)
      {  /* go up from the left subtree */
         for (;;)
         {  q = p->up;
            if (q == NULL) break;
            if (p->flag == 0) break;
            p = q;
         }
      }
      else
      {  /* go down into the right subtree */
         for (;;)
         {  p = q->left;
            if (p == NULL) break;
            q = p;
         }
      }
      return q;
}

/* eof */
