/* glpavl/delete_node.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpavl.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- delete_node - delete given node from AVL-tree.
--
-- *Synopsis*
--
-- #include "glpavl.h"
-- void delete_node(AVLTREE *tree, AVLNODE *node);
--
-- *Description*
--
-- The delete_node deletes the given node from the AVL-tree. It is
-- assumed that the deleted node belongs to the AVL-tree; otherwise the
-- behavior is undefined. (This operation is applicable in both cases
-- when nodes have keys and when nodes haven't keys, since the order of
-- nodes in the AVL-tree is preserved.)
--
-- Should note that exactly the given node will be deleted, so all
-- references to some other nodes (if they exist) will remain valid.
--
-- Actually the memory allocated to the deleted node will not be freed,
-- the routine just returns that memory to the memory pool to reuse it
-- in the future if it will be necessary. */

void delete_node(AVLTREE *tree, AVLNODE *node)
{     AVLNODE *f, *p = node, *q, *r, *s, *x, *y;
      short int flag;
      if (node == NULL)
         fault("delete_node: node pointer not specified");
      /* if the deleted node has two non-empty subtrees, it should be
         interchanged with the next node that always has at least one
         empty subtree */
      if (p->left == NULL || p->right == NULL) goto skip;
      f = p->up; q = p->left;
      r = next_node(tree, p); s = r->right;
      if (p->right == r)
      {  if (f == NULL)
            tree->root = r;
         else
            if (p->flag == 0) f->left = r; else f->right = r;
         r->rank = p->rank; r->up = f;
         r->flag = p->flag; r->bal = p->bal;
         r->left = q; r->right = p;
         q->up = r;
         p->rank = 1; p->up = r; p->flag = 1;
         p->bal = (short int)(s == NULL ? 0 : +1);
         p->left = NULL; p->right = s;
         if (s != NULL) s->up = p;
      }
      else
      {  x = p->right; y = r->up;
         if (f == NULL)
            tree->root = r;
         else
            if (p->flag == 0) f->left = r; else f->right = r;
         r->rank = p->rank; r->up = f;
         r->flag = p->flag; r->bal = p->bal;
         r->left = q; r->right = x;
         q->up = r; x->up = r; y->left = p;
         p->rank = 1; p->up = y; p->flag = 0;
         p->bal = (short int)(s == NULL ? 0 : +1);
         p->left = NULL; p->right = s;
         if (s != NULL) s->up = p;
      }
skip: /* now the deleted node [p] has at least one empty subtree; go up
         to the root and correct rank field of all nodes affected by the
         deletion */
      q = p; f = q->up;
      while (f != NULL)
      {  if (q->flag == 0) f->rank--;
         q = f; f = q->up;
      }
      /* delete the given node from the AVL-tree */
      f = p->up; flag = p->flag;
      q = p->left != NULL ? p->left : p->right;
      if (f == NULL)
         tree->root = q;
      else
         if (flag == 0) f->left = q; else f->right = q;
      if (q != NULL) q->up = f, q->flag = flag;
      tree->size--;
      /* go up to the root again and correct all nodes affected by the
         deletion */
      while (f != NULL)
      {  if (flag == 0)
         {  /* the height of the left subtree of [f] is decreased */
            if (f->bal == 0)
            {  f->bal = +1;
               break;
            }
            if (f->bal < 0)
               f->bal = 0;
            else
            {  f = rotate_avl(tree, f);
               if (f->bal < 0) break;
            }
            flag = f->flag; f = f->up;
         }
         else
         {  /* the height of the right subtree of [f] is decreased */
            if (f->bal == 0)
            {  f->bal = -1;
               break;
            }
            if (f->bal > 0)
               f->bal = 0;
            else
            {  f = rotate_avl(tree, f);
               if (f->bal > 0) break;
            }
            flag = f->flag; f = f->up;
         }
      }
      /* if the root reached, the height of entire tree is decreased */
      if (f == NULL) tree->height--;
      /* returns the memory allocated to the deleted node to the memory
         pool of the corresponding AVL-tree */
      free_atom(tree->pool, p);
      return;
}

/* eof */
