/* glpapi/glp_read_lpm.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <float.h>
#include <stddef.h>
#include "glplpm.h"
#include "glpk.h"

/*----------------------------------------------------------------------
-- glp_read_lpm - read problem data using LPM format.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_read_lpm(char *fname);
--
-- *Description*
--
-- The glp_read_lpm routine reads LP problem data using LPM format from
-- the text file named fname into the workspace.
--
-- As a rule the workspace should be empty before a call to the
-- glp_read_lpm routine, i.e. the workspace should contain no rows and
-- no columns.
--
-- For description of the LPM format see the program documentation.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the operation failed because of errors. All diagnostics was sent
--     to stderr. */

int glp_read_lpm(char *fname)
{     LPM *lpm;
      LPMROW *row;
      LPMCOL *col;
      /* load LPM data block */
      lpm = load_lpm(fname);
      if (lpm == NULL) goto fail;
      /* set problem name */
      glp_set_cpar("problem", lpm->name);
      /* create rows */
      for (row = lpm->first_row; row != NULL; row = row->next)
      {  int type;
         if (glp_create_item(GLP_ROW, row->name))
         {  error("glp_read_lpm: error on creating row `%s'",
               row->name);
            goto fail;
         }
         if (row->lb == -DBL_MAX && row->ub == +DBL_MAX)
            type = GLP_FR;
         else if (row->ub == +DBL_MAX)
            type = GLP_LO;
         else if (row->lb == -DBL_MAX)
            type = GLP_UP;
         else if (row->lb != row->ub)
            type = GLP_DB;
         else
            type = GLP_FX;
         glp_set_bounds(GLP_ROW, type, row->lb, row->ub);
      }
      /* create columns */
      for (col = lpm->first_col; col != NULL; col = col->next)
      {  int type;
         if (glp_create_item(GLP_COL, col->name))
         {  error("glp_read_lpm: error on creating column `%s'",
               col->name);
            goto fail;
         }
         glp_set_kind(GLP_COL, col->kind ? GLP_YES : GLP_NO);
         if (col->lb == -DBL_MAX && col->ub == +DBL_MAX)
            type = GLP_FR;
         else if (col->ub == +DBL_MAX)
            type = GLP_LO;
         else if (col->lb == -DBL_MAX)
            type = GLP_UP;
         else if (col->lb != col->ub)
            type = GLP_DB;
         else
            type = GLP_FX;
         glp_set_bounds(GLP_COL, type, col->lb, col->ub);
      }
      /* create constraint matrix */
      for (row = lpm->first_row; row != NULL; row = row->next)
      {  LPMLFE *e;
         glp_find_item(GLP_ROW, row->name);
         for (e = row->form; e != NULL; e = e->next)
         {  glp_find_item(GLP_COL, e->col->name);
            glp_new_coef(e->coef);
         }
      }
      /* set optimization direction */
      glp_set_ipar("obj_dir", lpm->obj_dir == '-' ? GLP_MIN : GLP_MAX);
      /* set the objective function */
      if (lpm->obj_row == NULL)
      {  glp_set_ipar("option", GLP_ANY);
         glp_set_cpar("obj_row", "");
      }
      else
      {  glp_set_ipar("option", GLP_FIN);
         glp_set_cpar("obj_row", lpm->obj_row->name);
      }
      /* free LPM data block */
      free_lpm(lpm);
      /* return to the application program */
      return 0;
fail: /* the operation failed */
      if (lpm != NULL) free_lpm(lpm);
      return 1;
}

/* eof */
