/* glpapi/glp_get_ipar.c */

/*----------------------------------------------------------------------
-- This file is a part of the GLPK package.
--
-- Copyright (C) 2000, 2001 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                          Department for Applied Informatics,
--                          Moscow Aviation Institute, Moscow, Russia.
--                          All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <string.h>
#include "glpapi.h"

/*----------------------------------------------------------------------
-- glp_get_ipar - get value of integer control parameter.
--
-- *Synopsis*
--
-- #include "glpk.h"
-- int glp_get_ipar(char *name, int *val);
--
-- *Description*
--
-- The glp_get_ipar routine stores the value of the integer control
-- parameter with the given name to the location, which the parameter
-- val points to.
--
-- See the documentation for descriptions of all control parameters.
--
-- *Returns*
--
-- 0 - no errors;
-- 1 - the given name is invalid. */

int glp_get_ipar(char *name, int *val)
{     if (strcmp(name, "mip_branch") == 0)
      {  /* branching technique flag */
         *val = glp->mip_branch;
      }
      else if (strcmp(name, "mip_btrack") == 0)
      {  /* backtracking technique flag */
         *val = glp->mip_btrack;
      }
      else if (strcmp(name, "mps_one_entry") == 0)
      {  /* write MPS file not using fields 5 and 6 */
         *val = glp->mps_one_entry;
      }
      else if (strcmp(name, "mps_pedantic") == 0)
      {  /* write MPS file using pedantic style */
         *val = glp->mps_pedantic;
      }
      else if (strcmp(name, "mps_skip_empty") == 0)
      {  /* don't write empty columns to MPS file */
         *val = glp->mps_skip_empty;
      }
      else if (strcmp(name, "mps_use_names") == 0)
      {  /* use original names as templates to generate MPS names */
         *val = glp->mps_use_names;
      }
      else if (strcmp(name, "nc") == 0)
      {  /* total number of columns */
         *val = glp->col_tab->size;
      }
      else if (strcmp(name, "nc_bin") == 0)
      {  /* number of binary columns */
         GLPITEM *col;
         int nc_bin = 0;
         for (col = glp->first_col; col != NULL; col = col->next)
            if (col->kind && col->type == GLP_DB &&
                col->lb == 0.0 && col->ub == 1.0) nc_bin++;
         *val = nc_bin;
      }
      else if (strcmp(name, "nc_int") == 0)
      {  /* number of integer columns */
         GLPITEM *col;
         int nc_int = 0;
         for (col = glp->first_col; col != NULL; col = col->next)
            if (col->kind) nc_int++;
         *val = nc_int;
      }
      else if (strcmp(name, "nr") == 0)
      {  /* total number of rows */
         *val = glp->row_tab->size;
      }
      else if (strcmp(name, "nz") == 0)
      {  /* total number of non-zeros in the constraint matrix */
         *val = glp->coef_pool->count;
      }
      else if (strcmp(name, "obj_dir") == 0)
      {  /* optimization direction flag */
         *val = glp->obj_dir;
      }
      else if (strcmp(name, "option") == 0)
      {  /* what solution should be found by the solver */
         *val = glp->option;
      }
      else if (strcmp(name, "round") == 0)
      {  /* round the computed values of basis variable */
         *val = glp->round;
      }
      else if (strcmp(name, "scale") == 0)
      {  /* scale the problem before solving */
         *val = glp->scale;
      }
      else if (strcmp(name, "spx_form") == 0)
      {  /* the form of the basis matrix used by the solver */
         *val = glp->spx_form;
      }
      else if (strcmp(name, "spx_relax") == 0)
      {  /* use the ratio test proposed by P.Harris */
         *val = glp->spx_relax;
      }
      else if (strcmp(name, "spx_steep") == 0)
      {  /* use the steepest edge technique */
         *val = glp->spx_steep;
      }
      else if (strcmp(name, "spx_use_dual") == 0)
      {  /* use dual simplex to search for feasible solution */
         *val = glp->spx_use_dual;
      }
      else if (strcmp(name, "status") == 0)
      {  /* status of the computed solution (set bythe solver) */
         *val = glp->status;
      }
      else if (strcmp(name, "sum_aij") == 0)
      {  /* sum multiplets of the constraint matrix */
         *val = glp->sum_aij;
      }
      else
      {  /* invalid parameter name */
         return 1;
      }
      return 0;
}

/* eof */
