/* glpmat/count_nz.c */

/*----------------------------------------------------------------------
-- This file is a part of the GNU LPK package.
--
-- Copyright (C) 2000 Andrew Makhorin <mao@mai2.rcnet.ru>,
--                    Department for Applied Informatics,
--                    Moscow Aviation Institute, Moscow, Russia.
--                    All rights reserved.
--
-- This code is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This software is distributed "as is" in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
-- General Public License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
----------------------------------------------------------------------*/

#include <stddef.h>
#include "glpmat.h"
#include "glpset.h"

/*----------------------------------------------------------------------
-- count_nz - count non-zeros of sparse matrix.
--
-- *Synopsis*
--
-- #include "glpmat.h"
-- int count_nz(MAT *A, int k);
--
-- *Description*
--
-- If k = 0, the count_nz routine counts non-zero elements of entire
-- matrix A.
--
-- If k > 0, the count_nz routine counts non-zero elements in k-th row
-- of the matrix A.
--
-- If k < 0, the count_nz routine counts non-zero elements in k-th
-- column of the matrix A.
--
-- *Returns*
--
-- The count_nz routine returns the counted number of non-zeros. */

int count_nz(MAT *A, int k)
{     ELEM *e;
      int i, j, nz = 0;
      if (k == 0)
      {  /* count non-zeros of entire matrix */
         for (i = 1; i <= A->m; i++)
            for (e = A->row[i]; e != NULL; e = e->row) nz++;
      }
      else if (k > 0)
      {  /* count non-zeros in k-th row */
         i = +k;
         if (!(1 <= i && i <= A->m))
            fault("count_nz: invalid row number");
         for (e = A->row[i]; e != NULL; e = e->row) nz++;
      }
      else
      {  /* count non-zeros in k-th column */
         j = -k;
         if (!(1 <= j && j <= A->n))
            fault("count_nz: invalid column number");
         for (e = A->col[j]; e != NULL; e = e->col) nz++;
      }
      return nz;
}

/* eof */
